#include "augment.h"

// These are implemented in augment.cc but are not listed in the
// header file. We are declaring them here so that we can test them.
void get_ap(std::string* formula_str, sset_t* container);
void formula2conj(std::string* formula, svector_t* conj, alpha_reduction_t ar);
void optimize_conj(global_params* gp, monitor_params* mp, sset_t* mutex_conj);

int num_tests;
int num_fail;


/**
 * All elements of the "inner" set need to be also in the "outer" set.
 */
template< class T >
bool set_contains(std::set< T > *outer, std::set< T > *inner) {
  
  typename std::set<T>::iterator it;
  

  for (it = inner->begin();
       it != inner->end();
       it++) 
    {
      if (outer->find(*it) == outer->end()) {
	return false;
      }
    }
  return true;
}


/**
 * Set comparison by checking mutual containment.
 */
template< class T>
bool sets_equal(std::set<T>* left, std::set<T>* right) {
  if (set_contains(left, right)) {
    return set_contains(right, left);
  }
  else {
    return false;
  }
}


/**
 * Test if the atoms extracted from a formula are what we would have
 * expected
 */
void test_extraction(std::string formula, sset_t* exp_set, bool exp_result = true) {
  num_tests++;
  
  sset_t* output = new sset_t();
  get_ap(&formula, output);

  if (! sets_equal(output, exp_set)) {
    if (exp_result == true) {
      num_fail++;
      std::cout << "Error extracting the atoms from: " << formula << std::endl;
      
      std::cout << "Expected atoms: " << std::endl;
      for (sset_t::const_iterator it = exp_set->begin();
	   it != exp_set->end();
	   it++)
	{
	  std::cout << *it << " ";
	}
      
      std::cout << std::endl;
      std::cout << "Actual output: " << std::endl;
      for (sset_t::const_iterator it = output->begin();
	   it != output->end();
	   it++)
	{
	  std::cout << *it << " ";
	}
      std::cout << std::endl;
    }
  }
  else {
    if (exp_result == false) {
      num_fail++;
      std::cout << "Atoms extraction did not fail as expected from: " << formula << std::endl;
    }
  }
  
}

void extraction_testing() {
  
  sset_t* expected = new sset_t;
  expected->insert("p");

  test_extraction(std::string("p"), expected);
  test_extraction(std::string("G p"), expected);
  test_extraction(std::string("G (X( p))"), expected);

  test_extraction(std::string("G p && X(X( q))"), expected, false);
  test_extraction(std::string("p && (p U q)"), expected, false);
  test_extraction(std::string("G(p && (p U q))"), expected, false);
  test_extraction(std::string("(p || q) && G(p && (p U q))"), expected, false);

  expected->insert("q");
  
  test_extraction(std::string("p"), expected, false);
  test_extraction(std::string("G p"), expected, false);
  test_extraction(std::string("G (X( p))"), expected, false);

  test_extraction(std::string("G p && X(X( q))"), expected);
  test_extraction(std::string("p && (p U q)"), expected);
  test_extraction(std::string("G(p && (p U q))"), expected);
  test_extraction(std::string("(p || q) && G(p && (p U q))"), expected);
  test_extraction(std::string("(p || q || p) && G(p && (p U q))"), expected);
  test_extraction(std::string("(1 || 0 || 1) && G(p && (p U q))"), expected);
  test_extraction(std::string("(1 || 0 ) || G(p && (p U q))"), expected);

  test_extraction(std::string("\"some_object->some_variable\""), expected, false);
  test_extraction(std::string("G \"some_object->some_variable\""), expected, false);
  test_extraction(std::string("G (X(\"some_object->some_variable\"))"), expected, false);

  delete expected;

  expected = new sset_t;
  expected->insert("some_object->some_variable");
  test_extraction(std::string("(1 || 0 ) || G(p && (p U q))"), expected, false);
  test_extraction(std::string("(p || q || p) && G(p && (p U q))"), expected, false);
  test_extraction(std::string("(1 || 0 || 1) && G(p && (p U q))"), expected, false);

  test_extraction(std::string("\"some_object->some_variable\""), expected);
  test_extraction(std::string("G \"some_object->some_variable\""), expected);
  test_extraction(std::string("G (X(\"some_object->some_variable\"))"), expected);

  expected->insert("some_object->variable1");
  expected->insert("some_object->variable2");
  expected->insert("some_object->variable3");
  
  test_extraction(std::string("G (X(\"some_object->some_variable\"))"), expected, false);
  test_extraction(std::string("\"some_object->variable1\" && "
			      "\"some_object->variable1\" U ("
			      "\"some_object->variable2\" &&"
			      "\"some_object->variable2\" U ("
			      "\"some_object->variable3\" &&"
			      "\"some_object->variable3\" U "
			      "\"some_object->some_variable\"))"), expected);

  delete expected;
}

/**
 * Prints the elements of the list, one at a time
 */
void print_list(svector_t* l) {
  
  int count = 0;
  
  for (svector_t::const_iterator it = l->begin();
       it != l->end();
       it ++)
    {
      std::cout << count << ": " << *it << std::endl; 
      count ++;
    }
}



/**
 * Prints the elements of the set, one at a time
 */
void print_set(sset_t* l) {
  
  int count = 0;
  
  for (sset_t::const_iterator it = l->begin();
       it != l->end();
       it ++)
    {
      std::cout << count << ": " << *it << std::endl; 
      count ++;
    }
}


void test_conjunction(std::string formula) {

  svector_t* conj = new svector_t;
  formula2conj(&formula, conj, PARTIAL_REDUCTION);

  std::cout << "Conjuctions for formula: " << std::endl;
  std::cout << formula << std::endl;
  print_list(conj);
  delete conj;
}


void conjustion_testing() {
  
  test_conjunction(std::string("p"));
  test_conjunction(std::string("p && q"));
  test_conjunction(std::string("!p && q"));
  test_conjunction(std::string("G(!p && q)"));
  test_conjunction(std::string("G(!\"p\" && (q || (r && s)))"));
  test_conjunction(std::string("\"s->var1\" && "
			       "\"s->var1\" U ("
			       "\"s->var2\" &&"
			       "\"s->var2\" U ("
			       "\"s->var3\" &&"
			       "\"s->var3\" U "
			       "\"s->var4\"))"));
}



int main(int argc, char** argv) {

  num_tests = 0;
  num_fail = 0;
  
  //extraction_testing();
  //conjustion_testing();
  global_params* gp = new global_params(argc, argv);
  gp->to_stream(std::cout);

  sset_t* bad_conj = new sset_t;
  mpset_t::const_iterator it = gp->all_monitors.begin();
  
  // optimize_conj(gp, *it, bad_conj);
  
  //print_set(bad_conj);

  std::string augmented;
  augment_formula(gp, *it, &augmented);
  
  std::cout << "Augmented formula: " << augmented << std::endl;

  delete gp;
  delete bad_conj;
  
}
