#include <cassert>
#include <sstream>
#include <ctype.h>
#include <ostream>
#include "tostring_paren.h"
#include "ltlast/visitor.hh"
#include "ltlast/allnodes.hh"


static bool
is_bare_word(const char* str)
{
  // 2010-06-09: Force parens because they are assumed downstream.
  return false;  

  
  
  // Bare words cannot be empty, start with the letter of a unary
  // operator, or be the name of an existing constant.  Also they
  // should start with an letter.
  if (!*str
      || *str == 'F'
      || *str == 'G'
      || *str == 'X'
      || !(isalpha(*str) || *str == '_')
      || !strcasecmp(str, "true")
      || !strcasecmp(str, "false"))
    return false;
  // The remaining of the word must be alphanumeric.
  while (*++str)
    if (!(isalnum(*str) || *str == '_'))
      return false;
  return true;
}

class to_string_visitor_paren: public spot::ltl::const_visitor
{
public:
  to_string_visitor_paren(std::ostream& os)
    : os_(os), top_level_(true)
  {
  }

  virtual
  ~to_string_visitor_paren()
  {
  }

  void
  visit(const spot::ltl::atomic_prop* ap)
  {
    std::string str = ap->name();

    if (!is_bare_word(str.c_str()))
      {
	os_ << "\"(" << str << ")\"";
      }
    else
      {
	os_ << "(" << str << ")";
      }
  }

  void
  visit(const spot::ltl::constant* c)
  {
    os_ << c->val_name();
  }

  void
  visit(const spot::ltl::binop* bo)
  {
    bool top_level = top_level_;
    top_level_ = false;
    if (!top_level)
      os_ << "(";

    bo->first()->accept(*this);

    switch (bo->op())
      {
      case spot::ltl::binop::Xor:
	os_ << " ^ ";
	break;
      case spot::ltl::binop::Implies:
	os_ << " => ";
	break;
      case spot::ltl::binop::Equiv:
	os_ << " <=> ";
	break;
      case spot::ltl::binop::U:
	os_ << " U ";
	break;
      case spot::ltl::binop::R:
	os_ << " R ";
	break;
      case spot::ltl::binop::W: //KYR added from spot's ltlvisit/tostring.cc
	os_ << " W ";
	break;
      case spot::ltl::binop::M: //KYR added from spot's ltlvisit/tostring.cc
	os_ << " M ";
	break;
      }

    bo->second()->accept(*this);
    if (!top_level)
      os_ << ")";
  }

  void
  visit(const spot::ltl::unop* uo)
  {
    // The parser treats F0, F1, G0, G1, X0, and X1 as atomic
    // propositions.  So make sure we output F(0), G(1), etc.
    bool need_parent = !!dynamic_cast<const spot::ltl::constant*>(uo->child());
    switch (uo->op())
      {
      case spot::ltl::unop::Not:
	os_ << "!";
	need_parent = false;
	break;
      case spot::ltl::unop::X:
	os_ << "X";
	break;
      case spot::ltl::unop::F:
	os_ << "F";
	break;
      case spot::ltl::unop::G:
	os_ << "G";
	break;
      case spot::ltl::unop::Finish: //KYR added to satisfy warnings
	os_ << "finish"; //from spot's ltlvisit/tostring.cc function
	need_parent = true;
	//Die with an error here? This should not happen...
	break;
      }

    top_level_ = false;
    if (need_parent)
      os_ << "(";
    uo->child()->accept(*this);
    if (need_parent)
      os_ << ")";
  }

  void
  visit(const spot::ltl::multop* mo)
  {
    bool top_level = top_level_;
    top_level_ = false;
    if (!top_level)
      os_ << "(";
    unsigned max = mo->size();
    mo->nth(0)->accept(*this);
    const char* ch = " ";
    switch (mo->op())
      {
      case spot::ltl::multop::Or:
	ch = " | ";
	break;
      case spot::ltl::multop::And:
	ch = " & ";
	break;
      }

    for (unsigned n = 1; n < max; ++n)
      {
	os_ << ch;
	mo->nth(n)->accept(*this);
      }
    if (!top_level)
      os_ << ")";
  } //KYR: end multop

  //KYR: added to get this to compile
  void visit(const spot::ltl::automatop* ao) {}

protected:
  std::ostream& os_;
  bool top_level_;
};

class to_systemc_string_visitor : public to_string_visitor_paren
{
public:
  to_systemc_string_visitor(std::ostream& os)
    : to_string_visitor_paren(os)
  {
  }

  virtual
  ~to_systemc_string_visitor()
  {
  }

  void
  visit(const spot::ltl::binop* bo)
  {
    bool top_level = top_level_;
    top_level_ = false;
    if (!top_level)
      os_ << "(";

    switch (bo->op())
      {
      case spot::ltl::binop::Xor:
	os_ << "(!";
	bo->first()->accept(*this);
	os_ << " && ";
	bo->second()->accept(*this);
	os_ << ") || (";
	bo->first()->accept(*this);
	os_ << " && !";
	bo->second()->accept(*this);
	os_ << ")";
	break;
      case spot::ltl::binop::Implies:
	os_ << "!";
	bo->first()->accept(*this);
	os_ << " || ";
	bo->second()->accept(*this);
	break;
      case spot::ltl::binop::Equiv:
	os_ << "(";
	bo->first()->accept(*this);
	os_ << " && ";
	bo->second()->accept(*this);
	os_ << ") || (!";
	bo->first()->accept(*this);
	os_ << " && !";
	bo->second()->accept(*this);
	os_ << ")";
      case spot::ltl::binop::U:
	bo->first()->accept(*this);
	os_ << " U ";
	bo->second()->accept(*this);
	break;
      case spot::ltl::binop::R:
	bo->first()->accept(*this);
	os_ << " V ";
	bo->second()->accept(*this);
	break;
	
	//KYR added from spot's ltlvisit/tostring.cc to satisfy warnings
	/* W and M are not supported by Spin */
      case spot::ltl::binop::W:
	bo->first()->accept(*this);
	os_ << " W ";
	bo->second()->accept(*this);
	break;
      case spot::ltl::binop::M:
	bo->first()->accept(*this);
	os_ << " M ";
	bo->second()->accept(*this);
	break;
      }

    if (!top_level)
      os_ << ")";
  }

  void
  visit(const spot::ltl::unop* uo)
  {
    // The parser treats X0, and X1 as atomic propositions.	 So
    // make sure we output X(0) and X(1).
    bool need_parent = false;
    switch (uo->op())
      {
      case spot::ltl::unop::Not:
	os_ << "!";
	break;
      case spot::ltl::unop::X:
	need_parent = !!dynamic_cast<const spot::ltl::constant*>(uo->child());
	os_ << "X";
	break;
      case spot::ltl::unop::F:
	os_ << "<>";
	break;
      case spot::ltl::unop::G:
	os_ << "[]";
	break;
      case spot::ltl::unop::Finish: //KYR added to satisfy warnings
	os_ << "finish"; //from spot's ltlvisit/tostring.cc function
	need_parent = true;
	//Die with an error here? This should not happen...
	break;
      }

    top_level_ = false;
    if (need_parent)
      os_ << "(";
    uo->child()->accept(*this);
    if (need_parent)
      os_ << ")";
  }

  void
  visit(const spot::ltl::multop* mo)
  {
    bool top_level = top_level_;
    top_level_ = false;
    if (!top_level)
      os_ << "(";
    unsigned max = mo->size();
    mo->nth(0)->accept(*this);
    const char* ch = " ";
    switch (mo->op())
      {
      case spot::ltl::multop::Or:
	ch = " || ";
	break;
      case spot::ltl::multop::And:
	ch = " && ";
	break;
      }

    for (unsigned n = 1; n < max; ++n)
      {
	os_ << ch;
	mo->nth(n)->accept(*this);
      }
    if (!top_level)
      os_ << ")";
  }
};


class to_systemc_string_visitor_with_quotes : public to_systemc_string_visitor {

public: 
  
  /**
   * Constructor
   */
  to_systemc_string_visitor_with_quotes(std::ostream& os)
    : to_systemc_string_visitor(os) {}


  virtual
  ~to_systemc_string_visitor_with_quotes() {}
  
  void
  visit(const spot::ltl::atomic_prop* ap)
  {
    std::string str = ap->name();
    if (!is_bare_word(str.c_str()))
      {
	os_ << "\"(" << str << ")\"";
      }
    else
      {
	os_ << "\"" << str << "\"";
      }
  }

}; //to_systemc_string_visitor_with_quotes

std::ostream&
to_string_paren(const spot::ltl::formula* f, std::ostream& os)
{
  to_string_visitor_paren v(os);
  f->accept(v);
  return os;
}

std::string
to_string_paren(const spot::ltl::formula* f)
{
  std::ostringstream os;
  to_string_paren(f, os);
  return os.str();
}

std::ostream&
to_systemc_string_paren(const spot::ltl::formula* f, std::ostream& os)
{
  to_systemc_string_visitor v(os);
  f->accept(v);
  return os;
}

std::string
to_systemc_string_paren(const spot::ltl::formula* f)
{
  std::ostringstream os;
  to_systemc_string_paren(f, os);
  return os.str();
}

std::ostream&
to_systemc_string_paren_with_quotes(const spot::ltl::formula* f, std::ostream& os)
{
  to_systemc_string_visitor_with_quotes v(os);
  f->accept(v);
  return os;
}

std::string
to_systemc_string_paren_with_quotes(const spot::ltl::formula* f)
{
  std::ostringstream os;
  to_systemc_string_paren_with_quotes(f, os);
  return os.str();
}
