#include "ksat.h"

Formula::Formula()
{
  opMain=OP_PROP;
  pfLeft=pfRight=NULL;
  iSubscript=iVarOrder=iVarNextOrder=0;
  bddCache=bddWitness=bddTrans=ddManager.bddOne();
  bNegated=bNegLeft=bNegRight=false;
  iRefCount=0;
  iIndex=-1;
  mark=false;
}

Formula::Formula(const Formula &form)
{
  opMain=form.opMain;
  iRefCount=0;
  pfLeft=pfRight=NULL;
  SetLeftChild(form.pfLeft);
  SetRightChild(form.pfRight);
  iSubscript=form.iSubscript;
  iVarOrder=iVarNextOrder=0;
  iIndex=-1;
  bddCache=bddWitness=bddTrans=ddManager.bddOne();
  bNegLeft=form.bNegLeft;
  bNegRight=form.bNegRight;
  mark=false;
}

Formula::~Formula()
{
  assert(iRefCount>=0);
  if (pfLeft!=NULL) {
    pfLeft->iRefCount--;
    if (pfLeft->iRefCount==0)
      delete pfLeft;
  }
  if (pfRight!=NULL) {
    pfRight->iRefCount--;
    if (pfRight->iRefCount==0)
      delete pfRight;
  }
 }

Formula *Formula::Clone()
{
  Formula *left=NULL, *right=NULL;
  if (pfLeft)
    left=pfLeft->Clone();
  if (pfRight)
    right=pfRight->Clone();
  Formula *result=new Formula(*this);
  result->SetLeftChild(left);
  result->SetRightChild(right);
  return result;
}

BDD Formula::GetBDD()
{
  return ddManager.bddVar(iVarOrder);
}

void Formula::Traverse(FormulaFunction &funcNode, Options &opt, bool TopDown)
{
  assert(Valid());
  // This node in DAG we already visited
  if (mark==true)
    return;
  mark=true;
  if (TopDown)
    funcNode(this, opt);
  // Traverse graph
  switch (opMain) {
  case OP_AND:
  case OP_OR:
    assert(pfLeft!=NULL);
    pfLeft->Traverse(funcNode, opt, TopDown);
    assert(pfRight!=NULL);
    pfRight->Traverse(funcNode, opt, TopDown);
    break;
  case OP_NOT:
  case OP_BOX:
  case OP_DIAMOND:
  case OP_SNAP:
    assert(pfLeft!=NULL);
    pfLeft->Traverse(funcNode, opt, TopDown);
    break;
  }
  if (!TopDown) {
    assert(funcNode.bNodeCreateChild==false);
    funcNode(this, opt);
  }
}

void Formula::Clear()
{
  assert(Valid());
  if (mark==false)
    return;
  mark=false;
  bddCache=ddManager.bddOne();
  
  switch (opMain) {
  case OP_NOT:
  case OP_BOX:
  case OP_DIAMOND:
  case OP_SNAP:
    pfLeft->Clear();
    assert(pfRight==NULL);
    break;
  case OP_AND:
  case OP_OR:
    pfLeft->Clear();
    pfRight->Clear();
    break;
  case OP_PROP:
  case OP_TOP:
  case OP_BOTTOM:
    assert(pfLeft==NULL);
    assert(pfRight==NULL);
    break;
  default:
    assert("Bad Node"==NULL);
    break;
  }
}

void Formula::TreeTraverse(FormulaFunction &funcNode, Options &opt, bool TopDown, bool recall)
{
  Valid();
  if (TopDown)
    funcNode(this, opt);
  // Traverse graph
  switch (opMain) {
  case OP_AND:
  case OP_OR:
    assert(pfLeft!=NULL);
    pfLeft->TreeTraverse(funcNode, opt, TopDown, recall);
    if (TopDown&&recall) {
      funcNode(this, opt);
    }
    assert(pfRight!=NULL);
    pfRight->TreeTraverse(funcNode, opt, TopDown, recall);
    break;
  case OP_NOT:
  case OP_BOX:
  case OP_DIAMOND:
  case OP_SNAP:
    assert(pfLeft!=NULL);
    pfLeft->TreeTraverse(funcNode, opt, TopDown, recall);
    break;
  }
  if (!TopDown) {
    assert(funcNode.bNodeCreateChild==false);
    funcNode(this, opt);
  }
} 

bool Formula::Valid(AssertionPhase phase)
{
#ifndef NDEBUG
  assert(iRefCount>0);
  assert(this!=pfLeft);
  assert(this!=pfRight);
  switch (opMain) {
  case OP_PROP:
  case OP_TOP:
  case OP_BOTTOM:
    assert(pfLeft==NULL);
    // FALLTHRU
  case OP_NOT:
  case OP_DIAMOND:
  case OP_BOX:
  case OP_SNAP:
    assert(pfRight==NULL);
    break;
  }
  if (phase>=BUILDBDD) {
    assert(opMain!=OP_SNAP);
    assert(iIndex>=0);
  }
#endif
  return true;
}

ostream &operator<<(ostream &os, Formula &form)
{
  form.Valid();
  switch (form.opMain) {
  case OP_TOP:
    return os<<"true";
  case OP_BOTTOM:
    return os<<"false";
  case OP_PROP:
    return os<<"p"<<form.iSubscript;
  case OP_NOT:
    return os<<"(~"<<*(form.pfLeft)<<")";
  case OP_BOX:
    return os<<"(box "<<*(form.pfLeft)<<")";
  case OP_DIAMOND:
    return os<<"(dia "<<*(form.pfLeft)<<")";
  case OP_AND:
    return os<<"("<<*(form.pfLeft)<<" & "<<*(form.pfRight)<<")";
  case OP_OR:
    return os<<"("<<*(form.pfLeft)<<" v "<<*(form.pfRight)<<")";
  case OP_SNAP:
    return os<<"%"<<*(form.pfLeft);
  }
  return os;
}

/**
Build formula stub for parser. Builds one formula node according to reductions.
**/

void *build_formula(void *left, Operator op, int value, void *right)
{
  Formula *work=new Formula;
  Formula *fLeft=(Formula*)left;
  Formula *fRight=(Formula*)right;
  work->SetLeftChild(fLeft);
  work->SetRightChild(fRight);
  work->iSubscript=value;
  work->opMain=op;
  return (void*)work;
}

/**
Clone formula stub for parser. Copies a formula tree.
**/

void *clone_formula(void *formula)
{
  Formula *form=(Formula *)formula;
  Formula *clone=form->Clone();
  return (void *)clone;
}
