#ifndef __FORMULA_H
#define __FORMULA_H

#ifdef __cplusplus

//using namespace std;

#include <iostream>
#include <functional>
#include <vector>

#include <assert.h>

#include "cuddObj.hh"

#include "Options.h"
#include "globals.h"

extern "C" {
#include "st.h"
};
#endif /*__cplusplus */

typedef enum {OP_PROP, OP_AND, OP_OR, OP_NOT, OP_TOP, OP_BOTTOM, OP_BOX, OP_DIAMOND, OP_SNAP} Operator;

typedef enum {BUILDGRAPH, BUILDBDD, CHECKSAT} AssertionPhase;

#ifdef __cplusplus

// Definitions

class Formula;
class FormulaFunction;

/// Each operator and proposition in a formula is represented by a node.

class Formula 
{
public:
  /// Mark for traversal of the formula
  bool mark;
  /// Refence Count for the node
  int iRefCount;
  /// The main operator of this formula
  Operator opMain;
  /// Tree links
  Formula *pfLeft,*pfRight;
  /// Is the edge negated (for atoms)
  bool bNegLeft,bNegRight;
  /// A tempory variable to hold the negated status of this node for Normal Forms
  bool bNegated;
  /// Proposition or Modal subscript
  int iSubscript;
  /// Variable order of this subformula
  int iVarOrder;
  /// Variable order of the corresponding image of this subformula
  int iVarNextOrder;
  /// A temp variable to keep level info when building. After building, store the highest occuring level of this node.
  int iLevel;
  /// The index of the node
  int iIndex;
  /// Cached BDD for recursive building
  BDD bddCache;
  /// The witness of this (diamond) operator
  BDD bddWitness;
  /// The transition relation for this (box) operator
  BDD bddTrans;
  // Methods starts here
  /// Constructor
  Formula();
  /// Copy Constructor
  Formula(const Formula& form);
  /// Destructor: Update reference count for children
  ~Formula();
  /// Cloning Method
  Formula *Clone();
  /// Get the BDD representation of this subformula
  BDD GetBDD();
  /**
     Traverse the DAG of the formula. For every node visited, call funcNode on the node. Nodes are marked to ensure each node is only visited once.
  **/
  void Traverse(FormulaFunction &funcNode, Options &opt, bool TopDown=false);
  /**
     Traverse the DAG of the formula as a tree. For every node visited, call funcNode on the node. Nodes are not marked.
  **/
  void TreeTraverse(FormulaFunction &funcNode, Options &opt, bool TopDown=false, bool recall=false);
/**
Clears the DAG for future traversal.
Traverses the DAG recursively and removes the marks. It also clears the temperory BDDs.
**/
  void Clear();

  // Inline methods
  /**
Returns whether the top operator of this formula is some form of diamond.
Check whether the top level operator can be used as a diamond, i.e. a diamond operator or a box in atoms.
  **/
  inline bool IsDiamond(Options &opt) 
  {
    return (opMain==OP_DIAMOND)||((opMain==OP_BOX)&&(opt.Particles==false));  
  }
  /**
Returns whether the node is represented as a BDD variable.
For full vectors, all nodes are represented.
For lean vectors, AND and OR are not represented.
The NOT operator is not represented for atoms, but it doesnot need to be checked here since all NOTs are already converted to negated edges for atoms.
  **/
  inline bool NodeUsed(Options &opt)
  {
    if (opt.LeanVectors)
      if ((opMain==OP_AND)||(opMain==OP_OR))
	return false;
    return true;
  }
  /// Wrapper for updating reference count on left child pointer.
  inline void SetLeftChild(Formula *form)
  {
    assert(form!=this);
    if (form!=NULL)
      form->iRefCount++;
    if (pfLeft!=NULL) {
      pfLeft->Valid();
      pfLeft->iRefCount--;
      if (pfLeft->iRefCount==0)
	delete pfLeft;
    }
    pfLeft=form;
  }
  /// Wrapper for updating reference count on right child pointer.
  inline void SetRightChild(Formula *form)
  {
    assert(form!=this);
    if (form!=NULL) {
      assert(opMain!=OP_SNAP);
      form->iRefCount++;
    }
    if (pfRight!=NULL) {
      pfRight->Valid();
      pfRight->iRefCount--;
      if (pfRight->iRefCount==0)
	delete pfRight;
    }
    pfRight=form;
  }
  
  // Debug methods
  /// Assertion function to ensure the validity of the formula node.
  bool Valid(AssertionPhase phase=BUILDGRAPH);
  /// Dump the formula in LWB form.
  friend ostream &operator<<(ostream &os, Formula &form);
};

/// An abstract class for functions on Formula nodes
class FormulaFunction:public binary_function<Formula *,Options &, void> 
{
public:
  /// Whether the function can modify the node links itself
  bool bNodeCreateChild;
  /// Must implement a work function
  virtual void operator ()(Formula *form, Options &opt)=0;
};

/// A function object to build the witnesses and transition relations for modal operators

class ModalWitness:public FormulaFunction
{
  vector<Formula *> *DiamondNodeVector;
public:
  ModalWitness(vector<Formula *> *TargetDiamondNodeVector) 
  {
    DiamondNodeVector=TargetDiamondNodeVector;
    bNodeCreateChild=false;
  }
  /// The work function for this object.
  virtual void operator ()(Formula *form, Options &opt);
};

/**
A function object to build the restriction vector for bottomup/particle cases. 
In top down cases, the constraint enforced by negation operators on assignments is coded by the initial state set but for bottom up cases it is not. So it needs to be encoded seperately and constrained onto the state set after every iteration.
**/
class BuildRestrictionVector:public FormulaFunction
{
public:
  BDD RestrictionVector;
  BuildRestrictionVector()
  {
    bNodeCreateChild=false;
    RestrictionVector = ddManager.bddOne();
  }
  /// The work function for this object.
  virtual void operator ()(Formula *form, Options &opt);
};

/// A function object to setup the initial state set of the iteration process.

class InitialStateSetConstraint:public FormulaFunction
{
public:
  vector<vector<bool> > &NodeLevel;
  int MaxLevel;
  vector<BDD> &LevelBasedInitial;
  InitialStateSetConstraint(vector<vector<bool> > &NL, vector<BDD> &LBI, int ML):NodeLevel(NL),LevelBasedInitial(LBI)
  {
    MaxLevel=ML;
    bNodeCreateChild=false;
  }
  virtual void operator ()(Formula *form, Options &opt);
};

/// A function object to convert diamond to boxes for atoms.

class ConvertDiamondToBox:public FormulaFunction
{
public:
  ConvertDiamondToBox()
  {
    bNodeCreateChild=true;
  }
  virtual void operator ()(Formula *form, Options &opt);
};

/// A function object to convert formula to NNF for diamonds/semi-NNF/BNF for atoms.

class ConvertToNormalForm:public FormulaFunction
{
public:
  ConvertToNormalForm()
  {
    bNodeCreateChild=true;
  }
  virtual void operator ()(Formula *form, Options &opt);
};

/// A function object to convert all not nodes to negated edges for BNF.

class SnapBNFNegEdges:public FormulaFunction
{
public:
  SnapBNFNegEdges()
  {
    bNodeCreateChild=true;
  }
  virtual void operator ()(Formula *form, Options &opt);
};

// The two functions are for hash table implementation.
extern "C" {
  int FormulaNodeCompare(const char *a, const char *b);
  int FormulaNodeHash(char *a, int size);
};

/// A function object to convert the tree into a DAG.

class ConvertToDAG:public FormulaFunction
{
public:
  st_table *Table;
  ConvertToDAG()
  {
    Table=st_init_table(FormulaNodeCompare,FormulaNodeHash);
    bNodeCreateChild=false;
  }
  ~ConvertToDAG();
  virtual void operator ()(Formula *form, Options &opt);
};

/// A function object to return the modal depth of the formula.

class FindMaxLevel:public FormulaFunction
{
public:
  FindMaxLevel()
  {
    bNodeCreateChild=false;
  }
  virtual void operator ()(Formula *form, Options &opt);
};

/// A function object to return the size of the DAG representing the formula.

class GetSize:public FormulaFunction
{
public:
  GetSize()
  {
    bNodeCreateChild=false;
  }
  virtual void operator ()(Formula *form, Options &opt);
};

/// A function object to generate a table of which nodes appearing in which modal levels.

class GenerateNodeLevel:public FormulaFunction
{
public:
  vector<vector<bool> > *NodeLevel;
  GenerateNodeLevel(vector<vector<bool> > *TargetNodeLevel, int iNumNodes, int iLevel)
  {
    NodeLevel=TargetNodeLevel;
    int i,j;
    bNodeCreateChild=false;
    NodeLevel->resize(iNumNodes);
    for (i=0;i<iNumNodes;i++) {
      (*NodeLevel)[i].resize(iLevel);
      for (j=0;j<iLevel;j++) 
	(*NodeLevel)[i][j]=false;
    }
  }
  virtual void operator ()(Formula *form, Options &opt);
};

/// A function object to build the conjunction of all box witnesses as the transition relation.

class BuildMonolithicTransitionRelation:public FormulaFunction
{
public:
  BDD bddTrans;
  BuildMonolithicTransitionRelation()
  {
    bNodeCreateChild=false;
    bddTrans=ddManager.bddOne();
  }
  virtual void operator ()(Formula *form, Options &opt);
};

class GenerateNodeMinimalModalDepth:public FormulaFunction
{
public:
  GenerateNodeMinimalModalDepth()
  {
    bNodeCreateChild=false;
  }
  virtual void operator ()(Formula *form, Options &opt);
};

class BuildNodeList:public FormulaFunction
{
public:
  vector<Formula *> &NodeList;
  BuildNodeList(vector<Formula *> &NL):NodeList(NL) {
    bNodeCreateChild=false;
  }
  virtual void operator ()(Formula *form, Options &opt);
};

// The following is for parser

extern "C" {
#endif /* __cplusplus */
  void *build_formula(void *, Operator, int, void *);
  void *clone_formula(void *);
  int lwbparse();
  int tptpparse();
#ifdef __cplusplus
};
#endif

#endif
