#include "ksat.h"

/**
Build the transition relation recursively.
**/

void ModalWitness::operator()(Formula *form, Options &opt)
{
  // Build bdd for current formula
  BDD Left, Right;
  assert(opt.CurrentPhase==NORMALIZED);
  assert(form->Valid(BUILDBDD));
  
  if (opt.LeanVectors==false) {
    form->bddCache=ddManager.bddVar(form->iVarOrder);
    if ((form->opMain==OP_DIAMOND)||((opt.Particles==false)&&(form->opMain==OP_BOX))) {
      if (form->iIndex!=0xffffff) {
	//       cout<<"Adding Diamond"<<endl;
        DiamondNodeVector->push_back(form);
      }
    }
  } else { // Lean Vectors
    switch (form->opMain) {
    case OP_PROP:
    case OP_BOX:
    case OP_DIAMOND:
      form->bddCache=ddManager.bddVar(form->iVarOrder);
      if ((form->opMain==OP_DIAMOND)||((opt.Particles==false)&&(form->opMain==OP_BOX))) {
	if (form->iIndex!=0xffffff) {
	  //	  cout<<"Adding Diamond"<<endl;
	  DiamondNodeVector->push_back(form);
	}
      }
      break;
    /* For TOP/BOTTOM/NOT, they are part of the lean vectors so the standard
       witness should be ddManager.bddVar(form->iVarOrder). But since their
       value is garenteed, the following is used. */
    case OP_TOP:
      form->bddCache=ddManager.bddOne();
      break;
    case OP_BOTTOM:
      form->bddCache=ddManager.bddZero();
      break;
    case OP_NOT:
      assert(form->pfLeft!=NULL);
      form->bddCache=!(form->pfLeft->bddCache);
      break;
    case OP_AND:
    case OP_OR:
      assert(form->pfLeft!=NULL);
      assert(form->pfRight!=NULL);
      if (opt.Particles) {
	if (form->opMain==OP_AND) 
	  form->bddCache=(form->pfLeft->bddCache)*(form->pfRight->bddCache);
	else // OP_OR
	  form->bddCache=(form->pfLeft->bddCache)+(form->pfRight->bddCache);
      } else {
	if (form->bNegLeft)
	  Left=!form->pfLeft->bddCache;
	else
	  Left=form->pfLeft->bddCache;
	if (form->bNegRight)
	  Right=!form->pfRight->bddCache;
	else
	  Right=form->pfRight->bddCache;
	if (form->opMain==OP_AND) 
	  form->bddCache=Left*Right;
	else // OP_OR
	  form->bddCache=Left+Right;
      } 
      break;
    }
  }
  if (form->IsDiamond(opt)) {
    assert(form->pfLeft!=NULL);
    if ((opt.Particles==false)&&(!form->bNegLeft))
      form->bddWitness=!form->pfLeft->bddCache;
    else
      form->bddWitness=form->pfLeft->bddCache;
  }
  if (form->opMain==OP_BOX) {
    assert(form->pfLeft!=NULL);
    if ((opt.Particles==false)&&(form->bNegLeft))
      Left=!form->pfLeft->bddCache;
    else
      Left=form->pfLeft->bddCache;
    //    Left=form->pfLeft->bddCache;
    Left=Left.Permute(ImagePermutationVector);
    form->bddTrans=(!form->bddCache)+Left;
  }

  // Early clearing of the bdd cache. Needs tree traversal. May not work
  /*  if (form->pfLeft!=NULL)
    form->pfLeft->bddCache=ddManager.bddOne();
  if (form->pfRight!=NULL)
  form->pfRight->bddCache=ddManager.bddOne();*/
}

/**
Build the constraint of the start state set.
**/

void InitialStateSetConstraint::operator()(Formula *form, Options &opt)
{
  BDD Left,Right,Temp,Work;
  assert(opt.CurrentPhase==NORMALIZED);
  assert(form->Valid(BUILDBDD));

  switch (form->opMain) {
  case OP_PROP:
    Work=ddManager.bddOne();
    break;
  case OP_TOP:
    Work=ddManager.bddVar(form->iVarOrder);
    break;
  case OP_BOTTOM:
    Work=!ddManager.bddVar(form->iVarOrder);
    break;
  case OP_DIAMOND:
    if (opt.TopDown)
      Work=ddManager.bddOne();
    else
      Work=!ddManager.bddVar(form->iVarOrder);
    break;
   case OP_BOX:
     if ((opt.TopDown)||(opt.Particles))
      Work=ddManager.bddOne();
    else
      Work=ddManager.bddVar(form->iVarOrder);
    break;
  case OP_NOT: // Particles only
    assert(form->pfLeft!=NULL);
    assert(opt.Particles);
    Work=!(ddManager.bddVar(form->iVarOrder)*ddManager.bddVar(form->pfLeft->iVarOrder));
    break;
  default: // OP_AND, OP_OR constraints only generated for full vectors
    if (opt.LeanVectors)
      Work=ddManager.bddOne();
    else { // Full Vector
      switch (form->opMain) {
      case OP_AND:
      case OP_OR:
	assert(form->pfLeft!=NULL);
	assert(form->pfRight!=NULL);
	if ((opt.Particles==false)&&(form->bNegLeft))
	  Left=!ddManager.bddVar(form->pfLeft->iVarOrder);
	else
	  Left=ddManager.bddVar(form->pfLeft->iVarOrder);
	if ((opt.Particles==false)&&(form->bNegRight))
	  Right=!ddManager.bddVar(form->pfRight->iVarOrder);
	else
	  Right=ddManager.bddVar(form->pfRight->iVarOrder);
	if (form->opMain==OP_AND)
	  Temp=Left*Right;
	else // OP_OR
	  Temp=Left+Right;
	Work=!ddManager.bddVar(form->iVarOrder)+Temp;
	if (!opt.Particles)
	  Work*=!Temp+ddManager.bddVar(form->iVarOrder);
	break;
      } 
    }
  }
  if (!opt.UseLevelBasedInitialConstraint) {
  // All the propositional constraints for all nodes are added together
    if (form->pfLeft!=NULL) 
      Work=Work*form->pfLeft->bddCache;
    if (form->pfRight!=NULL)
      form->bddCache=Work*form->pfRight->bddCache;
    else
      form->bddCache=Work;
    if (form->pfLeft!=NULL)
      form->pfLeft->bddCache=ddManager.bddOne();
    if (form->pfRight!=NULL)
      form->pfRight->bddCache=ddManager.bddOne();
  } else {
    for (int i=0;i<MaxLevel;i++) {
      if (NodeLevel[form->iIndex][i])
	LevelBasedInitial[i]*=Work;
    }
  }
}

void BuildRestrictionVector::operator()(Formula *form, Options &opt)
{
  assert(opt.CurrentPhase==NORMALIZED);
  assert(form->Valid(BUILDBDD));
  form->bddCache=ddManager.bddOne();
  if (form->NodeUsed(opt)) {
    BDD l,r;
    if ((form->pfLeft!=NULL)&&(form->pfLeft->NodeUsed(opt))) {
      l = form->pfLeft->GetBDD();
    } else {
      l = ddManager.bddOne();
    }
    if (form->bNegLeft) {
      l = !l;
    }
    if ((form->pfRight!=NULL)&&(form->pfRight->NodeUsed(opt))) {
      r = form->pfRight->GetBDD();
    } else {
      r = ddManager.bddOne();
    }
    if (form->bNegRight) {
      r = !r;
    }
    switch (form->opMain) {
    case OP_OR:
      if (opt.Particles) {
	RestrictionVector*=(!(form->GetBDD())+(l+r));
      } else {
	RestrictionVector*=!(form->GetBDD())+(l+r);
	RestrictionVector*=form->GetBDD()+!(l+r);	
      }
      break;
    case OP_AND:
      if (opt.Particles) {
	RestrictionVector*=(!(form->GetBDD())+(l*r));
      } else {
	RestrictionVector*=(!(form->GetBDD())+(l*r));
	RestrictionVector*=form->GetBDD()+!(l*r);	
      }
      break;
    case OP_NOT:
      RestrictionVector*=!(form->GetBDD()*form->pfLeft->GetBDD());
      break;
    }
  }
}


/**
Convert all the diamond operators to not Box not for atoms.
**/

void ConvertDiamondToBox::operator()(Formula *form, Options &opt)
{
  Formula *newForm;
  Formula *newForm2;
  // No BDD have been build yet
  assert(opt.CurrentPhase==FILE_READ);
  assert(form->Valid());

  if (form->opMain==OP_DIAMOND) {
    newForm=new Formula(*form);
    // Copy this into newForm
    newForm->iRefCount=0;
    newForm->opMain=OP_BOX;
    // this becomes the not before the box
    form->opMain=OP_NOT;
    form->iSubscript=0;
    // newForm2 is the not after the box
    newForm2=new Formula;
    newForm2->opMain=OP_NOT;
    newForm2->SetLeftChild(form->pfLeft);
    newForm->SetLeftChild(newForm2);
    form->SetLeftChild(newForm);
  }
}

/**
Convert to NNF or BNF according to opt.
**/

void ConvertToNormalForm::operator()(Formula *form, Options &opt)
{
  Formula *newForm;
  assert(opt.CurrentPhase==FILE_READ);
  assert(form->Valid());

  if (form->bNegated) {
    switch (form->opMain) {
    case OP_TOP:
      form->opMain=OP_BOTTOM;
      break;
    case OP_BOTTOM:
      form->opMain=OP_TOP;
      break;
    case OP_PROP:
      newForm=new Formula(*form);
      newForm->bNegated=false;
      form->bNegated=false;
      form->opMain=OP_NOT;
      form->iSubscript=0;
      form->SetLeftChild(newForm);
      break;
    case OP_NOT: // Will be removed at a later phase
      form->opMain=OP_SNAP;
      break;
    case OP_AND:
    case OP_OR:
      form->pfLeft->bNegated=true;
      form->pfRight->bNegated=true;
      if (form->opMain==OP_AND)
	form->opMain=OP_OR;
      else // OP_OR
	form->opMain=OP_AND;
      break;
    case OP_DIAMOND: // Particles Only
      form->opMain=OP_BOX;
      form->pfLeft->bNegated=true;
      break;
    case OP_BOX:
      if (opt.Particles) {
	form->opMain=OP_DIAMOND;
	form->pfLeft->bNegated=true;
      } else {
	newForm=new Formula(*form);
	newForm->bNegated=false;
	form->bNegated=false;
	form->opMain=OP_NOT;
	form->SetLeftChild(newForm);
      }
      break;
    }
  } else { // Not negated
    if (form->opMain==OP_NOT) {
      form->opMain=OP_SNAP; // Remove it later
      form->pfLeft->bNegated=true;
      return;
    }
  }
  form->bNegated=false;
  assert(form->Valid());
}

/**
Snaps all not operaters to negated edges.
 **/

void SnapBNFNegEdges::operator()(Formula *form, Options &opt)
{
  assert(opt.CurrentPhase==PREPROCESSED);
  assert(form->Valid());
  
  if (form->pfLeft!=NULL) {
    while (form->pfLeft->opMain==OP_NOT) {
      form->bNegLeft=(form->bNegLeft?false:true);
      form->SetLeftChild(form->pfLeft->pfLeft);
    }  
  }
  if (form->pfRight!=NULL) {
    while (form->pfRight->opMain==OP_NOT) {
      form->bNegRight=(form->bNegRight?false:true);
      form->SetRightChild(form->pfRight->pfLeft);
    }  
  }     
}

/**
Assigns a depth for the node depends on the depth of its children.
**/

void FindMaxLevel::operator()(Formula *form, Options &opt)
{
  assert(opt.CurrentPhase==NORMALIZED);
  assert(form->Valid(BUILDBDD));
  switch (form->opMain) {
  case OP_DIAMOND:
  case OP_BOX:
    form->iLevel=form->pfLeft->iLevel+1;
    break;
  case OP_NOT:
    form->iLevel=form->pfLeft->iLevel;
    break;
  case OP_AND:
  case OP_OR:
    form->iLevel=max(form->pfLeft->iLevel, form->pfRight->iLevel);
    break;
  case OP_TOP:
  case OP_BOTTOM:
  case OP_PROP:
    form->iLevel=0;
    break;
  }
}

/**
Uses a counter to count the size of the DAG.
**/

void GetSize::operator()(Formula *form, Options &opt)
{
  assert(form->Valid());
  switch (form->opMain) {
  case OP_DIAMOND:
  case OP_BOX:
    form->iIndex=opt.iSize++;
    //    form->iVarOrder=opt.iBDDVars*2;
    //    form->iVarNextOrder=form->iVarOrder+1;
    //    opt.iBDDVars++;
    break;
  case OP_TOP:
  case OP_BOTTOM:
  case OP_PROP:
    form->iIndex=opt.iSize++;
    //    form->iVarOrder=opt.iBDDVars*2;
    //    form->iVarNextOrder=form->iVarOrder+1;
    //    opt.iBDDVars++;
    break;
  case OP_NOT:
    form->iIndex=opt.iSize++;
    //    form->iVarOrder=opt.iBDDVars*2;
    //    form->iVarNextOrder=form->iVarOrder+1;
    //    opt.iBDDVars++;
    break;
  case OP_AND:
  case OP_OR:
    form->iIndex=opt.iSize++;
    if (!opt.LeanVectors) {
      //      form->iVarOrder=opt.iBDDVars*2;
      //      form->iVarNextOrder=form->iVarOrder+1;
      //      opt.iBDDVars++;
    }
    break;
  }  
}

/**
Updates the node level table for each occurance of an operator at each level.
**/

void GenerateNodeLevel::operator()(Formula *form, Options &opt)
{
  assert(opt.CurrentPhase==NORMALIZED);
  assert(form->Valid(BUILDBDD));
  //  cout<<form->iIndex<<" "<<form->iLevel;
  switch (form->opMain) {
  case OP_DIAMOND:
  case OP_BOX:
    //    cout<<":"<<form->pfLeft->iIndex<<"%"<<endl;
    (*NodeLevel)[form->iIndex][form->iLevel]=true;
    form->pfLeft->iLevel=form->iLevel+1;
    break;
  case OP_TOP:
  case OP_BOTTOM:
  case OP_PROP:
    //    cout<<":"<<endl;
    (*NodeLevel)[form->iIndex][form->iLevel]=true;
    break;
  case OP_NOT:
    //    cout<<":"<<form->pfLeft->iIndex<<endl;
    (*NodeLevel)[form->iIndex][form->iLevel]=true;
    form->pfLeft->iLevel=form->iLevel;
    break;
  case OP_AND:
  case OP_OR:
    //    cout<<":"<<form->pfLeft->iIndex<<","<<form->pfRight->iIndex<<endl;
    if (opt.LeanVectors==false)
      (*NodeLevel)[form->iIndex][form->iLevel]=true;
    form->pfLeft->iLevel=form->iLevel;
    form->pfRight->iLevel=form->iLevel;
    break;
  }  
}

void BuildMonolithicTransitionRelation::operator()(Formula *form, Options &opt)
{
  /*  if (form->opMain==OP_BOX) {
    bddTrans*=form->bddTrans;
    }*/
  switch (form->opMain) {
  case OP_AND:
  case OP_OR:
    form->bddTrans=form->pfLeft->bddTrans*form->pfRight->bddTrans;
    form->pfLeft->bddTrans=form->pfRight->bddTrans=ddManager.bddOne();
    break;
  case OP_DIAMOND:
  case OP_NOT:
    form->bddTrans=form->pfLeft->bddTrans;
    form->pfLeft->bddTrans=ddManager.bddOne();
    break;
  case OP_BOX:
    form->bddTrans*=form->pfLeft->bddTrans;
    form->pfLeft->bddTrans=ddManager.bddOne();
    break;
  }
}

void GenerateNodeMinimalModalDepth::operator()(Formula *form, Options &opt)
{
  assert(opt.CurrentPhase==NORMALIZED);
  assert(form->Valid(BUILDBDD));
  switch (form->opMain) {
  case OP_DIAMOND:
  case OP_BOX:
    form->pfLeft->iLevel=min(form->pfLeft->iLevel,form->iLevel+1);
    break;
  case OP_NOT:
    form->pfLeft->iLevel=min(form->pfLeft->iLevel,form->iLevel);
    break;
  case OP_AND:
  case OP_OR:
    form->pfLeft->iLevel=min(form->pfLeft->iLevel,form->iLevel);
    form->pfRight->iLevel=min(form->pfRight->iLevel,form->iLevel);
    break;
  }
}

void BuildNodeList::operator ()(Formula *form, Options &opt)
{
  assert(form->Valid());
  if (form->NodeUsed(opt)) {
    NodeList.push_back(form);
  }
}
