#include "ksat.h"
#include <set>
#include "graph.h"
#include <algorithm>
#include <fstream>
#include <vector>
#include <hash_set>
#include "cuddInt.h"
#include "Formula.h"

void AssignSimpleVarOrdering(Formula *form, int &offset, Options &opt)
{
  if (form->mark)
    return;
  form->mark=true;
  if (form->NodeUsed(opt)) {
    form->iVarOrder=offset*2;
    form->iVarNextOrder=offset*2+1;
    offset++;
    opt.iBDDVars = MAX(opt.iBDDVars, offset*2);
  }
  switch (form->opMain) {
  case OP_AND:
  case OP_OR:
    AssignSimpleVarOrdering(form->pfLeft, offset, opt);
    AssignSimpleVarOrdering(form->pfRight, offset, opt);
    break;
  case OP_NOT:
  case OP_BOX:
  case OP_DIAMOND:
    AssignSimpleVarOrdering(form->pfLeft, offset, opt);
    break;
  }    
}

void AssignAlternateVarOrdering(Formula *form, int &offset, Options &opt)
{
  if (form->mark)
    return;
  form->mark=true;
  switch (form->opMain) {
  case OP_AND:
  case OP_OR:
    AssignAlternateVarOrdering(form->pfLeft, offset, opt);
    AssignAlternateVarOrdering(form->pfRight, offset, opt);
    break;
  case OP_NOT:
  case OP_BOX:
  case OP_DIAMOND:
    AssignAlternateVarOrdering(form->pfLeft, offset, opt);
    break;
  }    
  if (form->NodeUsed(opt)) {
    form->iVarOrder=offset*2;
    form->iVarNextOrder=offset*2+1;
    offset++;
    opt.iBDDVars = MAX(opt.iBDDVars, offset*2);
  }
}

void AssignLevelVarOrdering(Formula *form, KSATChecker &checker)
{
  vector<Formula *> &f=checker.NodeList;
  vector<vector<bool> > &nl = checker.Support.NodeLevel;

  for (int i=0;i<f.size();i++) {
    f[i]->iVarOrder = -1;
  }
  int maxLevel = nl[0].size();
  int var = 0;
  for (int i=0;i<maxLevel;i++) {
    for (int j=0;j<f.size();j++) {
      int index = f[j]->iIndex;
      if (nl[index][i]) {
	if (f[j]->iVarOrder == -1) {
	  f[j]->iVarOrder = var*2;
	  f[j]->iVarNextOrder = var*2+1;
	  var++;
	}
      }
    }
  }
}

set<int> getChildren(Formula *form)
{
  set<int> s,r,t;
  switch (form->opMain) {
  case OP_AND:
  case OP_OR:
    t = getChildren(form->pfLeft);
    r = getChildren(form->pfRight);
    t.insert(r.begin(), r.end());
    return t;
  case OP_NOT:
    s = getChildren(form->pfLeft);
    s.insert(form->iVarOrder);
    return s;
  case OP_BOX:
  case OP_DIAMOND:
  case OP_PROP:
      s.insert(form->iVarOrder);
      return s;
  }
  return s;
}

int getMaxProposition(Formula *form)
{
  int result,a,b;
  switch (form->opMain) {
  case OP_AND: case OP_OR:
    a = getMaxProposition(form->pfLeft);
    b = getMaxProposition(form->pfRight);
    result=(a>b?a:b);
    break;
  case OP_NOT: case OP_BOX: case OP_DIAMOND:
    result=getMaxProposition(form->pfLeft);
    break;
  case OP_PROP:
    result=form->iSubscript;
    break;
  case OP_TOP: case OP_BOTTOM:
    result=0;
    break;
  }
  return result;
}

int getDepth(Formula *form)
{
  int result;
  switch (form->opMain) {
  case OP_AND: case OP_OR:
    result=max(getDepth(form->pfLeft), getDepth(form->pfRight));
    break;
  case OP_NOT:
    result=getDepth(form->pfLeft);
    break;
  case OP_BOX: case OP_DIAMOND:
    result=getDepth(form->pfLeft)+1;
    break;
  case OP_PROP: case OP_TOP: case OP_BOTTOM:
    result=1;
    break;
  }
  return result;
}

void getAllChildren(Formula *form, vector<set<int> >& siblings, vector<Formula *>& forms, bool expend, Options &opt)
{
  assert(form->iVarOrder < forms.size());
  forms[form->iVarOrder] = form;
  if (form->NodeUsed(opt))
    siblings[form->iVarOrder] = getChildren(form);
  switch (form->opMain) {
  case OP_AND:
  case OP_OR:
    getAllChildren(form->pfLeft, siblings, forms, false, opt);
    getAllChildren(form->pfRight, siblings, forms, false, opt);
    break;
  case OP_NOT:
  case OP_BOX:
  case OP_DIAMOND:
    getAllChildren(form->pfLeft, siblings, forms, form->opMain!=OP_NOT, opt);
    break;
  }
}

void updateSiblings(Formula *form, vector<set<int> >&siblings, vector<Formula *>& forms, Formula *root, Options &opt)
{
  switch (form->opMain) {
  case OP_AND:
  case OP_OR:
    updateSiblings(form->pfLeft, siblings, forms, root,opt);
    updateSiblings(form->pfRight, siblings, forms, root,opt);
    break;
  case OP_NOT:
    updateSiblings(form->pfLeft, siblings, forms, root,opt);
    break;
  case OP_BOX:
  case OP_DIAMOND:
    updateSiblings(form->pfLeft, siblings, forms, form,opt);
    break;
  }
  if (form->NodeUsed(opt))
    siblings[form->iVarOrder].insert(siblings[root->iVarOrder].begin(), siblings[root->iVarOrder].end());
}

void BuildDependencyGraph(Formula *form, Options &opt, vector<set<int> > siblings,vector<Formula *>& forms,  Graph &g)
{
  if (form->NodeUsed(opt)) {
    set<int> sib = siblings[form->iIndex];
    for (set<int>::iterator p=sib.begin();p!=sib.end();p++) {
      if (*p!=form->iIndex) {
	g.addEdge(form->iIndex, *p);
	// self w/ sibling's children?
	Formula *sf = forms[*p];
	if ((sf->opMain==OP_BOX)||(sf->opMain==OP_DIAMOND)) {
	  set<int> s2 = siblings[sf->pfLeft->iIndex];
	  //for (set<int>::iterator q=s2.begin();q!=s2.end();q++)
	    //g.addEdge(form->iIndex, *q);
	    g.addEdge(form->iIndex, sf->pfLeft->iIndex);
	}
      }
    }
  }
}

void UpdateNodePropositionRelation(Formula *form, vector<set<pair<int, int> > >&nodeProp, int MaxLevel)
{
  pair<int, int> work2;
  work2.first = form->iIndex;
  //cout<<"E "<<form->iIndex<<" "<<form->opMain<<endl;
  switch (form->opMain) {
  case OP_AND: case OP_OR:
    UpdateNodePropositionRelation(form->pfLeft, nodeProp, MaxLevel);
    UpdateNodePropositionRelation(form->pfRight, nodeProp, MaxLevel);
    for (int i=0;i<nodeProp.size();i++) {
      set<pair<int, int> > *propRel = &nodeProp[i];
      pair<int, int> work;
      work.first = form->pfLeft->iIndex;
      for (int j=0;j<MaxLevel;j++) {
	work.second = j;
	set<pair<int, int> >::iterator p;
	if ((p=propRel->find(work))!=propRel->end()) {
	  work2.second = p->second;
	  propRel->insert(work2);
	}
      }      
    }
    for (int i=0;i<nodeProp.size();i++) {
      set<pair<int, int> > *propRel = &nodeProp[i];
      pair<int, int> work;
      work.first = form->pfRight->iIndex;
      for (int j=0;j<MaxLevel;j++) {
	work.second = j;
	set<pair<int, int> >::iterator p;
	if ((p=propRel->find(work))!=propRel->end()) {
	  work2.second = p->second;
	  propRel->insert(work2);
	}
      }      
    }
    break;
  case OP_NOT:
    UpdateNodePropositionRelation(form->pfLeft, nodeProp, MaxLevel);
    for (int i=0;i<nodeProp.size();i++) {
      set<pair<int, int> > *propRel = &nodeProp[i];
      pair<int, int> work;
      work.first = form->pfLeft->iIndex;
      for (int j=0;j<MaxLevel;j++) {
	work.second = j;
	set<pair<int, int> >::iterator p;
	if ((p=propRel->find(work))!=propRel->end()) {
	  work2.second = p->second;
	  propRel->insert(work2);
	}
      }      
    }
    break;
  case OP_BOX:
  case OP_DIAMOND:
    UpdateNodePropositionRelation(form->pfLeft, nodeProp, MaxLevel);
    for (int i=0;i<nodeProp.size();i++) {
      set<pair<int, int> > *propRel = &nodeProp[i];
      pair<int, int> work;
      work.first = form->pfLeft->iIndex;
      for (int j=0;j<MaxLevel;j++) {
	//	cout<<j<<flush;
	work.second = j;
	set<pair<int, int> >::iterator p;
	if ((p=propRel->find(work))!=propRel->end()) {
	  work2.second = p->second+1;
	  propRel->insert(work2);
	}
      }      
    }
    break;
  case OP_PROP:
    set<pair<int, int> > *propRel = &nodeProp[form->iSubscript];
    work2.second = 0;
    propRel->insert(work2);
    break;
  }
  //cout<<"X "<<form->iIndex<<endl;
}

void setNodeLevel(Formula *form, vector<vector<int> >&nodeLevel, int level)
{
  switch(form->opMain) {
  case OP_AND: case OP_OR:
    setNodeLevel(form->pfLeft, nodeLevel, level);
    setNodeLevel(form->pfRight, nodeLevel, level);
    nodeLevel[level][form->iIndex] = 1;
    break;
  case OP_NOT:
    setNodeLevel(form->pfLeft, nodeLevel, level);
    nodeLevel[level][form->iIndex] = 1;
    break;
  case OP_BOX: case OP_DIAMOND:
    setNodeLevel(form->pfLeft, nodeLevel, level+1);
    nodeLevel[level][form->iIndex] = 1;
    break;
  case OP_PROP:
    nodeLevel[level][form->iIndex] = 1;
    break;
  }
}

int getFormulaSize(Formula *form)
{
  int result;
  switch (form->opMain) {
  case OP_AND: case OP_OR:
    result=max(form->iIndex, max(getFormulaSize(form->pfLeft), getFormulaSize(form->pfRight)));
    break;
  case OP_NOT: case OP_BOX: case OP_DIAMOND:
    result=max(form->iIndex, getFormulaSize(form->pfLeft));
    break;
  case OP_PROP: case OP_TOP: case OP_BOTTOM:
    result=form->iIndex;
    break;
  }
  //cout<<result<<endl;
  return result;
}

void BuildAlternateDependencyGraph(Formula *form, Options &opt, Graph &g, vector<Formula *> f)
{
  int maxProps = getMaxProposition(form)+1;
  vector<set<pair<int, int> > > nodeProp;
  cout<<"M:"<<maxProps<<endl;
  nodeProp.resize(maxProps);
  int maxLevel = getDepth(form);
  cout<<"L:"<<maxLevel<<endl;
  UpdateNodePropositionRelation(form, nodeProp, maxLevel);
  int formSize = getFormulaSize(form)+1;
  cout<<"S:"<<formSize<<endl;
  vector<vector<int> > nodeLevel;
  nodeLevel.resize(maxLevel);
  for (int i=0;i<maxLevel;i++) {
    nodeLevel[i].resize(formSize);
    for (int j=0;j<formSize;j++)
      nodeLevel[i][j] = 0;
  }
  setNodeLevel(form, nodeLevel, 0);
  /*  HyperGraph gg;
  // Build hypergraph here
  gg.Vertices = formSize;
  for (int l=0;l<maxProps;l++) {
    for (int m=0;m<maxLevel-1;m++) {
      vector<int> work;
      work.resize(0);
      for (int i=0;i<formSize;i++) {
	set<pair<int, int> > *look = &nodeProp[l];
	//if (look->find(pair<int, int>(i, m))==look->end()) continue;
	if ((look->find(pair<int, int>(i, m))==look->end()) && (look->find(pair<int, int>(i, m))==look->end())) continue;
	if (f[i]->NodeUsed(opt))
	  work.push_back(i);
      }
      if (work.size()==0) continue;
      gg.addEdge(work);
    }
  }
  gg.outputCAPO();
  */
  /*
  for (int i=0;i<formSize;i++) {
    for (int j=i+1;j<formSize;j++) {
      for (int k=0;k<maxLevel;k++) {
	if (!nodeLevel[k][i]) continue;
	if (!nodeLevel[k][j]) continue;
	// ok same level
	for (int l=0;l<maxProps;l++) {
	  for (int m=0;m<maxLevel;m++) {
	    set<pair<int, int> > *work = &nodeProp[l];
	    if (work->find(pair<int, int>(i, m))==work->end()) continue;
	    if (work->find(pair<int, int>(j, m))==work->end()) continue;
	    //Both vars appear
	    g.addEdge(i,j);
	    goto next;
	  }
	}
	//	cout<<i<<" "<<j<<" "<<k<<" "<<endl;
      }
    next:
      ;
    }
  }
  */
  
  vector<set<int> > siblings;
  //  vector<Formula *> f;
  f.resize(opt.iSize);
  siblings.resize(opt.iSize);
  getAllChildren(form, siblings, f, true, opt);
  updateSiblings(form, siblings, f, form, opt);
  
  for (int i=0;i<formSize;i++) {
    if (f[i]->opMain!=OP_BOX) continue;
    if (f[i]->opMain!=OP_DIAMOND) continue;
    int child = f[i]->pfLeft->iIndex;
    for (set<int>::iterator p=siblings[child].begin();p!=siblings[child].end();p++) {
      int j = *p;
      if (i==j) continue;
      for (int k=0;k<maxProps;k++) {
        for (int l=0;l<maxLevel-1;l++) {
	  set<pair<int, int> > *work = &nodeProp[k];
	  if (work->find(pair<int, int>(i,l+1))==work->end()) continue;
	  if (work->find(pair<int, int>(j,l))==work->end()) continue;
	  g.addEdge(i,j);
	  cout<<"#"<<flush;
	  goto next2;
	}
      }
    }
  next2:
    ;
  }
  
  /*
  for (int i=0;i<formSize;i++) {
    for (int j=0;j<formSize;j++) {
      if (i==j) continue;
      for (int m=0;m<maxLevel-1;m++) {
	if (!nodeLevel[m][i]) continue;
	if (!nodeLevel[m+1][j]) continue;	
	for (int k=0;k<maxProps;k++) {
	  for (int l=0;l<maxLevel-1;l++) {
	    set<pair<int, int> > *work = &nodeProp[k];
	    if (work->find(pair<int, int>(i,l+1))==work->end()) continue;
	    if (work->find(pair<int, int>(j,l))==work->end()) continue;
	    g.addEdge(i,j);
	    goto next2;
	  }
	}
      }
    next2:
      ;
    }
  }
  */
}

void BuildSimpleGraph(Formula *form, vector<Formula *> &forms, Options &opt, Graph &g)
{
  forms[form->iIndex] = form;
  switch (form->opMain) {
  case OP_AND:
  case OP_OR:
    g.addEdge(form->iIndex, form->pfLeft->iIndex);
    g.addEdge(form->iIndex, form->pfRight->iIndex);
    BuildSimpleGraph(form->pfLeft, forms, opt, g);
    BuildSimpleGraph(form->pfRight, forms, opt, g);
    break;
  case OP_NOT:
  case OP_BOX:
  case OP_DIAMOND:
    g.addEdge(form->iIndex, form->pfLeft->iIndex);
    BuildSimpleGraph(form->pfLeft, forms, opt, g);
    break;
  }    
}

void AdjustOrder(vector<Formula *> f, vector<int> &order)
{
  // Move negations together with the corresponding proposition
  for (int i=0;i<order.size();i++) {
    int x = order[i];
    if (f[x]->opMain==OP_NOT) {
      int j;
      int y;
      for (j=0;j<order.size();j++) {
	y = order[j];
	if ((f[y]->opMain==OP_PROP)&&(f[x]->pfLeft==f[y]))
	  break;
      }
      if (abs(x-y)!=1)
	cout<<"Dist="<<x-y<<" "<<x<<" "<<y<<endl;
      if (abs(x-y)==1) continue;
      if (j<i) {
	// slide j to i
	for (int k=j;k<i-1;k++)
	  order[k]=order[k+1];
	order[i-1] = y;
      } else {
	for (int k=j;k>i+1;k--)
	  order[k]=order[k-1];
	order[i+1]=y;
      }
    }
  }
  // reverse
  for (int i=0;i<order.size()/2;i++)
    swap(order[i], order[order.size()-i-1]);
}

void AssignTestVarOrdering(Formula *form, Options &opt)
{
  vector<set<int> > siblings;
  vector<Formula *> f;
  Graph g;
  f.resize(opt.iSize);
  siblings.resize(opt.iSize);
  g.setSize(opt.iSize);
  getAllChildren(form, siblings, f, true, opt);
  /*updateSiblings(form, siblings, f, form);
  for (int i=0;i<f.size();i++) {
    BuildDependencyGraph(f[i], opt, siblings, f, g);
    }*/
  BuildAlternateDependencyGraph(form, opt, g, f);
  vector<int> result = HyperGraph::readCAPO(opt.iSize);
  //vector<int> result = g.OptimizeMatlab();
  //BuildSimpleGraph(form, f, opt, g);
  //  g.printInfo();
  //  BandwidthOptimizer b(&g);
  //b.init();
  //b.greedy2();
  //cout<<f.size()<<endl;
  //cout<<b.bandwidth()<<endl;
  //  b.optimize();
  /*  b.hillClimb();
      //  cout<<b.bandwidth()<<endl;
  */
  //vector<int> result= b.getOrder();
  //  result = b.BranchAndBound();
  //  result = b.getOrder();
  
  /*for (int i=0;i<result.size();i++)
    cout<<result[i]<<" "<<flush;*/
  //AdjustOrder(f, result);
  for (int i=0;i<f.size();i++) {
    if (f[i]->NodeUsed(opt)) {
      int j;
      for (j=0;j<result.size();j++) {
	if (result[j]==i) break;
      }
      f[i]->iVarOrder = j*2;
      f[i]->iVarNextOrder = j*2+1;
    }
  }

  opt.iBDDVars = f.size()*2;
}

void FillFormulaVector(Formula** f, int *n, Formula *form, Options &opt)
{
  if (form->NodeUsed(opt)) {
    f[form->iVarOrder] = form;
    f[form->iVarNextOrder] = form;
    n[form->iVarNextOrder] = 1;
    //    cout<<form->iVarOrder<<","<<form->iVarNextOrder<<endl;
  }
  switch (form->opMain) {
  case OP_AND:
  case OP_OR:
    FillFormulaVector(f, n, form->pfLeft, opt);
    FillFormulaVector(f, n, form->pfRight, opt);
    break;
  case OP_NOT:
  case OP_DIAMOND:
  case OP_BOX:
    FillFormulaVector(f, n, form->pfLeft, opt);
    break;
  }
}

bool Subformula(Formula *a, Formula *b, Options &opt, bool top)
{
  if (a == b)
    return true;
  if (a->NodeUsed(opt) && !top)
    return false;
  switch (a->opMain) {
  case OP_AND:
  case OP_OR:
    return (Subformula(a->pfLeft, b, opt, 0) || Subformula(a->pfRight, b, opt, false));
    break;
  case OP_NOT:
    return Subformula(a->pfLeft, b, opt, false);
    break;
  case OP_DIAMOND:
  case OP_BOX:
    break;
  }
  return false;
}

static int dvo_count = 0;

void DumpVO(KSATChecker &checker)
{
  DdManager *mgr = ddManager.getManager();
  int vars = checker.iBDDVars;
  int *perm = mgr->perm;
  char fn[256];
  sprintf(fn, "var.order%d", dvo_count++);
  FILE *f = fopen(fn, "w");
  for (int i=0;i<vars;i++) {
    fprintf(f, "%d\n", perm[i]);
  }
  fclose(f);
}

void RenameVariables(Formula *f, int *perm, Options &opt)
{
  if (f->mark==true)
    return;
  if (f->NodeUsed(opt)) {
    f->iVarOrder = perm[f->iVarOrder];
    f->iVarNextOrder = perm[f->iVarNextOrder];
  }
  f->bddWitness = f->bddTrans = ddManager.bddOne();
  f->bddCache = ddManager.bddOne();
  f->mark = true;
  switch (f->opMain) {
  case OP_AND: case OP_OR:
    RenameVariables(f->pfLeft, perm, opt);
    RenameVariables(f->pfRight, perm, opt);
    break;
  case OP_NOT: case OP_DIAMOND: case OP_BOX:
    RenameVariables(f->pfLeft, perm, opt);
    break;
  }
}

void LoadVO(KSATChecker &checker)
{
  DdManager *mgr = ddManager.getManager();
  int vars = checker.iBDDVars;
  int *perm = new int[vars];
  int i;
  FILE *f = fopen("order.dat","r");
  for (i=0;i<vars;i++) {
    int j;
    fscanf(f, "%d", &j);
    perm[j] = i;
  }
  fclose(f);
  if (checker.opt.Test2) {
    ddManagerReordered = Cudd(vars);
    /*    for (int i=0;i<vars;i++) {
      ddManagerReordered.bddNewVarAtLevel(i);
      }*/
    DdManager *mgr = ddManagerReordered.getManager();
    memcpy(mgr->perm, perm, vars*sizeof(int));
    //   ddManagerReordered.ShuffleHeap(perm);
    checker.StateSet = checker.StateSet.Transfer(ddManagerReordered);
    for (int i=0;i<checker.LevelBasedInitial.size();i++) {
      checker.LevelBasedInitial[i] = checker.LevelBasedInitial[i].Transfer(ddManagerReordered);
    }
    checker.Support.tr.Transfer(ddManagerReordered);
    for (int i=0;i<checker.Support.ProjectionVectorByLevel.size();i++) {
      checker.Support.ProjectionVectorByLevel[i] = checker.Support.ProjectionVectorByLevel[i].Transfer(ddManagerReordered);
    }
  } else {
    //    if (!checker.opt.DynamicVariableReordering) {
    int *perminv = new int[vars];
    for (int i=0;i<vars;i++) 
      perminv[perm[i]] = i;
    
    RenameVariables(checker.formula, perminv, checker.opt);
    delete[] perminv;
    checker.formula->Clear();
    checker.LevelBasedInitial.resize(0);
    checker.Initial = checker.StateSet = checker.SentenceWitness = ddManager.bddOne();
    for (int i=0;i<vars;i++) 
    perm[i] = i;
    ddManager.ShuffleHeap(perm);
    //      ddManager.info();
      //      memcpy(mgr->perm, perm, vars*sizeof(int));
      //    } else {
      //ddManager.ShuffleHeap(perm);
      //    }
  }
  int *p = mgr->perm; 
  /*  for (i=0;i<vars;i++) {
    assert(perm[i]==p[i]);
    }*/
  delete[] perm;
}

void TryMATLABOrder()
{
  unsetenv("DISPLAY");
  system("matlab < varorder.m");
  
}
/*
vector<int> GetLevelOrder(int level, int vars, vector<vector<bool> > &nl)
{ 
  DdManager *mgr = ddManager.getManager();
  int *perm = mgr->perm;
  vector<pair<int, int> > work;
  for (int i=0;i<vars;i+=2) {
    if (nl[i][level]) {
      work.push_back(pair<int, int>(perm[i], i));
    }
  }
  sort(work.begin(), work.end());
  vector<int> ret;
  for (int i=0;i<work.size();i++) {
    ret.push_back(work[i].second);
  }
  return ret;
}
*/
vector<int> GetLevelOrder(int level, int vars, vector<vector<bool> > &nl, MediumVariableRelationGraph &rel)
{
  vector<int> work;

  for (int i=0;i<vars;i+=2) {
    if (!nl[i][level])
      continue;
    int v = i;
    vector<int> e;
    for (int j=0;j<work.size();j++) {
      int vv = work[j];
      if (rel.haveEdge(v, vv))
	e.push_back(j);
    }
    int ins;
    int size = e.size();
    if (size == 0) {
      ins = 0;
    } else if (size == 1) {
      ins = e[0];
    } else {
      ins = (e[size/2]+e[(size+1)/2])/2;
    }
    work.insert(&work[ins], v);
  }
  return work;
}

template<> class hash<pair<int, int> > {
public:
  size_t operator()(const pair<int, int> &p) const {
    hash<int> h;
    return (h(p.first)^h(p.second));
  }
};

/* hash set implementation of relation is too slow. */
/* Should use a ordered table. */

void MergeLevelOrder(vector<int> &all, vector<int> cur, vector<int> next, /*hash_set<pair<int, int> >*/ /*vector<hash_set<int> >*/ MediumVariableRelationGraph &rel, int vars)
{
  int lastpos=-1;
  for (int i=0;i<next.size();i++) {
    int weight = 0x7fffffff;
    int pos = -1;
    int candidate = next[i];

    int evars;
    evars = 0;
    int buf[vars];
    for (int j=0;j<all.size();j++) {
      int var = all[j];
      if (candidate==var)
	goto do_not_add;
      if (rel.haveEdge(candidate, var))
	buf[evars++] = j;
    }
    // take the median
    if (evars%2==0) { //even 
      int i1 = buf[(evars-1)/2];
      int i2 = buf[(evars-1)/2+1];
      pos = (i1+i2)/2;
    } else { //odd
      int i1 = buf[(evars-1)/2];
      pos = i1;
    } 
    /*    if (pos <= lastpos)
	  pos = lastpos+1;*/

#if 0
    for (int j=0;j<=all.size();j++) {
      int sum_distance;
      sum_distance = 0;
      for (int k=0;k<j;k++) {
	int var = all[k];
	if (candidate==var)
	  goto do_not_add;
	if (candidate <= var) {
	  //	  if (rel.find(pair<int, int>(candidate, var))!=rel.end()) 
	  //	  if (rel[candidate].find(var)!=rel[candidate].end())
	  if (rel.haveEdge(candidate, var))
	    sum_distance += j-k;
	} else {
	  //	  if (rel.find(pair<int, int>(var, candidate))!=rel.end()) 
	  //	  if (rel[var].find(candidate)!=rel[var].end())
	  if (rel.haveEdge(var, candidate))
	    sum_distance += j-k;
	}
      }
      for (int k=j;k<all.size();k++) {
	int var = all[k];
	if (candidate==var)
	  goto do_not_add;
	if (candidate<=var) {
	  //	  if (rel.find(pair<int, int>(candidate, var))!=rel.end()) 
	  //	  if (rel[candidate].find(var)!=rel[candidate].end())
	  if (rel.haveEdge(candidate, var))
	    sum_distance += k-j+1;
	} else {
	  //	  if (rel.find(pair<int, int>(var, candidate))!=rel.end()) 
	  //	  if (rel[var].find(candidate)!=rel[var].end())
	  if (rel.haveEdge(var, candidate))
	    sum_distance += k-j+1;
	}
      }
      if (sum_distance < weight) {
	weight = sum_distance;
	pos = j;
      }
    }
#endif
    // insert candidate at j;
    all.insert(&all[pos], candidate);
    lastpos = pos;
  do_not_add:
    ;
  }
}

void WriteLevelOrder(vector<int> order)
{
  FILE *f;
  f = fopen("order.dat","w");
  vector<int> ord;
  ord.resize(order.size()*2);
  for (int i=0;i<order.size();i++) {
    ord[order[i]] = i*2;
    ord[order[i]+1] = i*2+1;
  }
  for (int i=0;i<ord.size();i++) {
    fprintf(f, "%d\n", ord[i]);
  }
  fclose(f);
}

void BuildNLVars(vector<vector<bool> > &nl, Formula *form)
{
  switch (form->opMain) {
  case OP_DIAMOND:
  case OP_BOX:
    nl[form->iVarOrder][form->iLevel]=true;
    form->pfLeft->iLevel=form->iLevel+1;
    BuildNLVars(nl, form->pfLeft);
    break;
  case OP_TOP:
  case OP_BOTTOM:
  case OP_PROP:
    nl[form->iVarOrder][form->iLevel]=true;
    break;
  case OP_NOT:
    nl[form->iVarOrder][form->iLevel]=true;
    form->pfLeft->iLevel=form->iLevel;
    BuildNLVars(nl, form->pfLeft);
    break;
  case OP_AND:
  case OP_OR:
    form->pfLeft->iLevel=form->iLevel;
    BuildNLVars(nl, form->pfLeft);
    form->pfRight->iLevel=form->iLevel;
    BuildNLVars(nl, form->pfRight);
    break;
  }  
}

void TryOrder(KSATChecker &checker, /*hash_set<pair<int, int> >*/ /*vector<hash_set<int> >*/ MediumVariableRelationGraph &rel)
{
  vector<vector<bool> > nl;
  int maxLevel = checker.Support.NodeLevel[0].size();
  vector<int> curLevelOrder;
  vector<int> nextLevelOrder;
  vector<int> allLevelOrder;
  int vars = checker.iBDDVars;

  nl.resize(vars);
  for (int i=0;i<vars;i++) {
    nl[i].resize(maxLevel);
    for (int j=0;j<maxLevel;j++) {
      nl[i][j]=false;
    }
  }
  BuildNLVars(nl, checker.formula);

  /* Take the ordering of level 0 */
  //  curLevelOrder = GetLevelOrder(0, vars, nl);
  curLevelOrder = GetLevelOrder(0, vars, nl, rel);
  allLevelOrder = curLevelOrder;
  //cout<<allLevelOrder.size()<<endl;
  
  for (int i=1;i<maxLevel;i++) {
    //    nextLevelOrder = GetLevelOrder(i, vars, nl);
    nextLevelOrder = GetLevelOrder(i, vars, nl, rel);
    MergeLevelOrder(allLevelOrder, curLevelOrder, nextLevelOrder, rel, vars);
    //    cout<<nextLevelOrder.size()<<" "<<allLevelOrder.size()<<endl;
    curLevelOrder = nextLevelOrder;
    // cout<<"%"<<flush;
  }
  WriteLevelOrder(allLevelOrder);
}

void DumpVarOrder(KSATChecker &checker, Formula *form, Options &opt)
{
  DdManager *mgr = ddManager.getManager();
  int vars = checker.iBDDVars;
  Formula **f = new Formula*[vars];
  int *n = new int[vars];
  memset(n, 0, sizeof(int)*vars);
  for (int i=0;i<vars;i++) {
    f[i] = (Formula*)0xdeadbeef;
  }
  FillFormulaVector(f, n, form, opt);

    

  vector<pair<int, int> >buf;
  for (int i=0;i<vars;i++) {
    buf.push_back(pair<int, int>(mgr->perm[i], i));
  }
  //  sort(buf.begin(), buf.end());

  vector<set<int> > siblings;
  vector<Formula *> vf;
  vf.resize(vars);
  siblings.resize(vars);
  getAllChildren(form, siblings, vf, true, opt);
  updateSiblings(form, siblings, vf, form, opt);
  
//  FILE *of = fopen("order.info","w");
  //  ofstream os2("order.form");
  int vv = vars;
  /*  int *rel = new int[vv*vv];
      memset(rel, 0, sizeof(int)*vv*vv);*/
  //  hash_set<pair<int, int> > rel;
  //  vector<hash_set<int> > rel;
  //  rel.resize(vars);
  MediumVariableRelationGraph rel(vars);

  for (int i=0;i<vars;i++) {
    if (i%2==1)
      continue;
    set<int> child = getChildren(vf[i]);
    for (set<int>::iterator p=child.begin();p!=child.end();p++) {
      //      rel[i*vv+*p] = rel[*p*vv+i] = 1;
      if (*p<i) {
	//	rel.insert(pair<int, int>(*p, i));
	//	rel[*p].insert(i);
	rel.addEdge(*p, i);
      } else {
	//rel.insert(pair<int, int>(i, *p));
	//rel[i].insert(*p);
	rel.addEdge(i, *p);
      }
    }
    for (set<int>::iterator p=siblings[i].begin();p!=siblings[i].end();p++) {
      int s = *p;
      if (*p<i) {
	rel.addEdge(*p, i);
      } else {
	rel.addEdge(i, *p);
      }
      set<int> child = getChildren(vf[s]);
      for (set<int>::iterator q=child.begin();q!=child.end();q++) {
	//rel[i*vv+*q] = rel[*q*vv+i] = 1;
	if (*q<i) {
	  //	  rel.insert(pair<int, int>(*q, i));
	  //rel[*q].insert(i);
	  rel.addEdge(*q, i);
	} else {
	  //rel.insert(pair<int, int>(i, *q));
	  //  rel[i].insert(*q);
	  rel.addEdge(i, *q);
	}
      }
    }
  }
  /*for (int i=0;i<vars;i++) {
    if (n[i]) 
      os2<<"'"<<*f[buf[i].second]<<endl;
    else
      os2<<*f[buf[i].second]<<endl;
      }*/
 /* for (int i=0;i<vars;i++) {
    for (int j=0;j<vars-1;j++) {
      fprintf(of, "%d,", rel[i*vv+j]);
    }
    fprintf(of, "%d\n", rel[i*vv+vars-1]);
  }*/
  //os2.close();
  //fclose(of);
  //   TryMATLABOrder();
  TryOrder(checker, rel);
  delete[] n;
  delete[] f;
  //  delete[] rel;
}
