#include "alphabetize.h"


/**
 * Returns a vector of atomic propositions found in the formula
 * @formula. If @add_parens has been set, the atomic propositions will
 * have parens around them.
 */
void
mm::alphabetize::get_ap_vector(std::string* formula, svector_t* atoms_vector, bool add_parens) {
  sset_t atoms_set;
  get_ap(formula, &atoms_set, add_parens);
  
  for (sset_t::const_iterator it = atoms_set.begin();
       it != atoms_set.end();
       it ++)
    {
      std::string atom = *it;
#ifdef DEBUG_ALPHABETIZE
      std::cout << __FILE__ << ": " << __func__ << ": New atom found: "
		<< atom << std::endl;
#endif
      atoms_vector->push_back(atom);
    }
}


/**
 * Uses the canonical order of the atoms in @atoms_vector to generate
 * a vector of all possible truth assignments to the atomic
 * propositions. Returns the result in the provided container
 * @truth_assignments
 */
void
mm::alphabetize::vector2assgnmts(svector_t* atoms_vector, v_sbmap_t* truth_assignments) {

  int num_atomic = atoms_vector->size();
  bool ba[num_atomic];
  int bit = 0;

  for (int i = 0; i < (1 << num_atomic); i++) {
    
    // Get the bit representation of i
    for (int j = num_atomic - 1; j >= 0; j--) {
      bit = ((i >> j) & 1);
      ba[num_atomic - j - 1] = bit;
    }

    // Generate the corresponding truth assignment. An atom appears
    // positively if its bit is 1
    std::map< std::string, bool>* truth_assignment = new std::map< std::string, bool>;
    for (int i = 0; i < num_atomic; i++) {
      std::string atom = (*atoms_vector)[i];
      bool bool_value = ba[i];
      (*truth_assignment)[atom] = bool_value;
    }
    
    truth_assignments->push_back(truth_assignment);
  }
}


/**
 * Uses the atomic propositions in @formula to generate a vector of
 * all possible truth assignements to the atomic propositions.
 */
void
mm::alphabetize::form2assgnmts(std::string* formula, v_sbmap_t* truth_assignments, bool add_parens) {
  
  svector_t atoms_vector;
  get_ap_vector(formula, &atoms_vector, add_parens);
  vector2assgnmts(&atoms_vector, truth_assignments);
}


/**
 * Given a transition guard @trans, determines which truth assignments
 * from the set of truth assignments @ta would make the transition
 * true. The indices of those truth assignments are returned in the
 * set @letters.
 */
void
mm::alphabetize::trans2letters(std::string* trans, v_sbmap_t* ta, std::set<int>* letters) {
  
#ifdef DEBUG_ALPHABETIZE
  std::cout << __FILE__ << "::" <<__func__ << "(): trans = ->" << *trans << "<-" << std::endl;
#endif

  bool_parser bp;
  bool_formula* bf = bp.parse(*trans);

  for (unsigned int i=0; i < ta->size(); i++) {
    if (bf->eval((*ta)[i])) {
      letters->insert(i);
#ifdef DEBUG_ALPHABETIZE
  std::cout << __FILE__ << "::" <<__func__ << ": Matching index found:  = ->" << i << "<-" << std::endl;
#endif      
    }
  }
}






// /**
//  * Minimizes the @source automaton and stores the result into the
//  * @minimized container, which is then returned.
//  */
// intermediate_rep*
// alphabetize_and_minimize(intermediate_rep* source_automaton,
// 			 monitor_params* mp,
// 			 global_params* gp)
// {
//   alphabetization_t alphabetization = gp->get_alphabetization();
  
//   intermediate_rep* minimized = new intermediate_rep();
//   std::string ltl_string;
//   mp->get_ltl(&ltl_string);
//   alphabetize_and_minimize(source_automaton, minimized, &ltl_string, gp);
//   return minimized;
// }
