#include "augment.h"


namespace mm {
  namespace augment {



/**
 * Generates the set of all possible words of (@num_atoms - @index)
 * letters, where the atoms can only have indices bigger than @index.
 */
void bool_combinations(std::string* atoms_array, 
		       svector_t* conj, int num_atoms, int index) {
  
  if (index == num_atoms - 1) {
    conj->push_back(" (" + atoms_array[index] + ")");
    conj->push_back("!(" + atoms_array[index] + ")");
    conj->push_back(std::string());
    return;
  }
  
  svector_t* suffix = new svector_t;
  bool_combinations(atoms_array, suffix, num_atoms, index + 1);
  
  for (svector_t::const_iterator it = suffix->begin();
       it != suffix->end();
       it ++) 
    {
      if (it->size() == 0) {
	conj->push_back(" (" + atoms_array[index] + ")");
	conj->push_back("!(" + atoms_array[index] + ")");

	// Remove the "empty" atom at the top level
	if (index != 0) {
	  conj->push_back(std::string());
	}

      }
      else {
	conj->push_back(" (" + atoms_array[index] + ") && " + *it);
	conj->push_back("!(" + atoms_array[index] + ") && " + *it);
	conj->push_back(*it);
      }
    }
  delete suffix;
  return;
}





/**
 * Generates the set of all possible n-letter words where each letter
 * is an atom that occurs either positively or negatively, and n is
 * the number of atoms
 */
void bool_iterate(std::string* atoms_array, svector_t* conj, int num_atoms) {
  bool ba[num_atoms];
  int bit = 0;

  for (int i = 0; i < (1 << num_atoms); i++) {
    
    // Get the bit representation of i
    for (int j = num_atoms - 1; j >= 0; j--) {
      bit = ((i >> j) & 1);
      ba[num_atoms - j - 1] = bit;
    }

    // Generate the corresponding conjunction. An atom appears positively if its bit is 
    std::string current;
    for (int i = 0; i < num_atoms; i++) {
      current += "(";
      current += ((ba[i]) ? " " : "!");
      current += "(" + atoms_array[i] + ")";
      current += ")";
      current += ((i < num_atoms - 1) ? " && " : "");
    }
    
    conj->push_back(current);
  }
}


/**
 * Generates all possible conjuctions of the atoms in the set @atoms,
 * and returns them in a list of strings, where each element of the
 * list is a conjunction of atoms or their negations.
 */
void formula2conj(std::string* formula, svector_t* conj, alpha_reduction_t ar) {
  
  sset_t* atoms = new sset_t;
  get_ap(formula, atoms);
  
  int num_atoms = atoms->size();
  
  // Need some fixed order of the atoms
  std::string atoms_array[num_atoms];
  int count = 0;
  for (sset_t::const_iterator it = atoms->begin();
       it != atoms->end();
       it ++)
    {
      atoms_array[count] = *it;
      count ++;
    }

  delete atoms;

  if (ar == PARTIAL_REDUCTION) {
    bool_iterate(atoms_array, conj, num_atoms);
  }
  else if (ar == FULL_REDUCTION) {
    bool_combinations(atoms_array, conj, num_atoms, 0);
  }
  else if (ar == NO_REDUCTION) {
    std::cout << __FILE__ << ": " << __func__ << ": ";
    std::cout << "Called to form conjunctions in the formula " 
	      << "without setting the alpha_reduction." << std::endl;
    exit(1);
  }
  else {
    std::cout << __FILE__ << ": " << __func__ << ": ";
    std::cout << "Unknown type of alpha_reduction: " << ar << std::endl;
    exit(1);
  }
  
  
  
}


/**
 * Generates a test program for determining all possible augmentations
 * of the formula @formula. Result is output to the stream @os.
 */
void conj2code(svector_t* conj, global_params* gp, monitor_params* mp, std::ostream& os) {
  
  gp->includefiles_to_stream(os);
  os << std::endl << std::endl;
  os << "class " << gp->get_mon_name() << "0 {";
  os << "  int foo(int i";
  
  ssmap_t vartypes = mp->get_vartypes();
  for (ssmap_t::const_iterator it = vartypes.begin(); 
       it != vartypes.end();
       ++it )
    {
      os << ", " << it->second << " " << it->first;
    }
  
  os << ") {" << std::endl;
  
  for (svector_t::const_iterator it = conj->begin();
       it != conj->end();
       it ++)
    {
      // Remove the quotes from the atoms
      std::string unquoted;
      std::string quoted = *it;
      remove_str(&quoted, "\"", &unquoted);
      os << "    if (" << unquoted << ") { i++; }" << std::endl; 
    }
  os << "    return(i);" << std::endl;
  os << "  }" << std::endl;
  os << "}; //class" << std::endl;
  
}


/**
 * Return a string containing the file name of a file that does not
 * currently exist in the current directory. Therefore we can safely
 * create the file without overwriting anything that currently exists.
 */
// void get_unique_filenames(std::string* src_cont, // File with source code
// 			  std::string* obj_cont, // Object file
// 			  std::string* txt_cont, // Compilation output file
// 			  global_params* gp) 
// {
  
//   std::string scratch;
//   gp->get_scratch_location(&scratch);
  
//   // Make sure that the scratch directory ends with a "/"
//   if (scratch.find_last_of('/') != (scratch.size() - 1)) {
//     scratch.append("/");
//   }
  
//   std::string base_name = scratch + std::string("mm_alpha_optim");
//   std::string try_name = base_name;
  
//   struct stat src_file_info, obj_file_info, txt_file_info;
//   int src_status, obj_status, txt_status;
//   int counter = rand();

//   while (true) {
    
//     std::string src_name = try_name + ".cc";
//     std::string obj_name = try_name + ".o";
//     std::string txt_name = try_name + ".compile";
    
//     src_status = stat(src_name.c_str(), & src_file_info);
//     obj_status = stat(obj_name.c_str(), & obj_file_info);
//     txt_status = stat(txt_name.c_str(), & txt_file_info);

//     if ((src_status == 0) || (obj_status == 0) || (txt_status == 0)) {
//       // One of the files exists
//       counter = rand();
      
//       std::stringstream ss;
//       ss << counter;
//       try_name = base_name + ss.str();
//     }
//     else {
//       // Files do not exist
//       src_cont->assign(src_name);
//       obj_cont->assign(obj_name);
//       txt_cont->assign(txt_name);
      
//       return;
//     }
//   }
// }



/**
 * Generates a test file to determine which conjunctions are mutually exclusive
 */
void optimize_conj(global_params* gp, monitor_params* mp, sset_t* mutex_conj) {
  
  std::string src_filename, obj_filename, cmp_filename, scratch;
  std::string formula;
  mp->get_ltl( &formula );
  gp->get_scratch_location(&scratch);

  get_unique_file_name(& src_filename, & scratch, "alpha_optim", "cc");
  get_unique_file_name(& obj_filename, & scratch, "alpha_optim", "o");
  get_unique_file_name(& cmp_filename, & scratch, "alpha_optim", "compile");
  
  std::ofstream& out_strm = *open_for_output(src_filename.c_str());

  svector_t* conj = new svector_t;
  formula2conj(&formula, conj, gp->get_alpha_reduction());
  conj2code(conj, gp, mp, out_strm);

  close_file(& out_strm);

  std::string systemc_home;
  gp->get_systemc_home(&systemc_home);
  
  // Make sure that the directory name ends with a "/"
  if (systemc_home.find_last_of('/') != (systemc_home.size() - 1)) {
    systemc_home.append("/");
  }
  
  // Piece together the command line to compile the file
  std::string cmd_line = "gcc -O3 -DMONITOR_0 -c " + src_filename +
    " -o " + obj_filename +
    " -I " + systemc_home + "include/" +
    " &> " + cmp_filename;
  
  std::cout << __FILE__ << ": compiling the optimization file with " << std::endl;
  std::cout << cmd_line << std::endl;
  system(cmd_line.c_str());

  // Figure out which line of code has the first conjunction
  int offset = gp->get_includefiles().size() + 4;
  
  std::ifstream in_strm( cmp_filename.c_str(), std::ios::in);
  std::string curr_line;
  
  do {
    getline(in_strm, curr_line);
    
    if (curr_line.find("mutually exclusive equal-tests") != std::string::npos) {
      int index = curr_line.find(src_filename);
      index += src_filename.size();
      index = curr_line.find_first_of(':', index);
      int index2 = curr_line.find_first_of(':', index + 1);
      int line_number = atoi(curr_line.substr(index + 1, index2-index).c_str());

      mutex_conj->insert(conj->at(line_number - offset));
    }
  }while (! in_strm.eof());
  
  in_strm.close();
  std::cout << "Will be able to eliminate " << mutex_conj->size() 
	    << " out of " << conj->size() << " conjuctions." << std::endl;

  delete conj;
  return;
}


/**
 * Augments the formula with a conjunction of constraints on the atoms
 * in the formula of the type G!(atom_i && atom_j && ...). The
 * augmented formula is returned in the string @output.
 */
void augment_formula(global_params* gp, monitor_params* mp, std::string* output) {
  
  sset_t mutex_conj;
  mp->get_ltl(output);
  optimize_conj(gp, mp, &mutex_conj);

  for (sset_t::const_iterator it = mutex_conj.begin();
       it != mutex_conj.end();
       it++)
    {
      output->append(" && G!(" + *it + ")");
    }
}


/**
 * Augments the formula (in place) with a conjunction of constraints on the atoms
 * in the formula of the type G!(atom_i && atom_j && ...).
 */
void augment_in_place(global_params* gp, monitor_params* mp) {
  std::string new_ltl;
  augment_formula(gp, mp, &new_ltl);
  mp->reset_ltl(&new_ltl);
}


  } // namespace augment
} // namespace mm
