#include "automaton_tools.h"

namespace mm {

/**
 * A simple visitor class that counts the number of states in the
 * automaton. Used in automaton_tools::count_states() below.
 */
class sba_counter_visitor : public spot::tgba_reachable_iterator_breadth_first {
public:
  
  sba_counter_visitor(const spot::tgba_sba_proxy* a,
		      int* state_count_container,
		      int* trans_count_container) :
    tgba_reachable_iterator_breadth_first(a),
    accept_all_(-1), 
    states_count(state_count_container),
    trans_count(trans_count_container)
  {
  }
  
  
  void
  start()
  {
    init_ = automata_->get_init_state();
  }

  void
  end()
  {
    if (accept_all_ != -1) {
      // Add one more state from where we accept anything
      (*states_count) = (*states_count) + 1;
    }
      delete init_;
  }

  bool
  state_is_accepting(const spot::state *s)
  {
    return
      dynamic_cast<const spot::tgba_sba_proxy*>(automata_)->state_is_accepting(s);
  }

  
  void
  process_state(const spot::state* s, int n, spot::tgba_succ_iterator*)
  {
    (*states_count) = (*states_count) + 1;
    
    spot::tgba_succ_iterator* it = automata_->succ_iter(s);
    it->first();
    if (! it->done())
      {
	spot::state* current = it->current_state();
	if (state_is_accepting(s)
	    && it->current_condition() == bddtrue
	    && s->compare(init_) != 0
	    && s->compare(current) == 0)
	  {
	    accept_all_ = n;
	  }
	delete current;
      }
    delete it;
  }

  void
  process_link(const spot::state*, int in, const spot::state*, int out,
	       const spot::tgba_succ_iterator* si)
  {
    if (in != accept_all_)
      {
	(*trans_count) = (*trans_count) + 1;
      }
  }

private:
  int accept_all_;
  spot::state* init_;
  int* states_count;
  int* trans_count;
  
}; //Class




//////////////////////////////////////////
//
//  class automaton_tools
//
////////////////////////////////
const spot::tgba_sba_proxy*
automaton_tools::ltl2tgba(std::string ltl) {
  
  start_timer();
  bool debug_spot = 0;
  
  // Code fragments borrowed from SPOT's ltl2tgba.cc
  spot::ltl::environment& env(spot::ltl::default_environment::instance());
  dict = new spot::bdd_dict();
  
  f = spot::ltl::parse(ltl.c_str(), pel, env, debug_spot);
  spot::ltl::format_parse_errors(std::cerr, ltl.c_str(), pel);
  
  to_free = a = spot::ltl_to_tgba_lacim(f, dict);
  reduced = new spot::tgba_reduc(a);
  reduced->prune_scc();
  
  a = degeneralized = new spot::tgba_sba_proxy(reduced);
  a = tgba_dupexp_dfs(a);
  
  s = static_cast<const spot::tgba_sba_proxy*>(degeneralized);

  display_and_reset_timer("SPOT");

#ifdef AUTOMATON_STATS
  int sba_states = 0;
  int sba_transitions = 0;
  automaton_tools::count_states_and_trans(s, &sba_states, &sba_transitions);
  std::cout << "Number of states in SBA = " << sba_states << std::endl;
  std::cout << "Number of transitions in SBA = " << sba_transitions << std::endl;
#endif
  
  return s;
}




void
automaton_tools::cleanup() {
  
  if (f)
    spot::ltl::destroy(f);

  delete a;


//   delete degeneralized;
//   std::cout << "Deleted the degeneralized " << std::endl;
  delete to_free;
  delete reduced;
  

  
  assert( spot::ltl::atomic_prop::instance_count() == 0 );
  assert( spot::ltl::unop::instance_count() == 0 );
  assert( spot::ltl::binop::instance_count() == 0 );
  assert( spot::ltl::multop::instance_count() == 0 );
  //std::cout << "About to delete the dictionary " << std::endl;
  ///delete dict;
}


void
automaton_tools::count_states_and_trans(const spot::tgba_sba_proxy* g, int* st, int* tr) {
  
  sba_counter_visitor cntr(g, st, tr);
  cntr.run();
}


// void 
// automaton_tools::generate_callbacks(const sset_t clocks_set, 
// 				    std::ostream &os, 
// 				    const std::string fun_name) {
//   // Insert the appropriate callback
  
  
//   //   virtual void callback_statement_level_trigger(std::string statement);
//   //   virtual void callback_event_notified(sc_event* event);

//   for (sset_t::iterator it = clocks_set.begin();
//        it != clocks_set.end();
//        ++it)
//     {
//       std::string clk = *it;

//       if ((strcasecmp(clk.c_str(), "MON_INIT_PHASE_BEGIN") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_init_phase_begin() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;      
// 	}

//       if ((strcasecmp(clk.c_str(), "MON_INIT_PHASE_END") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_init_phase_end() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;
// 	}

//       if ((strcasecmp(clk.c_str(), "MON_INIT_UPDATE_PHASE_BEGIN") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_init_update_phase_begin() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;      
// 	}

//       if ((strcasecmp(clk.c_str(), "MON_INIT_UPDATE_PHASE_END") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_init_update_phase_end() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;   
// 	} 

//       if ((strcasecmp(clk.c_str(), "MON_INIT_DELTA_NOTIFY_PHASE_END") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_init_delta_notify_phase_begin() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;
// 	}


//       if ((strcasecmp(clk.c_str(), "MON_INIT_DELTA_NOTIFY_PHASE_END") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_init_delta_notify_phase_end() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;
// 	}


//       if ((strcasecmp(clk.c_str(), "MON_DELTA_CYCLE_BEGIN") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_delta_cycle_begin() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;      
// 	}


//       if ((strcasecmp(clk.c_str(), "MON_DELTA_CYCLE_END") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_delta_cycle_end() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;      
// 	}

//       if ((strcasecmp(clk.c_str(), "MON_EVALUATION_PHASE_BEGIN") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_evaluation_phase_begin() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;      
// 	}


//       if ((strcasecmp(clk.c_str(), "MON_EVALUATION_PHASE_END") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_evaluation_phase_end() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;      
// 	}
    

//       if ((strcasecmp(clk.c_str(), "MON_UPDATE_PHASE_BEGIN") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_update_phase_begin() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;      
// 	}


//       if ((strcasecmp(clk.c_str(), "MON_UPDATE_PHASE_END") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_update_phase_end() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;      
// 	}

//       if ((strcasecmp(clk.c_str(), "MON_DELTA_NOTIFY_PHASE_BEGIN") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_delta_notify_phase_begin() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;      
// 	}


//       if ((strcasecmp(clk.c_str(), "MON_DELTA_NOTIFY_PHASE_END") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_delta_notify_phase_end() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;      
// 	}


//       if ((strcasecmp(clk.c_str(), "MON_TIMED_NOTIFY_PHASE_BEGIN") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_timed_notify_phase_begin() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;      
// 	}


//       if ((strcasecmp(clk.c_str(), "MON_TIMED_NOTIFY_PHASE_END") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_timed_notify_phase_end() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;      
// 	}


//       if ((strcasecmp(clk.c_str(), "MON_METHOD_SUSPEND") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_method_suspend() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;      
// 	}


//       if ((strcasecmp(clk.c_str(), "MON_THREAD_SUSPEND") == 0) || 
// 	  strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
// 	{
// 	  os << "  virtual void callback_thread_suspend() {" << std::endl;
// 	  os << "    " << fun_name << ";" << std::endl;
// 	  os << "  }" << std::endl << std::endl;      
// 	}
//     }
// }


void 
automaton_tools::generate_monitor_registration_obsolete(const sset_t clocks_set, 
					       std::ostream &os,
					       const std::string obs_name) {
  
  for (sset_t::iterator it = clocks_set.begin();
       it != clocks_set.end();
       ++it)
    {
      std::string clk = *it;
      if (strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0) {
	os << "    " << obs_name << "->register_monitor(this, MON_INIT_PHASE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_INIT_PHASE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_INIT_PHASE_END);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_INIT_UPDATE_PHASE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_INIT_UPDATE_PHASE_END);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_INIT_DELTA_NOTIFY_PHASE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_INIT_DELTA_NOTIFY_PHASE_END);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_DELTA_CYCLE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_DELTA_CYCLE_END);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_EVALUATION_PHASE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_EVALUATION_PHASE_END);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_UPDATE_PHASE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_UPDATE_PHASE_END);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_DELTA_NOTIFY_PHASE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_DELTA_NOTIFY_PHASE_END);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_TIMED_NOTIFY_PHASE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_TIMED_NOTIFY_PHASE_END);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_METHOD_SUSPEND);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_THREAD_SUSPEND);" << std::endl;
      }
  
      else {
	os << "    " << obs_name << "->register_monitor(this, " << clk << ");" << std::endl;
      }
    }
}


/**
 * Here we assume that @formula is quote-escaped, so that we can
 * insert it directly in the std::cout stream without messing up the
 * quotes in the file.
 */
void
automaton_tools::generate_mon_helpers_obsolete(std::ostream& os, const std::string formula) {
  os << "  void property_failed() {" << std::endl;
  os << "#ifdef MONITOR_REPORT_FAIL_IMMEDIATELY" << std::endl;
  os << "    SC_REPORT_WARNING(\"Property failed\", \"Monitor for property " << formula << " has reached an error state\");" << std::endl;
  os << "    std::cout << \"Property failed after \" << num_steps << \" steps\" << std::endl;" << std::endl;
  os << "#endif" << std::endl;
  os << "    status = MON_FAIL;" << std::endl;    
  os << "  }" << std::endl << std::endl;    
  
  os << "  void property_satisfied() {" << std::endl;
  os << "#ifdef MONITOR_REPORT_PASS_IMMEDIATELY" << std::endl;
  os << "    std::cout << \"The property \" << to_string() << \"HOLDS\" << std::endl;" << std::endl;
  os << "    std::cout << \"Property cannot fail after \" << num_steps << \" steps\" << std::endl;" << std::endl;
  os << "#endif" << std::endl;
  os << "    status = MON_PASS;" << std::endl;
  os << "  }" << std::endl << std::endl;
  
  os << "  const char* to_string() const {" << std::endl;
  os << "    return \""<< formula  << "\";" << std::endl;
  os << "  }" << std::endl << std::endl;
  
  os << "  const mon_status_t get_status() const {" << std::endl;
  os << "#ifdef PRINT_NUM_STEPS_IN_STATUS_CALL" << std::endl;
  os << "    std::cout << to_string() << \" made \" << num_steps << std::endl;" << std::endl;
  os << "#endif" << std::endl;
  os << "    return status;" << std::endl;
  os << "  }" << std::endl << std::endl;
}





/**
 * This function generates the code for the local observer.
 */
void
automaton_tools::generate_local_observer(std::ostream& os,
					 std::ostream& header_os,
					 global_params* gp) {

  header_os << std::endl << std::endl;
  header_os << "/**" << std::endl;
  header_os << " * This object is derived from mon_observer defined in the SystemC" << std::endl;
  header_os << " * modified kernel. Thus, we can pass it as an argument to the classes" << std::endl;
  header_os << " * implementing the monitors." << std::endl;
  header_os << " */" << std::endl;  
  header_os << "class local_observer : public sc_core::mon_observer {" << std::endl << std::endl;
  header_os << " public:" << std::endl << std::endl;
  header_os << "  // The constructor" << std::endl;
  header_os << "  local_observer(unsigned int verbosity, unsigned int num_monitors";

  // Determine what objects need to be passed to the observer
  std::stringstream observer_objects;
  for(ssmap_t::const_iterator it = gp->all_objects.begin();
      it != gp->all_objects.end();
      ++it)
    {
      observer_objects << ", " << it->second << " " << it->first;
    }

  header_os << observer_objects.str() << ");"
	    << std::endl << std::endl;

  // Now the implementation
  os << std::endl << std::endl;
  os << "// The constructor" << std::endl;
  os << "local_observer::local_observer(unsigned int verbosity, unsigned int num_monitors";
  os << observer_objects.str() << ") : sc_core::mon_observer(verbosity) {"
     << std::endl << std::endl;

  // Instantiate the monitors
  for(mpset_t::iterator it = gp->all_monitors.begin();
      it != gp->all_monitors.end();
      ++it)
    {
      monitor_params* mp = *it;

      std::string id;
      mp->get_id_str(&id);
      std::string this_monitor_name = gp->get_mon_name() + id;

      std::string this_monitor_inst = this_monitor_name + "_inst";

      // Allow multiple instances of the monitor, for testing purposes
      os << "  for (int i = 0; i < num_monitors; i++) {" << std::endl;
      
      os << "    " << this_monitor_name << "* " << this_monitor_inst
	 << " = new " << this_monitor_name << "(this";

      //Determine if this monitor needs any objects passed
      for(ssmap_t::const_iterator sit = mp->get_vartypes().begin();
	  sit != mp->get_vartypes().end();
	  ++sit)
	{
	  os << ", " << sit->first;
	}
      
      os << ");" << std::endl;
      
      os << "    mons.push_back(" << this_monitor_inst << ");" << std::endl;
      os << "  } // for loop" << std::endl;
    }
  os << "} // constructor" << std::endl << std::endl << std::endl;
  // Constructor ends here.

  header_os << "  void report_status(std::ostream& os);" << std::endl;
  
  os << "/**" << std::endl;
  os << " * Report the status of each monitor. Output to @os" << std::endl;
  os << " */" << std::endl;
  os << "void" << std::endl;
  os << "local_observer::report_status(std::ostream& os) {" << std::endl;
  os << "  mon_status_t status = MON_FAIL;" << std::endl;
  os << "  os << \"Monitor Status Report\" << std::endl;" << std::endl << std::endl;
  os << "  for (sc_core::vect_mon_t::const_iterator it = mons.begin();" << std::endl;
  os << "       it != mons.end();" << std::endl;
  os << "       it++)" << std::endl;
  os << "    {" << std::endl;
  os << "      status = (*it)->get_status();" << std::endl;
  os << "      os << \"Monitor \\t\" << (*it)->to_string() << std::endl;" << std::endl;
  os << "      os << \"Status: \\t\" << ((status == MON_FAIL)? \"FAILED\" : (status == MON_PASS) ? \"HOLDS\" : \"UNDETERMINED (DID NOT FAIL)\") << std::endl << std::endl;" << std::endl << std::endl;
  os << "    }" << std::endl;
  os << "} // report_status" << std::endl;

  
  header_os << "}; // class local_observer" << std::endl;
  
}


} // namespace mm
