#include "bool_parser.h"

bool_parser::bool_parser() {
  index = 0;
  back_index = 0;
}


bool_formula*
bool_parser::parse(std::string s){
#ifdef DEBUG
  std::cout << "DEBUG: parse() ->" << s << "<-" << std::endl;
#endif

  input_size = s.size();
  input = new char[input_size + 1];
  strncpy(input, s.c_str(), input_size);
  
  //strncpy does not null-terminate the string
  input[input_size] = 0;
  skip_white_space();

#ifdef DEBUG
  std::cout << "DEBUG: parse() ->" << input << "<-" << std::endl;
  std::cout << "DEBUG: parse() length:" << input_size << std::endl;
#endif


  
  bool_formula* bf = parse_to_precedence(MAX_PREC);
  
#ifdef DEBUG
  std::cout << "DEBUG: parse() parsed ->" << bf->to_string() << "<-" << std::endl;
#endif


  // Check if there is any input left over.
  if (index != input_size) {
    fprintf(stderr, "Parsing complete before end of string. Leftover stuff: %s\n",
	    input + index);
  }

  return bf;
  
}


/**
 * This function parses a term according to the precedence of its
 * operators; for example, a term "a or b and c" will be parsed as a
 * or (b and c) since "and" has a lower precedence than "or".
 */
bool_formula* 
bool_parser::parse_to_precedence(op_prec_t prec) {

  bool_formula *f1, *f2;
  bool_formula* ret_val = 0;
  bool_operator* op;

#ifdef DEBUG
  std::cout << "DEBUG: parse_to_prec(" << prec << ")" << input + index << "<- " << std::endl;
#endif

  f1 = read_one_term();

#ifdef DEBUG
  std::cout << "DEBUG: read_one_term() done" << std::endl;
  std::cout << "DEBUG: finished parsing " << f1->to_string() << std::endl;
#endif

  back_index = index;

  // read an operator
  op_prec_t pr;   // the precedence of the operator
  std::string s;
  
#ifdef DEBUG
  std::cout << "DEBUG: parse_to_precedence(): about to enter the while() loop" << std::endl;
  std::cout << "DEBUG: where index = " << index << " and input_size = " << input_size << std::endl;
#endif

  while (index < input_size) {
    std::string* s_ptr = next_token();
    s.assign(*s_ptr);
    delete s_ptr;
    
    
    // if s is a closing bracket, then f1 is the largest subterm
    // that can be read
    if (s.compare(std::string(")")) == 0) {
      index = back_index;

#ifdef DEBUG
      std::cout << "DEBUG: parse_to_precedence(): about to return" << std::endl;
      std::cout << "DEBUG: this value: ->" << f1->to_string() << "<-" << std::endl;
#endif
      return f1;
    }
    
    op = get_operator(s);
    pr = op->get_precedence();

#ifdef DEBUG
    std::cout << "DEBUG: parse_to_precedence(): parsed the operator ->" << op->to_string() << "<-" << std::endl;
    std::cout << "DEBUG: right here prec = " << prec << " and pr = " << pr << std::endl;
#endif  
    
    // only read further if the precedence of the new operator
    // is no greater than prec
    if (pr > prec) {
      index = back_index;
      
#ifdef DEBUG
      std::cout << "DEBUG: parse_to_precedence(): about to return" << std::endl;
      std::cout << "DEBUG: this value: ->" << f1->to_string() << "<-" << std::endl;
#endif
      delete op;
      return f1;
    }
    else {

      // op has lower precedence
      f2 = parse_to_precedence(pr);
      ret_val = new bool_formula(op, f1, f2);

#ifdef DEBUG
      std::cout << "DEBUG: parse_to_precedence(): about to return" << std::endl;
      std::cout << "DEBUG: this value: " << ret_val->to_string() << std::endl;
#endif
      
      return ret_val;
    }
  } // while

#ifdef DEBUG
  std::cout << "DEBUG: parse_to_precedence(): about to return" << std::endl;
  std::cout << "DEBUG: this value: " << f1->to_string() << std::endl;
#endif
  return f1;
}


/**
 * Parse the smallest initial segment that can be read as a valid
 * term. Return a bool_formula* representing the initial term
 */
bool_formula*
bool_parser::read_one_term() {

#ifdef DEBUG
  std::cout << "DEBUG: read_one_term() called" << std::endl;
#endif
  
  bool_formula* f;
  std::string* s = next_token();

#ifdef DEBUG
  std::cout << "DEBUG: finished reading next token" << std::endl;
  std::cout << "DEBUG: token just read was ->" << *s << "<-" << std::endl;
#endif


  if (s->compare(std::string("(")) == 0) {
    f = parse_to_precedence(MAX_PREC);
    consume(std::string(")"));
    delete s;
    return f;    
  }

  if (s->compare(std::string("!")) == 0) {
    f = parse_to_precedence(NOT_PREC);
    delete s;
    return new bool_formula(new op_not(), f);    
  }

  if (s->compare(std::string("1")) == 0) {
    delete s;
    return new bool_formula(new op_true());
  }

  if (s->compare(std::string("0")) == 0) {
    delete s;
    return new bool_formula(new op_false());    
  }  

  f = new bool_formula(new op_atom(s));
  return f;
}


/**
 * Parse and return the smallest initial segment that can be read as a
 * token.
 */
std::string*
bool_parser::next_token() {

#ifdef DEBUG
  std::cout << "DEBUG: next_token() called" << std::endl;
#endif

  
  if (input[index] == '(') {
    index++;
    skip_white_space();
    return new std::string("(");
  }
  
  if (input[index] == ')') {
    index++;
    skip_white_space();
    return new std::string(")");
  }

  return read_token();
}


/**
 * Remove any white space in the input. The value of @index is
 * incremented while the current input is white space
 */
void 
bool_parser::skip_white_space() {
  while (( index < input_size ) && 
	 (input[index] == ' ' ||
	  input[index] == '\t' || 
	  input[index] == '\n' || 
	  input[index] == '\r'))
    {
      index++;
    }
}


/**
 * Reads and returns a token from the input
 */
std::string*
bool_parser::read_token() {
  
#ifdef DEBUG
  std::cout << "DEBUG: read_token() called" << std::endl;
  std::cout << "DEBUG: right now index = " << index << std::endl;
#endif

  
  int start = index;
  
  // See if we can parse a Boolean operator
  if (input[index] == '&') {
    // This must be a conjunction
    std::string* s = new std::string("&&");
    consume(*s);
    return s;
  }
  
  if (input[index] == '|') {
    // This must be a disjunction
    std::string* s = new std::string("||");
    consume(*s);
    return s;
  }

  if (input[index] == '!') {
    // This must be a negation
    std::string* s = new std::string("!");
    consume(*s);
    return s;
  }

  // Else, we are looking for an atom

  bool quoted_atom = (input[index] == '"');
  
  if (quoted_atom) {
#ifdef DEBUG
    std::cout << "DEBUG: read_token() looking for a quoted atom" << std::endl;
#endif
    // Look for the closing quote. Skip over everything else
    index++;
    while (input[index] != '"') {
      index++;
      if (index == input_size) {
	fprintf(stderr, "Ran out of input while trying to parse a quoted atom \n");
	fprintf(stderr, "from here: %s\n", input + start);
	exit(1);
      }
    }
    // Do not forget to "read" the closing quote
    index++;
  }
  
  else { 
    // Skip over all alpha-numeric characters  
    while (index < input_size && isalnum(input[index])) {
      index++;
    }
  }

#ifdef DEBUG
  std::cout << "DEBUG: read_token() will return a new string" << std::endl;
  std::cout << "DEBUG: read_token(): input = " << input <<", start = " << start << ", index = " << index << std::endl; 
#endif
  
  
  std::string* s = new std::string(input + start, index - start);
  skip_white_space();
  return s;
}


/**
 * Check that the next part of the input is a given string.
 */
void
bool_parser::consume(std::string s) {
  std::string* r = read_next(s.size());
  
  if ( r->compare(s) != 0) {
    std::cerr << "Error: expect to read the string \"" << s 
	      << "\" but instead read the string \"" << r << "\"" << std::endl;
  }
  delete r;
}


/**
 * Read a number of characters from the input.
 */
std::string*
bool_parser::read_next(int n) {
  
  if (index + n > input_size) {
    fprintf(stderr, "Expected to read %d characters from %s\n", n, input + n);
    fprintf(stderr, "but there is not enough characters left in the input\n");
    exit(1);
  }
  
  std::string* s = new std::string(input + index, n);
  index += n;
  skip_white_space();
  return s;
}



/**
 * Parse a string as a boolean_operator
 */
bool_operator*
bool_parser::get_operator(std::string s) {

#ifdef DEBUG
  std::cout << "DEBUG: get_operator() called with argument ->" << s << "<-" << std::endl;
#endif
  
  bool_operator* ret_val;
  
  if (s.compare(std::string("&&")) ==0) {
    ret_val = new op_and();
  }

  else if (s.compare(std::string("||")) ==0) {
    ret_val = new op_or();
  }
  
  else if (s.compare(std::string("!")) ==0) {
    ret_val = new op_not();
  }
  
  else {
    std::cerr << "Error in bool_parser::get_operator()" << std::endl;
    std::cerr << "Unable to parse a Boolean operator from string" << std::endl;
    std::cerr << "->" << s << "<-" << std::endl;
    exit(1);
  }
  
#ifdef DEBUG
  std::cout << "DEBUG: get_operator() about to return ->" << ret_val->to_string() << "<-" << std::endl;
#endif
  
  return ret_val;
}
