#include <iostream>
#include <string>
#include <cassert>
#include "bool_parser.h"

int num_tests = 0;
int num_fail = 0;



// Forward
void test_parsing();
void test_eval();
void parsetest();
void evaltest();


int main(char** args, int argc) {
  
  test_parsing();
  test_eval();

  std::cout << "Failed tests: " << num_fail << " out of " << num_tests << std::endl;
  
  exit(0);  
}


/**
 * Tests the evaluation functionality of the boolean formulas and the
 * operators.
 */
void evaltest(std::string input, sbmap_t* ta, bool exp_output) {
#ifdef TEST_PRINT
  std::cout << "Evaluating " << input << std::endl;
#endif
  
  num_tests++;

  bool_parser parser;
  bool_formula* bf = parser.parse(input);
  bool output = bf->eval(ta);

#ifdef TEST_PRINT
  std::cout << "Result: " << output << std::endl;
#endif

  if ( output != exp_output) {
    std::cerr << "Evaluation failed: " << std::endl;
    std::cerr << "Input string " << input << std::endl;
    std::cerr << "Expected output: " << exp_output << std::endl;
    std::cerr << "Actual output:   " << output << std::endl;
    num_fail ++;
  }

  delete bf;
  assert (bool_formula::unfreed_formulas() == 0);
  assert (bool_operator::unfreed_operators() == 0);
}



/**
 * Tests parsing/to_string() of the boolean formulas and the
 * operators.
 */
void parsetest(std::string input, std::string exp_output) {

#ifdef TEST_PRINT
  std::cout << "Testing " << input << std::endl;
#endif
  num_tests++;
  
  bool_parser parser;
  bool_formula* bf = parser.parse(input);
  std::string output = bf->to_string();

#ifdef TEST_PRINT
  std::cout << "Result: " << output << std::endl;
#endif

  
  if (output.compare(exp_output) != 0) {
    std::cerr << "to_string() failed: " << std::endl;
    std::cerr << "Input string " << input << std::endl;
    std::cerr << "Expected output: " << exp_output << std::endl;
    std::cerr << "Actual output:   " << output << std::endl;
    num_fail ++;
  }

  delete bf;
  assert (bool_formula::unfreed_formulas() == 0);
  assert (bool_operator::unfreed_operators() == 0);
}


void test_eval() {
  sbmap_t* ta = new sbmap_t;
  (*ta)["p"] = true;
  (*ta)["q"] = false;
  (*ta)["r"] = true;
  (*ta)["s"] = false;

  
  evaltest(std::string("p && q"),
	   ta, false);
  
  evaltest(std::string("   (p)&&q  "),
	   ta, false);

  evaltest(std::string("!p && q"),
	   ta, false);

  evaltest(std::string("p && !q"),
	   ta, true);

  evaltest(std::string("!p || q"),
	   ta, false);
  
  evaltest(std::string("p || q"),
	   ta, true);

  evaltest(std::string("((!(p)) || !!(!q))"),
	   ta, true);

  evaltest(std::string("p && 1"),
	   ta, true);

  evaltest(std::string("p || 0"),
	   ta, true);

  evaltest(std::string("q || 0"),
	   ta, false);

  evaltest(std::string("(p && q) || r"),
	   ta, true);

  evaltest(std::string("p && ((q) || r)"),
	   ta, true);

  evaltest(std::string("!p || ((!q) && r)"),
	   ta, true);
  
  evaltest(std::string("0 || !p || ((!q) && r)"),
	   ta, true);

  evaltest(std::string("1 && (0 || !p || ((!q) && r))"),
	   ta, true);

  evaltest(std::string("!(1 && (0 || !p || ((!q) && r)))"),
	   ta, false);

  evaltest(std::string("1 && 0"),
	   ta, false);

  evaltest(std::string("1 || 0"),
	   ta, true);

  evaltest(std::string("0"),
	   ta, false);

  evaltest(std::string("1"),
	   ta, true);

  delete ta;

  ta = new std::map<std::string, bool>();
  (*ta)["p"] = true;
  (*ta)["q"] = false;
  (*ta)["r"] = true;
  (*ta)["s"] = false;

  (*ta)["\"p\""] = false;
  (*ta)["\"q\""] = true;
  (*ta)["\"r\""] = false;

  evaltest(std::string("p && q"),
	   ta, false);
  
  evaltest(std::string("   (p)&&q  "),
	   ta, false);

  evaltest(std::string("!p && q"),
	   ta, false);

  evaltest(std::string("p && !q"),
	   ta, true);

  evaltest(std::string("!p || q"),
	   ta, false);
  
  evaltest(std::string("p || q"),
	   ta, true);

  evaltest(std::string("((!(p)) || !!(!q))"),
	   ta, true);

  evaltest(std::string("p && 1"),
	   ta, true);

  evaltest(std::string("p || 0"),
	   ta, true);

  evaltest(std::string("q || 0"),
	   ta, false);

  evaltest(std::string("(p && q) || r"),
	   ta, true);

  evaltest(std::string("p && ((q) || r)"),
	   ta, true);

  evaltest(std::string("!p || ((!q) && r)"),
	   ta, true);
  
  evaltest(std::string("0 || !p || ((!q) && r)"),
	   ta, true);

  evaltest(std::string("1 && (0 || !p || ((!q) && r))"),
	   ta, true);

  evaltest(std::string("!(1 && (0 || !p || ((!q) && r)))"),
	   ta, false);

//   (*ta)["p"] = true;
//   (*ta)["q"] = false;
//   (*ta)["r"] = true;
//   (*ta)["s"] = false;

//   (*ta)["\"p\""] = false;
//   (*ta)["\"q\""] = true;
//   (*ta)["\"r\""] = false;

  evaltest(std::string("!p || ((!\"q\") && r)"),
	   ta, false);

  evaltest(std::string("!\"p\" || ((!\"q\") && r)"),
	   ta, true);

  evaltest(std::string("!\"p\" && ((!q) && r)"),
	   ta, true);

  evaltest(std::string("!\"p\" && ((!q) && !\"r\")"),
	   ta, true);

  evaltest(std::string("(!\"p\" && ((!q) && !\"r\")) && (!\"p\" && ((!q) && r))"),
	   ta, true);

  evaltest(std::string("(!p || ((!\"q\") && r)) || (!\"p\" && ((!q) && !\"r\"))"),
	   ta, true);

  evaltest(std::string("1 || blah"),
	   ta, true);

  evaltest(std::string("0 && blah"),
	   ta, false);

  delete ta;

}


void test_parsing() {
  
  parsetest(std::string("p && q"),
	    std::string("a:p and a:q"));

  parsetest(std::string("p&& q"),
	    std::string("a:p and a:q"));

  parsetest(std::string("p&&q"),
	    std::string("a:p and a:q"));
  
  parsetest(std::string("   p&& q  "),
	    std::string("a:p and a:q"));

  parsetest(std::string("   (p)&&q  "),
	    std::string("a:p and a:q"));

  parsetest(std::string("   (p)&&(q)  "),
	    std::string("a:p and a:q"));

  parsetest(std::string("   ((p)&&(q))  "),
	    std::string("a:p and a:q"));

  parsetest(std::string("   p&&q  "),
	    std::string("a:p and a:q"));

  parsetest(std::string("!p && q"),
	    std::string(" not a:p and a:q"));

  parsetest(std::string("(!p) && q"),
	    std::string(" not a:p and a:q"));

  parsetest(std::string("(!(p)) && q"),
	    std::string(" not a:p and a:q"));

  parsetest(std::string("(!(p)) && (q)"),
	    std::string(" not a:p and a:q"));

  parsetest(std::string("((!(p)) && (q))"),
	    std::string(" not a:p and a:q"));

  parsetest(std::string(" !p && q"),
	    std::string(" not a:p and a:q"));

  parsetest(std::string(" ! p&& q"),
	    std::string(" not a:p and a:q"));

  parsetest(std::string("!  p&&q"),
	    std::string(" not a:p and a:q"));

  parsetest(std::string("!p&&q"),
	    std::string(" not a:p and a:q"));

  parsetest(std::string("!p && \"q\""),
	    std::string(" not a:p and a:\"q\""));

  parsetest(std::string("!p&&\"q\""),
	    std::string(" not a:p and a:\"q\""));

  parsetest(std::string("!p&&\" q\""),
	    std::string(" not a:p and a:\" q\""));

  parsetest(std::string("!p&&\"q \""),
	    std::string(" not a:p and a:\"q \""));

  parsetest(std::string("!p&&  \"q\""),
	    std::string(" not a:p and a:\"q\""));

  parsetest(std::string("!p || \"q\""),
	    std::string(" not a:p or a:\"q\""));

  parsetest(std::string("!p||\"q\""),
	    std::string(" not a:p or a:\"q\""));

  parsetest(std::string("! p||\"q\""),
	    std::string(" not a:p or a:\"q\""));

  parsetest(std::string("!p||\" q\""),
	    std::string(" not a:p or a:\" q\""));

  parsetest(std::string("!p || !\"q\""),
	    std::string(" not a:p or  not a:\"q\""));

  parsetest(std::string("1 || 0"),
	    std::string(" true  or  false "));

  parsetest(std::string("\"1\" || 0"),
	    std::string("a:\"1\" or  false "));

  parsetest(std::string("1||0"),
	    std::string(" true  or  false "));

  parsetest(std::string("1 || \"0\""),
	    std::string(" true  or a:\"0\""));

  parsetest(std::string("p || !(q || r)"),
	    std::string("a:p or  not  [ a:q or a:r ] "));

  parsetest(std::string("p || !((((q || r))))"),
	    std::string("a:p or  not  [ a:q or a:r ] "));

  parsetest(std::string("!(p || !(q || r))"),
	    std::string(" not  [ a:p or  not  [ a:q or a:r ]  ] "));

  parsetest(std::string(" ! ( p   ||   ! ( q  ||  r ) )  &&  t "),
	    std::string(" not  [ a:p or  not  [ a:q or a:r ]  ]  and a:t"));

  parsetest(std::string(" \"! ( p   ||   ! ( q  ||  r ) )\"  &&  t "),
	    std::string("a:\"! ( p   ||   ! ( q  ||  r ) )\" and a:t"));

  parsetest(std::string("1"),
	    std::string(" true "));

  parsetest(std::string("0"),
	    std::string(" false "));

  parsetest(std::string(" 1  "),
	    std::string(" true "));

  parsetest(std::string(" 0"),
	    std::string(" false "));

  parsetest(std::string("p"),
	    std::string("a:p"));

  parsetest(std::string("p1"),
	    std::string("a:p1"));

  parsetest(std::string("(((p1)))"),
	    std::string("a:p1"));

  parsetest(std::string("p1 && (p2 || p3)"),
	    std::string("a:p1 and  [ a:p2 or a:p3 ] "));

  parsetest(std::string("!(\"p&&q&&r\" && !(s && a && !(p || q))) || q"),
	    std::string(" not  [ a:\"p&&q&&r\" and  not  [ a:s and a:a and  not  [ a:p or a:q ]  ]  ]  or a:q"));

 
}
