#include "global_params.h"


// Constructor
global_params::global_params(int argc, char** argv) {

  if (argc == 2 && (strcasecmp("-help", argv[1]) == 0)) {
    help(argv[0]);
    exit(0);
  }
  
  // Initialize the two maps
  initialize_encoding_maps();
  
  
  // Set the default values
  conf_file = std::string("mm_config.txt");
  verbosity = 3;
  mon_name = "monitor";
  print_spin = false;
  output_file = "monitor.cc";
  header_output_file = "";
  encoding = FRONT_DET_SWITCH;
  reduction_type = NO_REDUCTION;
  scratch = std::string("/tmp");
  systemc_home = std::string("/projects/vardi/usr/local/systemc-2.2.0_gcc3.3.3_nodebug/");
  brics_root = std::string("/projects/vardi/src/automaton-1.11/src/");
  min_algo = std::string("Hopcroft");
  minimize = true;
  alphabetization_method = ASSIGNMENTS;

  
  // Length of configuration file
  unsigned int conf_length;

  // Parse the command line parameters to get the config file name (if
  // different from default)
  for (int i = 1; i < argc; i = i + 2) {
    if ( strcasecmp(argv[i], "-conf") == 0 ) {
      set_conf_file( argv[i+1] );
    }
  }

  if (conf_file.size() > 0) {
    // Determine the length of the configuration file
    std::ifstream is( conf_file.c_str(), std::ios::binary );
    if (! is) {
      std::cerr << "ERROR: Configuration file \'" << conf_file
		<< "\' could not be opened for reading!" << std::endl;
      exit (1);
    }
    
    is.seekg (0, std::ios::end);
    conf_length = is.tellg();
    is.close();
    
    // Now we are ready to parse everything from the config file
    std::cout << "Parsing the configuration file " << std::endl;
    parse_config_file(conf_length);
    std::cout << "Done" << std::endl;
  }


  // Parse the command line parameters (take precedence)
  std::cout << "Parsing the command line parameters " << std::endl;
  parse_command_line(argc, argv);
  std::cout << "Done" << std::endl;   

  std::cout << "Starting sanity checks " << std::endl;
  sanity_checks();
  std::cout << "Done" << std::endl;
}



/**
 * Check the values of the parsed parameters to ensure consistency
 */
void
global_params::sanity_checks() {
  if (! minimize) {
    if (encoding == FRONT_DET_SWITCH ||
	encoding == FRONT_DET_IFELSE ||
	encoding == BACK_ASS_ALPHA)
      {
	std::cerr << "Incompatible parameters: encoding " << encoding2char[encoding] << " only works with \ndeterministic automata. Please turn on minimization." << std::endl;
	exit(1);
      }
  }

  if (alphabetization_method == SYMBOLIC) {
    if (encoding == FRONT_DET_IFELSE ||
	encoding == FRONT_DET_SWITCH ||
	encoding == BACK_ASS_ALPHA)
      {
	std::cerr << "Incompatible parameters: encoding " << encoding2char[encoding] << " only works with \nassignment-based alphabetization." << std::endl;
	exit(1);
      }
  }

  if (strlen(get_output_file()) == 0) {
    std::cerr << "You must provide the name of the output file!" << std::endl;
    exit (1);
  }

  if (header_output_file.size() == 0) {
    std::cerr << "Warning: No header output file defined! " << std::endl;
    std::string::size_type dot_location = output_file.find_last_of(".");

    if (dot_location == std::string::npos) {
      // There is no dot in the file name??
      header_output_file = output_file;
      header_output_file.append(".h"); 
    }
    else {
      header_output_file = output_file.substr(0, dot_location);
      header_output_file.append(".h");
    }

    std::cerr << "Deriving a header file name from the name of the output file: " << header_output_file << std::endl;
      

  }

  
}

void
global_params::parse_command_line(int argc, char** argv) {
  for (int i = 1; i < argc; i = i + 2) {
    if ( strcasecmp(argv[i], "-verbosity") == 0 ) {
      set_verbosity( atoi(argv[i+1]) );
    }

    if ( strcasecmp(argv[i], "-encoding") == 0 ) {
      set_encoding( (argv[i+1]) );
    }

    if ( strcasecmp(argv[i], "-alphabetization") == 0 ) {
      set_alphabetization( (argv[i+1]) );
    }

    if ( strcasecmp(argv[i], "-print_spin") == 0 ) {
      set_print_spin( atoi(argv[i+1]) );
    }

    if ( strcasecmp(argv[i], "-alpha_reduction") == 0 ) {
      set_alpha_reduction( argv[i+1] );
    }

    if ( strcasecmp(argv[i], "-scratch") == 0 ) {
      scratch = std::string(argv[i+1]);
    }

    if ( strcasecmp(argv[i], "-systemc_home") == 0 ) {
      systemc_home = std::string(argv[i+1]);
    }

    if ( strcasecmp(argv[i], "-min_algo") == 0 ) {
      min_algo = std::string(argv[i+1]);
    }

    if ( strcasecmp(argv[i], "-brics_root") == 0 ) {
      brics_root = std::string(argv[i+1]);
    }

    if ( strcasecmp(argv[i], "-output_file") == 0 ) {
      set_output_file(argv[i+1]);
    }

    if ( strcasecmp(argv[i], "-header_output_file") == 0 ) {
      set_header_output_file(argv[i+1]);
    }

    if ( strcasecmp(argv[i], "-minimize") == 0 ) {
      minimize = ( atoi(argv[i+1]) );
    }
  }
}

void
global_params::set_alpha_reduction(char* ar) {
  if (strcasecmp(ar, "partial") == 0) {
    reduction_type = PARTIAL_REDUCTION;
  }

  else if (strcasecmp(ar, "full") == 0) {
    reduction_type = FULL_REDUCTION;
  }

  else if (strcasecmp(ar, "none") == 0) {
    reduction_type = NO_REDUCTION;
  }
  
  else {
    printf("Unknown alpha-reduction \"%s\". Expected one of {partial, full, none}.\n", ar);
    exit(1);
  }
}


void
global_params::set_alpha_reduction(std::string ar) {
  char* temp = new char[ar.size() + 1];
  strcpy(temp, ar.c_str());
  set_alpha_reduction(temp);
  delete[] temp;
}


void
global_params::set_encoding(const char* enc) {

  std::map<const char*, encoding_t>::const_iterator it;

  it = char2encoding.find(enc);
  if (it == char2encoding.end()) {
    std::cout << "Unknown encoding " << "\"" << enc
	      << "\". Expected one of " << std::endl << "{";
    for (std::map<encoding_t, const char*>::const_iterator it = encoding2char.begin();
	 it != encoding2char.end();
	 it++)
      {
	if (it != encoding2char.begin()) {
	  std::cout << ", ";
	}
	std::cout << it->second;
      }
    std::cout << "}" << std::endl;
    exit(1); 
  }
  
  encoding = char2encoding[enc];
}



void
global_params::set_encoding(std::string enc) {
  set_encoding(enc.c_str());
}


void
global_params::set_alphabetization(const char* enc) {

  if (strcasecmp(enc, "assignments") == 0) {
    alphabetization_method = ASSIGNMENTS;
  }
  else if (strcasecmp(enc, "symbolic") == 0) {
    alphabetization_method = SYMBOLIC;
  }
  else {
    std::cerr << "Unexpected argument \"" << enc << "\" to the alphabetization selection. Expected one of " << std::endl;
    std::cerr << "{assignments, symbolic}" << std::endl;
    exit(1);
  }
}



void
global_params::set_alphabetization(std::string alpha) {

  set_alphabetization(  alpha.c_str() );
}



void
global_params::to_stream(std::ostream &os) {
  os << "Values of parameters:" << std::endl;
  os << "\t configuration file = " << conf_file << std::endl;
  os << "\t verbosity = " << verbosity << std::endl;
  os << "\t monitor_name = " << mon_name << std::endl;
  os << "\t output file = " << output_file << std::endl;
  os << "\t header output file = " << header_output_file << std::endl;
  os << "\t SPIN printing = " << (print_spin ? 1 : 0) << std::endl;
  printf("\t Encoding: %s\n", encoding2char[encoding]);
  printf("\t Alpha-reduction: %s\n", 
	 (reduction_type == PARTIAL_REDUCTION)? "partial" : 
	 (reduction_type == FULL_REDUCTION)? "full" : "none");
  os << "\t scratch = " << scratch << std::endl;
  os << "\t SystemC home = " << systemc_home << std::endl;
  os << "\t Minimize: " << minimize << std::endl;
  printf("\t Alphabetization: %s\n", 
	 (alphabetization_method == ASSIGNMENTS)? "assignments" : 
	 (alphabetization_method == SYMBOLIC)? "symbolic" : "UNKNOWN");
  os << "\t Minimization algorithm = " << min_algo << std::endl;
  os << "\t Brics' Automaton root = " << brics_root << std::endl;
  os << "\t User-defined locations: " << std::endl;
  func_locs2stream(os);

  os << "\t User-defined values: " << std::endl;
  user_vals_to_stream(os);
  
  os << "\t observer objects:  " << std::endl;
  ssmap_to_stream(all_objects, os);

  os << "\t user-defined objects: " << std::endl;
  for (sset_t::const_iterator it = user_objects.begin();
       it != user_objects.end();
       it++) {
    std::cout << "\t\t" << *it << std::endl;
  }

  os << "\t Individual monitor params:  " << std::endl;
  mpset_to_stream(all_monitors, os);
}


/**
 * Formats and prints the includefiles as they should appear in the
 * generated output file
 */
void 
global_params::includefiles_to_stream(std::ostream &os) {
  for (sset_t::iterator it = includefiles.begin();
       it != includefiles.end();
       ++it ) 
    {
      if (it->find('<') != std::string::npos) {

	// Do not quote includes that contain "<>"
	assert(it->find('>') != std::string::npos);
	os << "#include " << *it << std::endl;

      }
      else if (it->find('"') != std::string::npos) {

	//... or contain quotes already
	os << "#include " << *it << std::endl;

      }
      else {
	os << "#include \"" << *it << "\"" << std::endl;

      }
    }
}



/**
 * Parse the configuration file for parameters
 */
void 
global_params::parse_config_file(unsigned int conf_length) {
  
  monitor_params* mp = 0;
  std::ifstream config_f(conf_file.c_str(), std::ios::in);
  
  std::string p, v;
  while (config_f >> p) {

#ifdef DEBUG_GLOBAL_PARAMS
    std::cout << "Read \"" << p << "\" from the configuration file." << std::endl;
#endif

    if (p == "verbosity") {
      if (config_f >> v) {
	set_verbosity( atoi(v.c_str()) );
      }
      
      // Ignore the rest of the line
      config_f.ignore(conf_length, '\n');
    }

    else if (p == "mon_name") {
      if (config_f >> v) {
	set_mon_name(v);
      }

      // Ignore the rest of the line
      config_f.ignore(conf_length, '\n'); 
    }

    else if (p == "encoding") {
      if (config_f >> v) {
	set_encoding(v.c_str());
      }
      // Ignore the rest of the line
      config_f.ignore(conf_length, '\n'); 
    }

    else if (p == "alphabetization") {

      if (config_f >> v) {
	set_alphabetization(v.c_str());
      }
      // Ignore the rest of the line
      config_f.ignore(conf_length, '\n'); 
    }
    
    else if (p == "systemc_home") {
      if (config_f >> v) {
	systemc_home = v;
      }
      
      // Ignore the rest of the line
      config_f.ignore(conf_length, '\n'); 
    }

    else if (p == "brics_root") {
      if (config_f >> v) {
	brics_root = v;
      }
      
      // Ignore the rest of the line
      config_f.ignore(conf_length, '\n'); 
    }

    else if (p == "min_algo") {
      if (config_f >> v) {
	min_algo = v;
      }
      
      // Ignore the rest of the line
      config_f.ignore(conf_length, '\n'); 
    }
    
    else if (p == "alpha_reduction") {
      if (config_f >> v) {
	set_alpha_reduction(v.c_str());
      }

      // Ignore the rest of the line
      config_f.ignore(conf_length, '\n'); 
    }

    else if (p == "scratch") {
      if (config_f >> v) {
	scratch = v;
      }

      // Ignore the rest of the line
      config_f.ignore(conf_length, '\n'); 
    }

    else if (p == "print_spin") {
      if (config_f >> v) {
	set_print_spin( atoi(v.c_str()) );
      }
      
      // Ignore the rest of the line
      config_f.ignore(conf_length, '\n');
    }

    else if (p == "minimize") {
      if (config_f >> v) {
	minimize = atoi(v.c_str());
      }
      
      // Ignore the rest of the line
      config_f.ignore(conf_length, '\n');
    }

    
    else if (p == "usertype") {
      std::string first;
      config_f >> first;
      user_objects.insert(first);

      // Ignore the rest of the line
      config_f.ignore(conf_length, '\n');
    }
    

    else if (p == "output_file") {
      getline(config_f, v);

      // only continue if characters were extracted
      if (config_f.fail()) {
	std::cerr << "ERROR: Unable to extract the output_file " 
		  << "from configuration file" << conf_file << std::endl;
      }
      else {
	set_output_file(v);
      }
      // We already consumed the whole line
    }


    else if (p == "header_output_file") {
      getline(config_f, v);

      // only continue if characters were extracted
      if (config_f.fail()) {
	std::cerr << "ERROR: Unable to extract the header_output_file " 
		  << "from configuration file" << conf_file << std::endl;
      }
      else {
	set_header_output_file(v);
      }
      // We already consumed the whole line
    }
    
    else if (p == "location") {
      getline(config_f, v);

      // only continue if characters were extracted
      if (config_f.fail()) {
	std::cerr << "ERROR: Empty location definition in configuration file " 
	     << conf_file << std::endl;
      }
      else {
	add_location(v);
      }

      // We already consumed the whole line
    }


    else if (p == "value") {
      getline(config_f, v);

      // only continue if characters were extracted
      if (config_f.fail()) {
	std::cerr << "ERROR: Empty variable definition in configuration file " 
		  << conf_file << std::endl;
      }
      else {
	add_value(v);
      }
      // We already consumed the whole line
    }

    else if (p == "formula") {
      mp = new monitor_params();
      add_mp(mp);
      getline(config_f, v);

      // only continue if characters were extracted
      if (config_f.fail()) {
	std::cerr << "ERROR: Unable to extract formula from configuration file " 
	     << conf_file << std::endl;
      }
      else {
	mp->set_formula(v);
      }

      // We already consumed the whole line
    }

    else if (p == "type") {
      std::string first, second;
      config_f >> first;
      config_f >> second;
#ifdef DEBUG_GLOBAL_PARAMS
      std::cout << "  first = \"" << first << "\", and second=\"" << second << "\"." << std::endl;
#endif
      mp->add_vartype(second, first);
      add_vartype(second, first);

#ifdef DEBUG_GLOBAL_PARAMS
      std::cout << "  Moving on." << std::endl;
#endif
      

      // Ignore the rest of the line
      config_f.ignore(conf_length, '\n');
    }




    else if (p == "include") {
      std::string first;
      config_f >> first;
      add_includefile(first);

      // Ignore the rest of the line
      config_f.ignore(conf_length, '\n');
    }

    else if (p[0] == '#') {
      // skip over comments

      // Ignore the rest of the line
      config_f.ignore(conf_length, '\n');
    }

    else { 
      std::cout << "Unknown parameter: " << p << " in the config file." << std::endl;
      // Ignore the rest of the line
      config_f.ignore(conf_length, '\n');
    }
  }

  config_f.close();
}


/**
 * Print a helpful message and exit
 */
void
global_params::help(char* prog_name) {
  std::cout << "Usage: " << prog_name << " [ PARAMETERS ]" << std::endl;
  std::cout << "-conf        path to configuration file (./mm_config)]" << std::endl;
  std::cout << "             You can set all options (except -conf) using the" << std::endl;
  std::cout << "             configuration file or the command-line switches below." << std::endl;
  std::cout << "-verbosity   verbosity of debugging output (3)]" << std::endl;  
  std::cout << "-encoding    one of {FRONT_NONDET, BACK_NONDET, FRONT_DET_IFELSE," << std::endl;
  std::cout << "             FRONT_DET_SWITCH, BACK_ASS_ALPHA}. (FRONT_DET_SWITCH)" << std::endl;
  std::cout << "-minimize    one of {0, 1}. (1)" << std::endl;
  std::cout << "-print_spin  one of {0, 1}. (0)" << std::endl;
  std::cout << "-scratch     path to a directory for temporary files (/tmp)" << std::endl;
  std::cout << "-brics_root  path to the root of BRICS Automaton tool code" << std::endl;
  std::cout << "-output_file file name (with path) of the generated monitor file" << std::endl;
  std::cout << "-header_output_file file name (with path) of the header file" << std::endl;
  std::cout << "-alphabetization    one of {SYMBOLIC, ASSIGNMENTS}. (ASSIGNMENTS)" << std::endl;  
  std::cout << "-alpha_reduction    one of {FULL, PARTIAL, NONE}. (NONE)" << std::endl;
  std::cout << "-systemc_home       path to the root of the SystemC installation " << std::endl;
  std::cout << std::endl << std::endl;
}



/**
 * Print a map of strings -> string
 */
void
global_params::ssmap_to_stream(ssmap_t my_map, std::ostream& os) {
  for (ssmap_t::const_iterator it = my_map.begin();
       it != my_map.end();
       it ++) 
    {
      os << "\t\t" << it->first << " of type " << it->second << std::endl;
    }
}


/**
 * Print a set of monitor_params objects
 */
void
global_params::mpset_to_stream(mpset_t my_set, std::ostream& os) {
  for (mpset_t::const_iterator it = my_set.begin();
       it != my_set.end();
       it++) 
    {
      (*it)->to_stream(os);
    }
}




/**
 * Clean up and set the name of the header output file
 */
void
global_params::set_header_output_file(std::string name) {
  
  copy_without_surrounding_whitespace(&name, &header_output_file);
}


/**
 * Clean up and set the name of the output file
 */
void
global_params::set_output_file(std::string name) {
  
  copy_without_surrounding_whitespace(&name, &output_file);
}



void
global_params::initialize_encoding_maps() {
  encoding2char[FRONT_NONDET]     = "front_nondet";
  encoding2char[FRONT_DET_SWITCH] = "front_det_switch";
  encoding2char[FRONT_DET_IFELSE] = "front_det_ifelse";
  encoding2char[BACK_NONDET]      = "back_nondet";
  encoding2char[BACK_ASS_ALPHA]   = "back_ass_alpha";

  for (std::map<encoding_t, const char*>::const_iterator it = encoding2char.begin();
       it != encoding2char.end();
       it++)
    {
      encoding_t enc = it->first;
      const char* val = it->second;
      char2encoding[val] = enc;
    }
}


/**
 * Parse the location definition and store it in the set locdefs
 */
void
global_params::add_location(std::string loc) {
  func_loc_t* func_loc = new func_loc_t();
  
  // Remove leading whitespace
  char const* delims = " \t\f\v\r\n";
  std::string::size_type notwhite = loc.find_first_not_of(delims);
  loc.erase(0, notwhite);

  // Get the user-given name of the location
  std::string::size_type white = loc.find_first_of(delims);
  func_loc->loc_name = std::string(loc.substr(0, white));
  loc.erase(0, white);

  // Remove leading whitespace again
  loc.erase(0, loc.find_first_not_of(delims));

  // Find the func_befter
  std::string::size_type colon_loc = loc.find_last_of(":");
  func_loc->func_identifier = loc.substr(0, colon_loc);

  if (loc.find("entry", colon_loc) != std::string::npos) {
    func_loc->func_befter = ENTRY;
  }
  
  else if (loc.find("exit", colon_loc) != std::string::npos) {
    func_loc->func_befter = EXIT;
  }
  
  else if (loc.find("call", colon_loc) != std::string::npos) {
    func_loc->func_befter = CALL;
  }

  else if (loc.find("return", colon_loc) != std::string::npos) {
    func_loc->func_befter = RETURN;
  }
  else if (loc.find("dummy", colon_loc) != std::string::npos) {
    func_loc->func_befter = DUMMY;
  }

  else {
    std::cerr << "Error parsing the function befter from the string \"" << loc << "\"" << std::endl;
    exit(1);
  }

  func_locs.push_back(func_loc);
}


const char*
global_params::funcbefter2str(func_befter_t fb) {
  if (fb == ENTRY)
    return "entry";
  if (fb == EXIT)
    return "exit";
  if (fb == CALL)
    return "call";
  if (fb == RETURN)
    return "return";

  assert (fb == DUMMY);
  return "dummy";
}

/**
 * Pretty pring of individual func_locs
 */
void
global_params::func_loc_to_stream(func_loc_t* loc, std::ostream& os) {
  os << loc->loc_name << "="
     << loc->func_identifier << ":"
     << funcbefter2str(loc->func_befter);
}


/**
 * Pretty printing of all func_locs
 */
void
global_params::func_locs2stream(std::ostream& os) {
  for (unsigned int i=0; i<func_locs.size(); i++) {
    func_loc_to_stream(func_locs[i], os);
    os << std::endl;
  }
}


/**
 * Parses and stores the user-defined value
 */
void
global_params::add_value(std::string val) {
  std::stringstream ss(val);
  std::string vartype, varname;
  std::stringbuf rest;
  
  ss >> vartype >> varname;
  ss.get(rest);
  std::string rest_str(rest.str());
  copy_without_surrounding_whitespace(& rest_str, & rest_str);

  // Find the parameter index
  std::string::size_type colon_pos = rest_str.find_last_of(":");
  std::string index = rest_str.substr(colon_pos + 1);
  int index_int = str2int( index);

  // The rest is the function description
  std::string description = rest_str.substr(0, colon_pos);
  copy_without_surrounding_whitespace(& description, & description);

  user_val_t* uv = new user_val_t;
  uv->val_type = vartype;
  uv->val_name = varname;
  uv->func_identifier = description;
  uv->param_index = index_int;

  user_vals.push_back(uv);
}


void
global_params::user_vals_to_stream(std::ostream& os) {
  for (unsigned int i = 0; i < user_vals.size(); i++) {
    this->user_val_to_stream(user_vals[i], os);
    os << std::endl;
  }
}



void
global_params::user_val_to_stream(user_val_t* uv, std::ostream& os) {
  os << uv->val_type
     << " " << uv->val_name
     << " = " << uv->func_identifier
     << " : " << uv->param_index;
}
