#ifndef INTERMEDIATE_REP_H
#define INTERMEDIATE_REP_H

#include <string>
#include <set>
#include <map>
#include <assert.h>
#include <iostream>
#include <fstream>
#include "ir_visitors/ir_visitor.h"

// Forward declaration
class transition_t;
class state_t;

// This class is in a different namespace
namespace mm {
  namespace ir_visitors {
    class ir_visitor;
  }
}


typedef std::set< transition_t* > trans_set_t;
typedef std::set< state_t* > states_set_t;


class transition_t {
  
 public:
  // Constructor
  transition_t(std::string g, int d, int s = -1);
  // Destructor
  ~transition_t() {num_transitions --;}

  // pretty print
  std::string to_string();

  // Inline functions below
  void  reset_source(int s) { source = s;}
  void  reset_guard(std::string g) {guard = g;}
  void  set_source(int s) {assert(source == -1); source = s;}
  void  set_dest(int d) {destination = d;}
  int   get_dest() {return destination;}
  int   get_source() {return source;}
  void  get_guard(std::string* g) { g->assign(guard); }
  std::string get_guard() {return guard;}
  
  static int unfreed_transitions() {return num_transitions;}

 protected:
  std::string guard;
  int destination;
  int source;
  
  static int num_transitions;
};







class state_t {

 public:
  // Constructors
  state_t(int id, bool acc);
  state_t(int id, bool init, bool acc);
  state_t(int id, bool init, bool acc, bool accept_all);
  
  // Destructor
  ~state_t();

  void initialize(int id, bool init, bool acc, bool accept_all);
  void add_outgoing_transition(transition_t* trans);
  void add_incoming_transition(transition_t* trans);
  void propagate_outgoing(std::map<int, state_t*> istate);
  void propagate_incoming(std::map<int, state_t*> istate);
  bool is_deterministic();
  void rename_transitions(std::map<int, state_t*> istate);
  // Visitor pattern
  void accept(mm::ir_visitors::ir_visitor* visitor);

  // Inline functions below
  void set_state_id(int new_id) {state_id = new_id;}
  int  get_state_id() {return state_id;}
  bool is_accept_all() { return accept_all; }
  bool is_initial() {return initial;}
  bool is_accepting() {return accepting;}
  int  get_num_incoming() {return incoming.size(); }
  
  static int unfreed_states() {return num_states;}
  
 protected:
  void delete_transitions(trans_set_t& trans_set);

  int state_id; // id assigned by external data structure
  int internal_id; // id assigned by the I.R. (always in [0..num_states-1])
  bool initial;
  bool accepting;
  bool accept_all;
  trans_set_t incoming;
  trans_set_t outgoing;
  static int num_states;
};







class intermediate_rep : public states_set_t {
  
 public:
  // Constructor
  intermediate_rep();
  // Destructor
  ~intermediate_rep();
  
  void parse_automaton(const char* source);
  state_t* get_initial_state();
  void restore_invariants();
  void restore_transition_invariants();
  void rename_states();
  bool is_determ() {return is_deterministic;}
  // Visitor pattern
  void accept(mm::ir_visitors::ir_visitor* visitor);

 protected:
  int parse_int(std::ifstream& in_strm);
  transition_t* parse_transition(std::ifstream& in_strm);
  void parse_state(std::ifstream& in_strm);
  bool has_int(std::ifstream& in_strm);
  void check_determinism();
  
  
  state_t* initial_state;
  bool is_deterministic;
}; // class


#endif
