#include "ir2monitor.h"

namespace mm {
  namespace ir_visitors {

ir2monitor::ir2monitor(global_params* g,
		       monitor_params* m,
		       std::ostream& o,
		       std::ostream& h,
		       bool ua) :
  gp(g),
  mp(m),
  os(o),
  header_os(h),
  use_array(ua)
{
  mp->get_ltl(&ltl_string);

  
  kernel_clocks.push_back("MON_INIT_PHASE_BEGIN");
  kernel_clocks.push_back("MON_INIT_PHASE_END");
  kernel_clocks.push_back("MON_INIT_UPDATE_PHASE_BEGIN"); 
  kernel_clocks.push_back("MON_INIT_UPDATE_PHASE_END"); 
  kernel_clocks.push_back("MON_INIT_DELTA_NOTIFY_PHASE_BEGIN"); 
  kernel_clocks.push_back("MON_INIT_DELTA_NOTIFY_PHASE_END"); 
  kernel_clocks.push_back("MON_DELTA_CYCLE_BEGIN"); 
  kernel_clocks.push_back("MON_DELTA_CYCLE_END"); 
  kernel_clocks.push_back("MON_EVALUATION_PHASE_BEGIN"); 
  kernel_clocks.push_back("MON_EVALUATION_PHASE_END"); 
  kernel_clocks.push_back("MON_UPDATE_PHASE_BEGIN"); 
  kernel_clocks.push_back("MON_UPDATE_PHASE_END"); 
  kernel_clocks.push_back("MON_DELTA_NOTIFY_PHASE_BEGIN"); 
  kernel_clocks.push_back("MON_DELTA_NOTIFY_PHASE_END"); 
  kernel_clocks.push_back("MON_TIMED_NOTIFY_PHASE_BEGIN"); 
  kernel_clocks.push_back("MON_TIMED_NOTIFY_PHASE_END"); 
  kernel_clocks.push_back("MON_METHOD_SUSPEND"); 
  kernel_clocks.push_back("MON_THREAD_SUSPEND");
}



/**
 * Generate and print to class variable @os the header of the monitor,
 * the parameters with which it was generated, and (optionally) the
 * SPIN version of the monitor.
 */
void
ir2monitor::monitor_header() {
  
  // We give each monitor an id in order to distinguish between them.
  std::string id;
  mp->get_id_str(&id);
  this_monitor_name = gp->get_mon_name() + id;

  os << "// Monitor parameters:" <<  std::endl;
  std::string commout;
  mp->to_string( &commout );
  comment_out_string( &commout );
  os << commout;
  os << std::endl << std::endl << std::endl;
}



    

/**
 * Generates the code for the constructor for the monitor.
 */
void
ir2monitor::monitor_constructor(int initial_state_id) {
  
#ifdef DEBUG_IR2MONITORS
  std::cout << __FILE__ << "::" << __func__ << "(initial_state_id="
	    << initial_state_id << ", use_array="
	    << use_array << ")" << std::endl;
#endif

  // First we do the class declaration in the header file
  header_os << "class " << this_monitor_name
	    << " : public sc_core::mon_prototype {" << std::endl << std::endl;
  header_os << "public:" << std::endl << std::endl;
  header_os << "  /* The constructor */" << std::endl;
  header_os << "  " << this_monitor_name << "(sc_core::mon_observer* obs";

  std::stringstream constructor_arguments;
  // The arguments for the constructor
  int i = 0;
  for (ssmap_t::const_iterator it = mp->get_vartypes().begin(); 
       it != mp->get_vartypes().end();
       ++it ) 
    {
      constructor_arguments << ", " << it->second << " obj" << i;
      i++;
    }
  
  header_os << constructor_arguments.str() << ");" << std::endl;

  // Then we do the function definitions in the monitor file
  os << "/**" << std::endl;
  os << " * The constructor" << std::endl;
  os << " */" << std::endl;
  os << this_monitor_name << "::" << this_monitor_name << "(sc_core::mon_observer* obs"
     << constructor_arguments.str() << ") : sc_core::mon_prototype() {" << std::endl;
  

  if (use_array) {
    // Store the current/next state in arrays if the automaton on which
    // this monitor is based is nondeterministic.
    os << "  next_state = new bool[" << num_states << "];" << std::endl;
    os << "  for (int i = 0; i < " << num_states << "; i++) {" << std::endl;
    os << "    next_state[i] = 0;" << std::endl;
    os << "  }" << std::endl << std::endl;
    os << "  next_state[" << initial_state_id << "] = 1;" << std::endl;
    
    os << "  current_state = NULL;" << std::endl;
  }
  else {
    // Store the current/next state in an int variable if the monitor
    // is based on a deterministic automaton.
    os << "  next_state = " << initial_state_id << "; // initial state id "<< std::endl;
    os << "  current_state = -1;" << std::endl;
  }

  
  os << "  observer = obs;" << std::endl;
  os << "  status = MON_UNDETERMINED;" << std::endl;
  os << "  num_steps = 0;" << std::endl;
    
  
  // Initialize the private class members
  i = 0;
  for (ssmap_t::const_iterator it = mp->get_vartypes().begin(); 
       it != mp->get_vartypes().end();
       ++it ) 
    {
      os << "  " << it->first << " = obj" << i << ";" << std::endl;
      i++;
    }

  // Initialize the location primitives
  std::vector<func_loc_t*>* locs = gp->get_func_locs();
  for (unsigned int i = 0; i < locs->size(); i++) {
    os << "  " << locs->at(i)->loc_name << " = false;" << std::endl;
  }

  // Initialize the value callbacks
  std::vector<user_val_t*>* uv = gp->get_user_vals();

  os << std::endl << std::endl
     << "  // Value callbacks (if any)" << std::endl;
  
  
  for (unsigned int i = 0; i < uv->size(); i++) {
    user_val_t* this_uv = uv->at(i);
    os << "  " << this_uv->val_name << " = NULL;" << std::endl;
  }
    
    register_with_observer();
  
  // End of constructor
  os << "} // Constructor" << std::endl << std::endl;
}



/**
 * Generates the code that registers the monitor with the
 * observer. Called from the monitor_constructor() function.
 */
void
ir2monitor::register_with_observer() {

  sset_t clocks_set = mp->get_clocks();
  char* obs_name = "observer";
  
  for (sset_t::iterator it = clocks_set.begin();
       it != clocks_set.end();
       ++it)
    {
      std::string clk = *it;

#ifdef DEBUG_IR2MONITOR
      std::cerr << "The clock is " << clk << std::endl;
#endif
      
      if (strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0) {
	
#ifdef DEBUG_IR2MONITOR
	std::cerr << "Generating callbacks for the default clock" << std::endl;
#endif
	
	for (unsigned int i = 0; i < kernel_clocks.size(); i++) {
	  os << "  " << obs_name << "->register_monitor(this, "
	     << kernel_clocks[i] << ");" << std::endl;
	  
#ifdef DEBUG_IR2MONITOR
	  std::cerr << "Generated a callbacks for i = " << i << ": "<< kernel_clocks[i] << std::endl;
#endif
	  
	}
      }
  
      else {
	// Only register the kernel clocks
	for (unsigned int i = 0; i < kernel_clocks.size(); i++) {
	  if (strcasecmp(clk.c_str(), kernel_clocks[i].c_str()) == 0) {
	    os << "  " << obs_name << "->register_monitor(this, " << clk << ");" << std::endl;
	  }
	}
      }
    } // loop over all clocks in the formula
}


/**
 * Generates the code to setup the beginning of the step()
 * function. Swaps the values of current state and next state, zeroes
 * out the next state, and starts the iteration over the states.
 */
void
ir2monitor::step_header() {
  header_os << std::endl << "  void step();" << std::endl;
  os << "/**" << std::endl;
  os << "* Simulate a step of the monitor." << std::endl;
  os << "*/" << std::endl;
  os << "void" << std::endl;
  os << this_monitor_name << "::step() {" << std::endl;
  os << "  if (status == MON_UNDETERMINED) {" << std::endl;
  os << "    num_steps++;" << std::endl;
  os << "    if (current_state != NULL) {" << std::endl;
  os << "      delete[] current_state;" << std::endl;
  os << "    }" << std::endl;  
  os << "    current_state = next_state;" << std::endl << std::endl;
  os << "    next_state = new bool[" << num_states  << "];" << std::endl;
  os << "    for (int i = 0; i < " << num_states  << "; i++) {" << std::endl;
  os << "      next_state[i] = 0;" << std::endl;
  os << "    }" << std::endl << std::endl;
}

    

/**
 * Generates the callback code for registering each monitor with the
 * observer
 */
void
ir2monitor::observer_callbacks() {
  // Insert the appropriate callback
  os << std::endl << std::endl;
  
  
  //   virtual void callback_event_notified(sc_event* event);

  sset_t clocks_set = mp->get_clocks();
  char* fun_name = "step()";
  header_os << std::endl << "  // Kernel callbacks and user-code callbacks" << std::endl;
  
  for (sset_t::iterator it = clocks_set.begin();
       it != clocks_set.end();
       ++it)
    {
      std::string clk = *it;
      bool is_kernel_clock = (strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0);
      
      // Here we handle the kernel callbacks
      for (unsigned int i = 0; i < kernel_clocks.size(); i++) {
	if (strcasecmp(clk.c_str(), kernel_clocks[i].c_str()) == 0) {
	  is_kernel_clock = true;
	}
      }

      // Either kernel clock or default clock
      if (is_kernel_clock) {
	if ((strcasecmp(clk.c_str(), "MON_INIT_PHASE_BEGIN") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_init_phase_begin();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_init_phase_begin() {" << std::endl;
	    os << "  " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;      
	  }

	if ((strcasecmp(clk.c_str(), "MON_INIT_PHASE_END") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_init_phase_end();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_init_phase_end() {" << std::endl;
	    os << "  " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;
	  }

	if ((strcasecmp(clk.c_str(), "MON_INIT_UPDATE_PHASE_BEGIN") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_init_update_phase_begin();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_init_update_phase_begin() {" << std::endl;
	    os << "  " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;      
	  }

	if ((strcasecmp(clk.c_str(), "MON_INIT_UPDATE_PHASE_END") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_init_update_phase_end();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_init_update_phase_end() {" << std::endl;
	    os << "  " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;   
	  } 

	if ((strcasecmp(clk.c_str(), "MON_INIT_DELTA_NOTIFY_PHASE_BEGIN") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_init_delta_notify_phase_begin();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_init_delta_notify_phase_begin() {" << std::endl;
	    os << "  " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;
	  }


	if ((strcasecmp(clk.c_str(), "MON_INIT_DELTA_NOTIFY_PHASE_END") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_init_delta_notify_phase_end();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_init_delta_notify_phase_end() {" << std::endl;
	    os << "  " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;
	  }


	if ((strcasecmp(clk.c_str(), "MON_DELTA_CYCLE_BEGIN") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_delta_cycle_begin();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_delta_cycle_begin() {" << std::endl;
	    os << "  " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;      
	  }


	if ((strcasecmp(clk.c_str(), "MON_DELTA_CYCLE_END") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_delta_cycle_end();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_delta_cycle_end() {" << std::endl;
	    os << "  " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;      
	  }

	if ((strcasecmp(clk.c_str(), "MON_EVALUATION_PHASE_BEGIN") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_evaluation_phase_begin();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_evaluation_phase_begin() {" << std::endl;
	    os << "  " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;      
	  }


	if ((strcasecmp(clk.c_str(), "MON_EVALUATION_PHASE_END") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_evaluation_phase_end();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_evaluation_phase_end() {" << std::endl;
	    os << "  " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;      
	  }
    

	if ((strcasecmp(clk.c_str(), "MON_UPDATE_PHASE_BEGIN") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_update_phase_begin();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_update_phase_begin() {" << std::endl;
	    os << "  " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;      
	  }


	if ((strcasecmp(clk.c_str(), "MON_UPDATE_PHASE_END") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_update_phase_end();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_update_phase_end() {" << std::endl;
	    os << "  " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;      
	  }

	if ((strcasecmp(clk.c_str(), "MON_DELTA_NOTIFY_PHASE_BEGIN") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_delta_notify_phase_begin();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_delta_notify_phase_begin() {" << std::endl;
	    os << "  " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;      
	  }


	if ((strcasecmp(clk.c_str(), "MON_DELTA_NOTIFY_PHASE_END") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_delta_notify_phase_end();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_delta_notify_phase_end() {" << std::endl;
	    os << "  " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;      
	  }


	if ((strcasecmp(clk.c_str(), "MON_TIMED_NOTIFY_PHASE_BEGIN") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_timed_notify_phase_begin();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_timed_notify_phase_begin() {" << std::endl;
	    os << "  " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;      
	  }


	if ((strcasecmp(clk.c_str(), "MON_TIMED_NOTIFY_PHASE_END") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_timed_notify_phase_end();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_timed_notify_phase_end() {" << std::endl;
	    os << " " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;      
	  }


	if ((strcasecmp(clk.c_str(), "MON_METHOD_SUSPEND") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_method_suspend();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_method_suspend() {" << std::endl;
	    os << "  " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;      
	  }


	if ((strcasecmp(clk.c_str(), "MON_THREAD_SUSPEND") == 0) || 
	    strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0)
	  {
	    header_os << "  virtual void callback_thread_suspend();" << std::endl;
	    os << "void" << std::endl;
	    os << this_monitor_name << "::callback_thread_suspend() {" << std::endl;
	    os << "  " << fun_name << ";" << std::endl;
	    os << "}" << std::endl << std::endl;      
	  }
      }

      // It is not a kernel clock, nor the DEFAULT_CLOCK
      else {
	// User-defined clock (e.g., function entry)
	header_os << "  void callback_userloc_" << clk << "();" << std::endl;
	os << "void" << std::endl;
	os << this_monitor_name << "::callback_userloc_" << clk << "() {" << std::endl;
	os << std::endl << "  if (DEBUG_USERLOCS) {" << std::endl;
	os << "    std::cout << \"" << this_monitor_name << "::" << clk << " callback"
	   << "\" << std::endl << std::endl;" << std::endl;
	os << "  }" << std::endl << std::endl;
	os << "  // Set the value of the primitive" << std::endl;
	os << "  " << clk << " = true;" << std::endl << std::endl;
	os << "  " << fun_name << ";" << std::endl << std::endl;
	os << "  // Reset the value of the primitive" << std::endl;
	os << "  " << clk << " = false;" << std::endl;
	os << "}" << std::endl << std::endl << std::endl;      
      }
    }
}


/**
 * Generates the callbacks with values of variables in the user code.
 */
void
ir2monitor::generate_value_callbacks() {
  std::vector<user_val_t*>* uv = gp->get_user_vals();

  header_os << std::endl << std::endl
	    << "  // Value callbacks (if any)" << std::endl;
  
  for (unsigned int i = 0; i < uv->size(); i++) {
    user_val_t* this_uv = uv->at(i);

    std::stringstream ss;
    ss << "value_callback_" << this_uv->val_name
       << "(" << this_uv->val_type;

    std::string actual_param_name;
    // Avoid variable name clashes
    if (this_uv->val_name.compare("val") == 0) {
      actual_param_name = "value";
    }
    else {
      actual_param_name = "val";
    }

    ss << " " << actual_param_name << ")";
    
    
    header_os << "  void " << ss.str() << ";" << std::endl;

    os << "void" << std::endl;
    os << this_monitor_name << "::" << ss.str() << " {" << std::endl;
    os << "  if (DEBUG_USERVALS) {" << std::endl;
    os << "    std::cout << \"" << this_monitor_name << "::" << ss.str() << ":\"" << std::endl;
    os << "              << \"The value passed is \" << " << actual_param_name << " << std::endl << std::endl;" << std::endl;
    os << "  }" << std::endl;
    os << "  " << this_uv->val_name << " = "
       << actual_param_name << ";" << std::endl;
    os << "}" << std::endl << std::endl;
  }
}


/**
 * Generates the code for the helper functions that handle the
 * property failure or passing, pretty-printing, and status reporting
 */
void
ir2monitor::helper_functions() {

  std::string formula;
  mp->get_f_quote_escaped(&formula);

  header_os << std::endl << "  // Helper functions" << std::endl;
  header_os << "  void property_failed(); " << std::endl;
  os << "void" << std::endl;
  os << this_monitor_name << "::property_failed() {" << std::endl;
  os << "#ifdef MONITOR_REPORT_FAIL_IMMEDIATELY" << std::endl; 
  os << "  SC_REPORT_WARNING(\"Property failed\", \"Monitor for property " << formula << " has reached an error state\");" << std::endl;
  os << "  std::cout << \"Property failed after \" << num_steps << \" steps\" << std::endl;" << std::endl;
  os << "#endif" << std::endl;
  os << "  status = MON_FAIL;" << std::endl;    
  os << "}" << std::endl << std::endl;    

  header_os << "  void property_satisfied();" << std::endl;
  os << "void" << std::endl;
  os << this_monitor_name << "::property_satisfied() {" << std::endl;
  os << "#ifdef MONITOR_REPORT_PASS_IMMEDIATELY" << std::endl;      
  os << "  std::cout << \"The property \" << to_string() << \"HOLDS\" << std::endl;" << std::endl;
  os << "  std::cout << \"Property cannot fail after \" << num_steps << \" steps\" << std::endl;" << std::endl;
  os << "#endif" << std::endl;
  os << "  status = MON_PASS;" << std::endl;
  os << "}" << std::endl << std::endl;

  header_os << "  const char* to_string() const;" << std::endl;
  os << "const char*" << std::endl;
  os << this_monitor_name << "::to_string() const {" << std::endl;
  os << "  return \"" << formula  << "\";" << std::endl;
  os << "}" << std::endl << std::endl;

  header_os <<"  const mon_status_t get_status() const;" << std::endl;
  os << "const mon_status_t" << std::endl;
  os << this_monitor_name << "::get_status() const {" << std::endl;
  os << "#ifdef PRINT_NUM_STEPS_IN_STATUS_CALL" << std::endl;
  os << "  std::cout << to_string() << \" made \" << num_steps << \" steps.\" << std::endl;" << std::endl;
  os << "#endif" << std::endl;  
  os << "  return status;" << std::endl;
  os << "}" << std::endl << std::endl;

  generate_value_callbacks();
}




/**
 * The class variables used by the monitor.
 */
void
ir2monitor::monitor_class_variables() {
  header_os << std::endl << std::endl << "private: " << std::endl;
  if (use_array) {
    // Use array to store the current/next state when the monitor is
    // based on a nondeterministic automaton.
    header_os << "  bool* current_state;" << std::endl;
    header_os << "  bool* next_state;" << std::endl;
  }
  else {
    // Use integers to store the current/next state when the monitor is
    // based on a deterministic automaton.
    header_os << "  int current_state;" << std::endl;
    header_os << "  int next_state;" << std::endl;
  }
  
  header_os << "  sc_core::mon_observer* observer;" << std::endl;
  header_os << "  mon_status_t status;" << std::endl;
  header_os << "  int num_steps;" << std::endl;

  
  for (ssmap_t::const_iterator it = mp->get_vartypes().begin(); 
       it != mp->get_vartypes().end();
       ++it ) 
    {
      header_os << "  " << it->second << " " << it->first << ";" << std::endl;
    }

  // Declare the location primitives
  std::vector<func_loc_t*>* locs = gp->get_func_locs();
  for (unsigned int i = 0; i < locs->size(); i++) {
    header_os << "  bool " << locs->at(i)->loc_name << ";" << std::endl;
  }

  // Declare the user value primitives
  std::vector<user_val_t*>* uv = gp->get_user_vals();

  for (unsigned int i = 0; i < uv->size(); i++) {
    user_val_t* this_uv = uv->at(i);
    header_os << "  " << this_uv->val_type << " "
	      << this_uv->val_name << ";" << std::endl;
  }
  header_os << "}; // class" << std::endl << std::endl;
}



  } // namespace ir_visitors
} // namespace mm
