/**
 * Monitor master main.
 */

#include <iomanip>
#include <fstream>
#include <iostream>
#include <assert.h>


// Helpers
#include "global_params.h"
#include "monitor_params.h"
#include "types.h"
#include "utils.h"
#include "minimize.h"
#include "conversions.h"
#include "aspect.h"
#include "ir_visitors/monitor_generators.h"
#include "ir_visitors/ir_state_trans_counter.h"



void generate_monitors(global_params* gp,
		       std::ofstream& out_strm,
		       std::ofstream& out_strm_h) {

  std::string capitalized;
  get_capitalized_name(gp->get_header_output_file(), & capitalized);
  out_strm_h << "#ifndef " << capitalized << std::endl;
  out_strm_h << "#define " << capitalized << std::endl << std::endl;

  out_strm_h << "#include <systemc>" << std::endl << std::endl;
  out_strm_h << "#define DEBUG_USERVALS 1" << std::endl;
  out_strm_h << "#define DEBUG_USERLOCS 1" << std::endl;
  out_strm << "// Files to be included at the user's request (if any)" << std::endl;
  gp->includefiles_to_stream(out_strm);
  out_strm << std::endl << std::endl;

  std::string naked_name;
  get_naked_file_name(gp->get_header_output_file(), & naked_name);
  out_strm << "#include \"" << naked_name << "\"" << std::endl << std::endl;
  
  for (sset_t::const_iterator it = gp->user_objects.begin();
       it != gp->user_objects.end();
       it++) {
    out_strm_h << "class " << *it << "; // User object. Forward declaration" << std::endl;
  }

  // we close the #ifndef at the end of main()
  
  
  for(mpset_t::const_iterator it = gp->all_monitors.begin();
      it != gp->all_monitors.end();
      ++it)
    {
      std::cout << "Starting to generate a monitor" << std::endl;
      monitor_params* mp = *it;
      encoding_t enc = gp->get_encoding();
      alpha_reduction_t alpha_reduction = gp->get_alpha_reduction();
      intermediate_rep* ir = new intermediate_rep();

      start_timer();
      
      if (alpha_reduction != NO_REDUCTION) {
	mm::augment::augment_in_place(gp, mp);
      }
      
      display_and_reset_timer("Alphabet reduction");
      
      if (gp->get_minimize()) {
	minimize(mp, gp, ir);
      }
      else {
	mm::conversions::mp2bool_ir(mp, ir);
      }

      display_and_reset_timer("LTL to (possibly minimized) IR time (including SPOT)");
      
#ifdef AUTOMATON_STATS
      int num_states = 0;
      int num_transitions = 0;
      mm::ir_visitors::count_ir_state_trans(ir, &num_states, &num_transitions);
      printf("Number of IR states: %d\n", num_states);
      printf("Number of IR transitions: %d\n", num_transitions);
#endif
	
      switch (enc) {
      case FRONT_NONDET:
	mm::ir_visitors::ir2fr_nondet(ir, gp, mp, out_strm, out_strm_h);
	break;
	
      case FRONT_DET_SWITCH:
	mm::ir_visitors::ir2fr_det_ass_alpha_switch(ir, gp, mp, out_strm, out_strm_h);
	break;
	
      case FRONT_DET_IFELSE:
	mm::ir_visitors::ir2fr_det_ass_alpha_ifelse(ir, gp, mp, out_strm, out_strm_h);
	break;
	
      case BACK_NONDET:
	mm::ir_visitors::ir2back_nondet(ir, gp, mp, out_strm, out_strm_h);
	break;

      case BACK_ASS_ALPHA:
	mm::ir_visitors::ir2back_ass_alpha(ir, gp, mp, out_strm, out_strm_h);
	break;

      default:
	std::cerr << "Unknown monitor encoding" << std::endl;
	exit(1);
      } // switch


      display_and_reset_timer("Monitor generating");

      delete ir;
      assert (state_t::unfreed_states() == 0);
      assert (transition_t::unfreed_transitions() == 0);

      std::cout << "Finished generating this monitor" << std::endl;
    } // for loop over all mp's
} // generate_monitors()


int main(int argc, char** argv) {
#ifdef TIMING
  std::cout << "Clock ticks per second: " << sysconf(_SC_CLK_TCK) << std::endl;
#endif

  global_params* gp = new global_params(argc, argv);
  gp->to_stream(std::cout);

  std::ofstream& out_strm = *open_for_output(gp->get_output_file());
  std::ofstream& out_strm_h = *open_for_output(gp->get_header_output_file());
  
  generate_monitors(gp, out_strm, out_strm_h);

  std::cout << "Done generating the monitor classes!" << std::endl;
  
  // Local observer at the end 
  mm::automaton_tools::generate_local_observer(out_strm, out_strm_h, gp);

  generate_aspect_file(gp);

  out_strm_h << std::endl << "#endif" << std::endl;
  close_file(&out_strm);
  close_file(&out_strm_h);
  return 0;
} // main

