#include "minimize.h"


/**
 * Minimizes the @source automaton and stores the result into the
 * @minimized container.  Uses Brics' Automaton tool to do the
 * minimization.
 */
void minimize(intermediate_rep* alphabetized_source,
	      intermediate_rep* minimized_dest,
	      global_params* gp) {
  
  std::string source_file, minimized_file, scratch;
  gp->get_scratch_location(&scratch);
  get_unique_file_name(&source_file, &scratch, "det+min", "in");
  get_unique_file_name(&minimized_file, &scratch, "det+min", "out");

  std::ofstream& source_stream = *open_for_output(source_file.c_str());

  // Write out the unminimized automaton
  mm::ir_visitors::ir2lbt(alphabetized_source, source_stream);
  close_file(& source_stream);

  // Piece together the call to Brics
  std::string brics_root, min_algo;
  gp->get_brics_root(&brics_root);
  gp->get_min_algo(&min_algo);
  
  std::string cmd_line = "java -cp " + brics_root +
    " -Xms4G -Xmx4G dk/brics/extra/Minimizer " +
    " -in " + source_file +
    " -out " + minimized_file +
    " -algo " + min_algo;
  
  std::cout << __FILE__ << ": " << __func__ << ":" << std::endl
	    << "Minimizing the automaton with  " << std::endl;
  std::cout << cmd_line << std::endl;
  system(cmd_line.c_str());

#ifdef DEBUG_MINIMIZE
  std::cout << __FILE__ << "::" << __func__ << std::endl
	    << "The system command returned successfully" << std::endl;
#endif
  
  minimized_dest->parse_automaton(minimized_file.c_str());

#ifdef DEBUG_MINIMIZE
  std::cout << __FILE__ << "::" << __func__ << std::endl
	    << "The minimized automaton was parsed successfully as an IR." << std::endl;
  mm::ir_visitors::ir2lbt(minimized_dest, std::cout);
#endif
}



void minimize(monitor_params* mp, global_params* gp, intermediate_rep* result) {

  std::string ltl_string;
  mp->get_ltl(&ltl_string);

  // LTL -> SPOT -> IR
  mm::automaton_tools at;
  const spot::tgba_sba_proxy* s = at.ltl2tgba(ltl_string);
  
  if (gp->get_print_spin()) {
    // Print the SPIN version of the automaton, for reference
    std::stringstream spin_stream(std::stringstream::out);
    spot::never_claim_reachable(spin_stream, s);
    
    std::cout << "// The SPIN version of the automaton: " << std::endl << std::endl;
    std::string commented_out = std::string( spin_stream.str() );
    comment_out_string( &commented_out );
    std::cout << commented_out;
    std::cout << std::endl << std::endl << std::endl;
  }
 
  intermediate_rep alphabetized;
  alphabetization_t alpha_method = gp->get_alphabetization();

  if (alpha_method == ASSIGNMENTS) {
    
#ifdef DEBUG_MINIMIZE 
    std::cout << __FILE__ << "::" << __func__ << ": " << std::endl
	      << "Minimizing using ASSIGNMENTS alphabetization: " << std::endl;
#endif
    
    // Create the vector with truth assignments
    v_sbmap_t ta;
    mm::alphabetize::form2assgnmts(&ltl_string, &ta, true);

#ifdef DEBUG_MINIMIZE 
    std::cout << __FILE__ << "::" << __func__ << ": " << std::endl
	      << "Finished creating the vector with truth assignments " << std::endl;
#endif    

    // Alphabetize and minimize the IR
    mm::conversions::sba2assign_ir(s, &alphabetized, &ta);
    minimize(&alphabetized, result, gp);

#ifdef DEBUG_MINIMIZE 
    std::cout << __FILE__ << "::" << __func__ << ": " << std::endl
	      << "Finished alphabetizing and minimizing the IR " << std::endl;
#endif        

    // Convert back from alpha to Boolean transitions
    mm::ir_visitors::trans_relabeler_ta* trans_relab = new mm::ir_visitors::trans_relabeler_ta(&ta);
    result->accept(trans_relab);
    delete trans_relab;
  }
  else if (alpha_method == SYMBOLIC) {
    
#ifdef DEBUG_MINIMIZE 
    std::cout << __FILE__ << "::" << __func__ << ":" << std::endl
	      << "Minimizing using SYMBOLIC alphabetization: " << std::endl;
#endif    
    
    std::map<int, std::string> dict;

    // Alphabetize and minimize the IR
    mm::conversions::sba2symb_ir(s, &alphabetized, &dict);
    minimize(&alphabetized, result, gp);

    // Convert back from alpha to Boolean transitions
    mm::ir_visitors::trans_relabeler_bdd* trans_relab = new mm::ir_visitors::trans_relabeler_bdd(&dict);
    result->accept(trans_relab);
    delete trans_relab;
  }
  else {
    std::cerr << __FILE__ << ": " << __func__ << std::endl
	      << "Unknown alpha_method: " << alpha_method << std::endl;
    exit(1);
  }

#ifdef DEBUG_MINIMIZE 
  std::cout << __FILE__ << "::" << __func__  << ": " << std::endl
	    << "Done minimizing the automaton: " << std::endl;
  std::cout << __FILE__ << "::" << __func__  << ": " << std::endl
	    << "IR automaton with guards properly recovered: " << std::endl;
    mm::ir_visitors::ir2lbt(result, std::cout);
#endif    
  
  
  at.cleanup();
}




				
