#include "monitor_params.h"

int monitor_params::num_monitors = 0;

/**
 * Constructor
 */
monitor_params::monitor_params() {
  id = num_monitors++;
}



void
monitor_params::get_id_str(std::string* container) {
  std::stringstream ss;
  ss << id;
  container->assign(ss.str());
}



/**
 * Cleans up the parsed_formula from which the monitor will be
 * generated. Expected format for the parsed_formula is LTL @ CLOCK ||
 * CLOCK || CLOCK ...
 */
void
monitor_params::set_formula(std::string parsed_formula) {
  
  // Remove leading whitespace
  char const* delims = " \t\f\v\r\n";
  std::string::size_type notwhite = parsed_formula.find_first_not_of(delims);
  parsed_formula.erase(0,notwhite);

  std::string::size_type separator = parsed_formula.find('@');
  
  if (separator == std::string::npos) {
    // No clock in the parsed_formula
    clocks.insert(std::string("DEFAULT_CLOCK"));
    ltl = parsed_formula;
  }
  
  else {
    ltl = parsed_formula.substr(0, separator);
    std::string clock_str = parsed_formula.substr(separator + 1);

    // trim leading garbage
    char const* clock_delims = " ()\t\f\v\n\r|";

    std::string::size_type notwhite = clock_str.find_first_not_of(clock_delims);
    
    if (notwhite == std::string::npos) {
      // Clock string is all white space or garbage
      clocks.insert(std::string("DEFAULT_CLOCK"));
    }

    else {
      clock_str.erase(0, notwhite);
    
      // Trim trailing garbage
      notwhite = clock_str.find_last_not_of(clock_delims);

      // There must be nonwhite characters, otherwise we would have
      // taken the other branch of the conditional
      assert(notwhite != std::string::npos);
      clock_str.erase(notwhite + 1);
      
      unsigned long int pos = 0;
      while (pos != std::string::npos) {
	std::string::size_type white = clock_str.find_first_of(clock_delims, pos);
      
	if (white == std::string::npos) {
	  // There are no white space characters
	  clocks.insert(clock_str.substr(pos));
	  // Reset the flag for the loop.
	  pos = white;
	}
	else {
	  clocks.insert(clock_str.substr(pos, white-pos));
	  pos = clock_str.find_first_not_of(clock_delims, white);
	}
      }
    }
  }
  
  
  // Here we generate a clean version of the full formula, i.e. ltl @
  // clock, for display back to the user
  formula = ltl;
  formula.append(" @ ");
  for (sset_t::const_iterator it = clocks.begin();
       it != clocks.end();
       it++)
    {
      if (it != clocks.begin()) {
	formula.append(" || ");
      }
      formula.append( *it);
    }

  // Finally, generate a quote-escaped version of the formula
  f_quote_escaped = formula;
  unsigned long int j = f_quote_escaped.find("\"");
  while (j < std::string::npos) {
    f_quote_escaped.replace(j, 1, "\\\"");
    j = f_quote_escaped.find("\"", j+2);
  }
} // set formula



void
monitor_params::to_stream(std::ostream &outstr) const {
  
  outstr << "\t LTL forumula: " << ltl << std::endl;
  outstr << "\t clock(s): " << std::endl; 
  
  for (sset_t::iterator it = clocks.begin(); 
       it != clocks.end(); 
       ++it )
    {
      outstr << "\t\t\t" << *it << std::endl;
    }
  outstr << "\t Type info:" << std::endl;
  outstr << "\t\t\t" << "Type\t\tVariable" << std::endl;
  outstr << "\t\t\t" << "------------------------" << std::endl;
  
  for (ssmap_t::const_iterator it = vartypes.begin(); 
       it != vartypes.end(); 
       ++it ) 
    {
      outstr << "\t\t\t" << it->second << "\t\t" << it->first << std::endl;
    }
}

void
monitor_params::to_string(std::string* container) {
  std::stringstream ss( std::stringstream::out );
  to_stream(ss);
  container->assign( ss.str() );
}
