#include "utils.h"


/**
 * Parses an integer from the string @str
 */
int str2int(const std::string& str) {
  std::stringstream ss(str);
  int num;
  if ((ss >> num).fail()) {
    std::cerr << __FILE__ << ": " << __func__ << "():" << std::endl;
    std::cerr << "Failed to parse an integer from the string " << str << std::endl;
    std::cerr << "The error is probably due to the caller of this function." << std::endl;
    exit(1);      
  }
  return num;
}








/**
 * Comments-out the string @input by adding a "//" at the beginning of
 * each line. After this modification, the string can be inserted into
 * any C/C++ generated file.
 */
void comment_out_string(std::string* input) {
  
  // Comment out the first line
  input->replace ( 0, 0, "//" );

  std::string::size_type pos = input->find ("\n", 0); 
  while (  pos != std::string::npos )
    {
      input->replace ( pos, 1, "\n//" );
      pos = input->find ("\n", pos + 3);
    }
}




/**
 * Removes any characters from @flags from @source. Rest is output to @os
 */
void remove_str(std::string source, std::string flags, std::ostream& os) {
  
  unsigned int left = 0;
  while (left < source.size()) {
      
    unsigned long int right = source.find_first_of(flags, left);
    if (right == std::string::npos) {
      // No more instances of the characters in @flags
      os << source.substr(left); // Get the rest of the string
      left = source.size(); // Break out of the while()
    }
    else {
      os << source.substr(left, right-left);
      left = right+1;
    }
  }
}
  

/**
 * Removes all characters mentioned in @flags from @input. The result
 * is returned inside @output.
 */
void remove_str(const std::string* input, const char* flags, std::string* output) {
  std::stringstream ss;
  remove_str(*input, std::string(flags), ss);
  output->assign(ss.str());
}



#include <ctype.h>
/**
 * Converts an LTL formula in prefix normal form into an in-order form.
 */
bool prefix2inorder(std::ifstream& in, std::string* dest) {
  
  std::string space = std::string(" ");
  std::string lparen = std::string("(");
  std::string rparen = std::string(")");
  std::string left;
  std::string right;
  std::string op;
  char ch;
  
  // Skip over the initial white space
  do {
    ch = in.get();
  }while(isspace(ch));

  switch (ch) {
  case 'U':
  case 'u':
  case 'V':
  case 'v':
  case 'i':
  case 'e':
  case '&':
  case '|':
    if (prefix2inorder(in, &left)) {

      if (prefix2inorder(in, &right)) {
	switch (ch) {
	case 'U':
	case 'u':
	case 'V':
	case 'v':
	  op = std::string(1, ch);
	  break;

	case '&':
	case '|':
	  // Repeat the character twice
	  op = std::string(2, ch);
	  break;
	  
	case 'i':
	  op = std::string("->");
	  break;
	  
	case 'e':
	  op = std::string("<->");
	  break;
	  
	}
	
	(*dest) = lparen + left + space + op + space + right + rparen;
	return true;
      }
    }
    return false;

  case '!':
  case 'X':
  case 'G':    
  case 'F':
    if (prefix2inorder(in, &right)) {
      (*dest) = lparen + std::string(1, ch) + space + right + rparen;
#ifdef DEBUG_UTILS
      std::cout << __FILE__ << ": " << __func__ << " Parsed " << *dest << std::endl;
#endif
      return true;
    }
    return false;

  case 't':
    (*dest) = std::string("1");
#ifdef DEBUG_UTILS
	std::cout << __FILE__ << ": " << __func__ << " Parsed " << *dest << std::endl;
#endif
    return true;
    
  case 'f':
    (*dest) = std::string("0");
#ifdef DEBUG_UTILS
	std::cout << __FILE__ << ": " << __func__ << " Parsed " << *dest << std::endl;
#endif
    return true;

  case 'p':    
    {	  
      while (!isspace(ch) && in.good()) {
	(*dest) += std::string(1, ch);
	ch = in.get();
      }
    }
#ifdef DEBUG_UTILS
	std::cout << __FILE__ << ": " << __func__ << " Parsed " << *dest << std::endl;
#endif    
    return true;
    
  case EOF:
    std::cerr << __FILE__ << ": " << __func__ << ": Parse error. Unexpected end of file." << std::endl;;
    return false;
    
  default:
    std::cerr << __FILE__ << ": " << __func__ << ": Parse error. Illegal character \""
	      << ch << "\"" << std::endl;;
    return false;
  
  }
}


/**
 * Removes leading and trailing whitespace from the string @source,
 * and copies the remaining to the string @destination.
 */
void copy_without_surrounding_whitespace(const std::string* source,
					 std::string* destination) {
  // Remove leading whitespace
  std::string source_copy = *source;
  char const* delims = " \t\f\v\r\n";
  std::string::size_type notwhite = source_copy.find_first_not_of(delims);
  source_copy.erase(0,notwhite);

  // Remove trailing whitespace
  notwhite = source_copy.find_last_not_of(delims);
  source_copy.erase(notwhite + 1);
  destination->assign(source_copy);
}


/**
 * Opens a file for output, and checks for errors before returning a
 * pointer to the file handle.
 */
std::ofstream*
open_for_output(const char* file_name) {
  std::ofstream* out_strm = new std::ofstream(file_name, std::ios::out);
  if (! out_strm->good() ) {
    std::cerr << "Unable to open \"" << file_name << "\" for output." << std::endl;
    exit (1);
  }
  return out_strm;
}

/**
 * Closes a file previously opened with open_for_output()
 */
void
close_file(std::ofstream* file_handle) {
  file_handle->close();
  delete file_handle;
}





/**
 * Returns a unique filename in the specified directory. It creates an
 * empty file to prevent multiple calls to this function from
 * returning the same file name.
 */
void get_unique_file_name(std::string* container,
			  std::string* directory,
			  const char* prefix,
			  const char* extension) {
  
  // Make sure that the directory name ends with a "/"
  if (directory->find_last_of('/') != (directory->size() - 1)) {
    directory->append("/");
  }
  
  std::string base_name = *directory + prefix;
  std::string try_name = base_name + "." + extension;
  
  struct stat file_info;
  int status;

  while (true) {
    status = stat(try_name.c_str(), &file_info);
    
    if (status == 0) {
      // A file with that name exists. Lets add a random integer at
      // the end of the file name, before the extension.
      std::stringstream ss;
      ss << rand();
      try_name = base_name + ss.str() + "." + extension;
    }
    else {
      // Unique file found!
      container->assign(try_name);

      // Open and close to reserve the filename
      std::ofstream& handle = *open_for_output(try_name.c_str());
      close_file(& handle);
      return;
    }
  }
}


/**
 * Removes the path from the name of the file and returns the naked
 * file name.
 */
void get_naked_file_name(const char* full_name, std::string* container) {

  std::string full_file_name = std::string(full_name);
  std::string naked_name;
  
  std::string::size_type slash_pos = full_file_name.find_last_of("/");
  if (slash_pos == std::string::npos) {
    // No slashes, so no path to strip
    naked_name = full_file_name;
  }
  else {
    naked_name = full_file_name.substr(slash_pos + 1);
  }
  container->assign(naked_name);
}


/**
 * Removes the path from the name of the file, capitalizes all
 * letters, and replaces dots with underscores. The returned name is
 * suitable to be used as an argument for #ifdef in the header files.
 */
void get_capitalized_name(const char* file_name, std::string* container) {

  std::string cn;
  get_naked_file_name(file_name, & cn);
  
  // Capitalize the string
  std::transform(cn.begin(), cn.end(), cn.begin(), toupper);
  
  // And convert the dots to underscores
  for (std::string::iterator it = cn.begin();
       it != cn.end();
       it++)
    {
      if (*it == '.') {
	*it = '_';
      }
    }
  container->assign(cn);
}


/**
 * Quote-escape all quotes in the string
 */
void
quote_escape(std::string source, std::string* result) {
  
  std::string source_copy = source;
  unsigned long int j = source_copy.find("\"");
  while (j < std::string::npos) {
    source_copy.replace(j, 1, "\\\"");
    j = source_copy.find("\"", j+2);
  }
  result->assign(source_copy);

}






#ifdef TIMING

#include <sys/times.h> // Timing (getrusage)
#include <stdint.h>

clock_t st_time;
struct tms st_cpu;
struct tms en_cpu;

void start_timer() {
  st_time = times(&st_cpu);
}


void display_and_reset_timer(const char* message) {
  clock_t en_time = times(&en_cpu);
  printf("%s, %jd, %jd, %jd\n",
	 message,
	 (intmax_t)(en_time - st_time),
	 (intmax_t)(en_cpu.tms_utime - st_cpu.tms_utime),
	 (intmax_t)(en_cpu.tms_stime - st_cpu.tms_stime));
  st_time = en_time;
}

#endif




