/***************************************************************************************************

 The containment analyzer is useful for checking the containment of two Buechi
 automata. Based on a SCT analyzer by: 
 Copyright (C) 2004  Chin Soon Lee
 Further copyright (C) 2009 Seth Fogarty

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

 The authors can be contacted by e-mail at cslee_sg@hotmail.com and sfogarty@gmail.com.

***************************************************************************************************/


/***************************************************************************************************

We are checking the containment of an automaton A in an automaton B. To do so,
we are complementing B and checking the intersection for emptiness. The
complementation is done using the Ramsey-based construction. This construction
uses 'supergraphs', the pair of an arc between two A states and a 0-1 labeled
graph over B states.


***************************************************************************************************/


typedef enum {EQUAL,GREATER,LESSER,INCOMPARABLE} SGraphRelation;

typedef enum {SRC,DST} NodeType;


typedef struct NodeRec {				/* an s.c.g. node */
  //NodeType type;
  struct BStateRec *bstate;				/* corresponding bstate (pointer) */
  //struct SGraphRec *sgraph;				/* corresponding s.c.g. */
  struct EdgeListRec *edges;			/* edges from this node */

  //int _flag;			/* working fields */
  struct EdgeListRec *_out_edges; 
} *Node;

typedef struct NodeListRec {			/* list of Node's */
  Node head;
  struct NodeListRec *tail;
} *NodeList;

typedef struct NodeRec* NodeArray;

typedef struct BStateRec {				/* a subject program bstate */
  string id;					/* its name */
  int index; /*it's index in the canoncial BStateArray */
  //struct AStateRec *astate;				/* corresponding astate */
	bool initial; /*SF: We don't need to store which are accepting, this is already
	encoded in the SGraph graphs */

  //NodeList src_nodes, dst_nodes;			/* s.c.g. nodes corresponding to bstate */
  //struct EdgeListRec *_in_edges;			/* edges entering nodes corresponding to bstate */ 
	/*I'm not sure this makes sense to have! In what graphs? */
	
  //struct EdgeListRec *_out_edges;			/* edges leaving nodes corresponding to bstate */

  bool _restriction;
  int _flag, _rdf_flag;			
  struct BStateListRec *_succ, *_pred;
  Node _src_node, _dst_node, _join_node;		/* for SCT */
  NodeList _src_nodes, _dst_nodes;	/* used for compositition */
} *BState;

typedef struct BStateListRec {			/* list of bstates */
  BState head;
  struct BStateListRec *tail;
} *BStateList;

typedef struct BStateRec* BStateArray;

#define DEC true
#define DEQ false


typedef struct EdgeRec { 			/* an s.c.g. (edgeendency) arc */
  bool accept;					/* its size label */
  Node src, dst;					/* corresponding src node */
  //struct SGraphRec *sgraph;				/* corresponding dst node */

//  int _flag;					/* working fields */
  bool _cln;					/* for SCP: removed by Cln? */
  //bool _ndg_interior;				/* for SCP: in current NDG interior? */
} *Edge ;

typedef struct EdgeListRec {			/* list of size-change arcs */
  Edge head;
  struct EdgeListRec *tail;
} *EdgeList;

typedef struct SGraphRec {				/* an s.c.g. */
  int index;					/* a numerical label */
  struct CProbRec *cp; /*a pointer to the problem, for the canonical list of
  bstates */
  struct AStateRec *src, *dst;			/* src and dst astate */
  int bsize;
  //NodeList src_nodes, dst_nodes;			/* src and dst nodes */
  NodeArray src_node_array, dst_node_array; /* point to an array of noderecs, not to an array of pointers to noderecs */ 
  BStateList reachable_from_init, reaching_one_scc; /*the bstates in this graph
  that are reachable from an initial state in one step or can reach a 1-labeled
  SCC in any number of steps, respectively */
  int _anchor;					/* working fields */
  //struct SGraphListRec *_composition;		/* for SCT critical (failure) path */
  struct SGraphListRec *_in_clo_pred;
} *SGraph;

typedef struct SGraphListRec {			/* list of SGraphs */
  SGraph head;
  struct SGraphListRec *tail;
} *SGraphList;

typedef struct AStateRec {				/* a subject program astate */
  string id;					        /* its name */
  //BStateList bstates;					/* corresponding bstates */
  SGraphList in_sgraphs, out_sgraphs;			/* corresponding s.c.g.'s to and from astate */
  bool accepting, initial;

  int _flag, _rdf_flag;				/* working fields */
  SGraphList _out_clo_sgraphs, _in_clo_sgraphs, _out_sgraphs;
	/* clo_sgraphs are ones computed in the closure. _out_sgraphs are the outgoing
	 * ORIGINAL sgraphs, which are not modified, appropriate to this SCC.
	 * It seems, however, that any sgraph which is not part of this SCC will never
	 * be part of an SCC. --S.F.
	 * _clo_sgraphs for SGraph */
  struct AStateListRec *_succ, *_pred;
} *AState;

typedef struct AStateListRec {			/* list of AState's */
  AState head;
  struct AStateListRec *tail;
} *AStateList;

typedef struct CProbRec {			/* CG [now ACG] */
  AStateList astates;
  int bsize; 
  BStateArray bstate_array; /* This is the canonical bstates, in an array */
  BStateList bstates;  /*This is a list pointing to all the canonical bstates,
  for cases where we want to do a depth-first search of all states. It is used
  to avoid reallocating the list skeleton every time we create a sgraph */
  SGraphList sgraphs;
} *CProb;

typedef enum {FORWARD,BACKWARD} 		/* follow the _succ or _pred links? */
	Direction;



/* Read CG using Paige and Yang's linear-time procedure. 
   Example of input: 
   	{ 
	  (ack ack {(m m dec)})
     	  (ack ack {(m m deq) (n n dec)}) 
	}^D
   Commas between items are also allowed. 
	SF: We now read input of the form
   	{ 
	  (ack ack {(m m dec)})
     	  (ack ack {(m m deq) (n n dec)}) 
	  }
		{ack foo bar}
	  where ack, foo, and bar are a list of accepting states. We add, in addition,
		a list of initial states.
	 
	 */
	
//CProb cg_read(void);

/* Writing out some of the above data structures */

void write_cp(CProb cg);
void write_sgraph(SGraph sgraph);
void write_sgraphs(SGraphList sgraphs);
void write_edges(EdgeList edges);

/* Freeing structures non-recursively */

void free_astates(AStateList astates);
void free_sgraphs(SGraphList sgraphs);
void free_bstates(BStateList bstates);
void free_nodes(NodeList nodes);
void free_edges(EdgeList edges);

/* Freeing structures recursively */

void dispose_astates(AStateList astates);
void dispose_sgraphs(SGraphList sgraphs);
void dispose_bstates(BStateList bstates);
void release_bstates(BStateList bstates);
void dispose_src_nodes(NodeArray nodes, int bsize);
void dispose_dst_nodes(NodeArray nodes, int bsize);
void dispose_edges(EdgeList edges);

void dispose_astate(AState astate);
void dispose_sgraph(SGraph sgraph);
void dispose_bstate(BState bstate);
void dispose_src_node(Node node);
void dispose_dst_node(Node node);
void dispose_edge(Edge edge);
void dispose_cp(CProb cg);

/* Misc */

int len_sgraphs(SGraphList sgraphs); 				/* length of */
int len_astates(AStateList astates);
int len_nodes(NodeList nodes);
int len_bstates(BStateList bstates);
int len_edges(EdgeList edges);

SGraphList copy_sgraphs(SGraphList sgraphs); 			/* copy top level */

/* Using a list as a database of AStates, and then BStates 
Lookups print an error and exit(1) if they fail.
*/

void astate_add(string id, bool init, bool accept, CProb cp); 

void bstate_add(string id, bool init, CProb cp); 

void sgraph_add(SGraph sgraph, CProb cp);

BStateList add_bstate_to_set(BState bstate, BStateList prev);
bool bstate_set_intersect(BStateList set1, BStateList set2);

SGraphList sgraph_prepend(SGraph Graph, SGraphList rest);

EdgeList edge_prepend(Edge Graph, EdgeList rest);
NodeList node_prepend(Node Graph, NodeList rest);

AState astate_lookup(string id, CProb cprob); 

BState bstate_lookup(string id, CProb cprob); 



/*Node lookup searches for a node already pointing to ID in the list of nodes.
 * If it doesn't find it, it creates one and points it to the proper BState */

NodeList new_nodelist(); 

// Make sure all our nodes point to us.
void finish_sgraph(SGraph sgraph);
CProb parse_cp();
extern int yyparse();

//NodeList new_nodes_from_array(int bsize, BStateArray vs, NodeType ty);
//NodeList new_untied_nodes_from_array(int bsize, BStateArray vs, NodeType ty);

//NodeArray new_untied_node_array(int bsize, BStateArray bstate_array, NodeType ty);
NodeArray new_node_array(int bsize, BStateArray bstate_array, NodeType ty);
BStateList bstates_listsort(BStateList list);
BStateArray array_of_bstatelist(BStateList list, int bsize);
BStateList list_of_bstatearray(BStateArray array, int bsize);

EdgeList edges_listsort(EdgeList list);
bool assert_edges_sorted(EdgeList edges);

Node node_array_lookup(string id, NodeArray narray, CProb cprob);

bool assert_consistent_bstate(BState a, BState b);
bool assert_consistent_cprob(CProb cp);
bool assert_consistent_nodes(SGraph sgraph);


void clear_out_edges(SGraph sgraph);
void clear_bstates(BStateArray bstates,int bsize); /* clear the _src_node and
 _dst_node fields of "bstates" */

