package dk.extra;
import dk.extra.Utils;
import dk.brics.automaton.*;

import java.io.*;
import java.util.*;
import dk.extra.Utils;

/** 
 * A program to investigate the probability of an automaton
 * being the universal automaton.
 */
public class AutomataComplementor {
    private static String _logFile = ""; 
    private static boolean warmup = true; //default
  
    /** Main entry to the program */
    public static void main(String args[]) {
    
    
	String logFile="";
	String pathToAutomata="";
	String pathToResults="";
	String idTag="";
	int rank=0;
	int numProcs=1;
	int minAlgorithm = -1;
    
	if (args.length < 5) {
	    System.err.println("Usage: \n"+
			       "[-l LogFile]  Store run time information in a log\n"+
			       "-s Source Directory  Indicate the path to the automata to be complemented\n" +
			       "-d Destination Directory  Path to the destination directory\n"+
			       "-a [Huffman | Hopcroft | Brzozowski]   Algorithm for minimization.\n"+
			       "-r Process Rank\n"+
			       "-p Number of Processors\n"+
			       "-warmup [ON | OFF] Default is ON \n"+
			       "[-i IDtag] id of the simulation.\n");
	    System.exit(1234);
	}
    
	/* Initialize the parameters observer */
	boolean[] requiredParameters = new boolean[5];
	for (int i=0; i<requiredParameters.length; i++)
	    requiredParameters[i] = false;
    
	/* Parse the command line options*/  
	for (int i=0; i<args.length; i++) {
      
	    if (args[i].equalsIgnoreCase("-l"))
		logFile = args[i+1];
      
	    if (args[i].equalsIgnoreCase("-s")) {
		pathToAutomata = args[i+1];
		requiredParameters[0] = false;
	    }
      
	    if (args[i].equalsIgnoreCase("-d")) {
		pathToResults = args[i+1];
		requiredParameters[1] = false;
	    }
      
	    if (args[i].equalsIgnoreCase("-i"))
		idTag = args[i+1];
      
	    if (args[i].equalsIgnoreCase("-r")) {
		rank = Integer.parseInt(args[i+1]);
		requiredParameters[2] = false;
	    }
      
	    if (args[i].equalsIgnoreCase("-p")) {
		numProcs = Integer.parseInt(args[i+1]);
		requiredParameters[3] = false;
	    }
      
	    if (args[i].equalsIgnoreCase("-warmup")) {
		warmup = parseSwitch(args[i+1]);
		// not a required parameter, default is true
	    }
	    if (args[i].equalsIgnoreCase("-a")) {
		requiredParameters[4] = false; 
		String algo = args[i+1].toLowerCase();
		if (algo.indexOf("huf")>-1)
		    minAlgorithm = Automaton.MINIMIZE_HUFFMAN;
		else if (algo.indexOf("hop")>-1)
		    minAlgorithm = Automaton.MINIMIZE_HOPCROFT;
		else if (algo.indexOf("br")>-1)
		    minAlgorithm = Automaton.MINIMIZE_BRZOZOWSKI;
		else {
		    System.err.println("Unknown algorithm: "+algo+" . I am exiting now!");
		    System.exit(1234);
		}
	    }
	} //Parse command line options
    
	for (int i=0; i<requiredParameters.length; i++) {
	    if (requiredParameters[i]) {
		System.err.println("Required parameter "+i+" could not be parsed. Exiting!");
		System.exit(1234);
	    }
	}
    
	if (rank == 0)
	    System.out.println("Parsed parameters: \n"+
			       "log: "+logFile+", \n"+
			       "source: " + pathToAutomata+ ", \n"+
			       "output: " + pathToResults + ", \n"+
			       "algorithm: " + ((minAlgorithm==Automaton.MINIMIZE_HUFFMAN)? "Huffman" :
						(minAlgorithm==Automaton.MINIMIZE_HOPCROFT)? "Hopcroft" : 
						(minAlgorithm==Automaton.MINIMIZE_BRZOZOWSKI)? "Brzozowski" : "Unknown") + ", \n"+
			       "idTag: "+idTag+", \n"+
			       "warmup: "+warmup + "\n"+
			       "rank = "+rank+", \n"+
			       "number of processors = "+numProcs );
    
	// check both source and output directories for existence, create if necessary
	File source = new File(pathToAutomata);
	File destination = new File(pathToResults);
    
	if (! source.exists()) {
	    System.err.println("The source directory:\n"+pathToAutomata+"\ndoesn't exist! I am exiting now!");
	    System.exit (9);
	}
	if (rank==0)
	    System.out.println("Source directory found!");
    
	if (rank==0) {
	    if (! destination.exists()) {
		System.out.println("This is process 0. Destination directory doesn't exist. I attempting to create it now!");
		if ( destination.mkdirs())
		    System.out.println("Destination directory successfully created!");
		else {
		    System.err.println("Couldn't create output directory. Exiting");
		    System.exit(10);
		}
	    }
	}
	else if (rank > 0) {
	    //All other processes: wait until the directory has been created!
	    while (! destination.exists())
		;
	}
    
	_logFile = logFile; //need later, don't want to pass along.
	startSimulation(logFile, source, pathToResults, idTag, minAlgorithm, rank, numProcs);
    }//main
  
    /*#############################################################################*/
  
    public static void startSimulation(String logFile, File dir, String pathToResults, 
				       String idTag, int minAlgo, int rank, int numProcs) {
    
	if (warmup)
	    warmup(minAlgo);
    
	/** Processes all of the files in the directory by calling singleAutomaton to handle each automaton */
	// This partitions the set of files for individual processors
	File[] simulationFiles = dir.listFiles(Utils.validNames(rank, numProcs));
	if (simulationFiles == null) {
	    System.err.println("Source directory doesn't exist or is empty. Exiting simulation!");
	    System.exit(1); 
	}

	System.out.println("Starting to process these "+simulationFiles.length + " automata.");
	for (int i=0; i<simulationFiles.length; i++) {
	    // extract data
	    //System.out.println("Automaton #"+i);
	    boolean[][] transitionMatrix_A = extractTransitionMatrix(simulationFiles[i], "A_");
	    boolean[][] transitionMatrix_B = extractTransitionMatrix(simulationFiles[i], "B_");
      
	    boolean[] finalStatesArray = extractFinalArray(simulationFiles[i]);
	    Automaton a = new RandomAutomaton(transitionMatrix_A, transitionMatrix_B, finalStatesArray, minAlgo);
      
	    // create the output File object
	    File f_pathToResults = new File(pathToResults);      
	    File outFile = new File(f_pathToResults, simulationFiles[i].getName()+"-processed-"+idTag);
	    analyzeSingleAutomaton(a, outFile, minAlgo);
	    //System.out.println("Automaton #"+i + " is done.");
	}//for
    }
  
    /*#############################################################################*/
  
    static boolean parseSwitch(String theSwitch) {
	if (theSwitch.equalsIgnoreCase("on"))
	    return true;
	if (theSwitch.equalsIgnoreCase("off"))
	    return false;
	else {
	    System.err.println("Cannot parse this switch: "+theSwitch);
	    System.exit(1234);
	    return false; //unreachable
	}
    }
  
    /*#############################################################################*/
  
    /** To avoid java quirks like increasing the heap size and such,
	first we spin around for about 5 minutes/processor as a warmup */
    public static void warmup(int algo) {
    
	File[] simulationFiles = new File("/users/dtabakov/scratch/50-50_warmup").listFiles();
	if (simulationFiles == null) {
	    System.err.println("Warmup directory doesn't exist or is empty. Continuing anyways.");
	    return;
	}
      
	long start = System.currentTimeMillis();
	for (int i=0; i<simulationFiles.length; i++) {
	    //System.out.println("Processing file "+simulationFiles[i]);
	    // extract data
	    boolean[][] transitionMatrix_A = extractTransitionMatrix(simulationFiles[i], "A_");
	    boolean[][] transitionMatrix_B = extractTransitionMatrix(simulationFiles[i], "B_");
      
	    boolean[] finalStatesArray = extractFinalArray(simulationFiles[i]);
	    Automaton a = new RandomAutomaton(transitionMatrix_A, transitionMatrix_B, finalStatesArray, algo);
      
	    // create the output File object
      
	    analyzeSingleAutomaton(a, null, Automaton.MINIMIZE_HOPCROFT);
	}//for
	System.out.println("Warmup finished. Total time taken for warmup: "+(System.currentTimeMillis() - start)+" ms.");
    }
  
    /*#############################################################################*/
    /** For debugging purposes... creates the automaton using the 
     * specified file to extract the transition matrix and final states 
     * array. Assumes that the file contains this info ;-).
     */
    public static Automaton createFromFile(String inputFile) {
    
    
	File in = new File (inputFile);
	boolean[][] transitionMatrix_A = extractTransitionMatrix(in, "A_");
	boolean[][] transitionMatrix_B = extractTransitionMatrix(in, "B_");
    
	boolean[] finalStatesArray = extractFinalArray(in);
	int minAlgo = Automaton.MINIMIZE_HOPCROFT;
	return new RandomAutomaton(transitionMatrix_A, transitionMatrix_B, finalStatesArray, minAlgo);
    
    }
  
    /*#############################################################################*/ 
  
    /** 
     * Complements a single automaton, and writes the data to the
     * output file
     */
    public static void analyzeSingleAutomaton(Automaton a, File outputFile, int minAlgo) {
    
    
	//save some time when running on partially processed data
	if (outputFile != null && // it is null during warmup
	    outputFile.exists() && 
	    outputFile.length() > 0) 
	    {
		return;
	    }
    
    
	boolean totalLanguage = false;
	int prunedSize = a.getNumberOfStates();
	long totalMemory = Runtime.getRuntime().totalMemory();
	long initiallyFreeMemory = Runtime.getRuntime().freeMemory();
	
	/** 
	 * Note that Hopcroft and Huffman determinize the automaton before
	 * they do the actual minimization. However, the method for
	 * determinizing is checking a flag in the automaton to see if it
	 * is deterministic already, and if so, it returns
	 * immediately. This flag gets set by the same method
	 * otherwise. So by determinizing here we know that the flag will
	 * be set and the minimization algo will not duplicate the work
	 * because it will see the flag.
	 * 
	 * Note also that brzozowski doesn't need a deterministic
	 * automaton, so for this min algo we skip the determinization
	 * step.
	 */
    
    
	//start the clock
	long clockStart = System.currentTimeMillis();
    
	/**
	 * Determinizing step 
	 */ 
	long detStart = clockStart;
	int earlyTermination = 0;
	if (minAlgo != Automaton.MINIMIZE_BRZOZOWSKI)
	    earlyTermination = a.determinizeET(); //or determinizeET() if we are using early termination
	//a.determinize();

	
	//stop the clock for determinizing
	long detDone = System.currentTimeMillis();
    
        /** With early termination, all of the remaining code is more or less
         * useless, because the automaton is in inconsistent state. That is why
         * we don't measure anything else.*/
        
	
        if (earlyTermination == -1) {
	    // The automaton is deterministic, so we returned without doing the 
	    // construction. This means that it may or may not be universal. This 
	    // should not happen, but I have this here for completeness 
	    //
	    totalLanguage = checkUniversality(a);
	    detDone = System.currentTimeMillis();  ///incorporate the time to check universality in this case
	    System.err.println("Unexpected: the automaton was already deterministic when calling determinize!");
        }
        else 
	    totalLanguage = (earlyTermination == 1);
        
        int detSize = 0, minSize = 0;
	long uniCheckStart = 0, uniCheckDone = 0, uniFreeMemory = 0;
        long minStart = 0, minDone = 0, minFreeMemory = 0;
        long detFreeMemory = 0;
        long clockStop = 0, endFreeMemory = 0;
	//*/
	


	//    
	//    Do not delete the commented lines! 
	//    =======================================
    
	


	/**
	 * Collect statistics before minimizing
	 
	
	int detSize = a.getNumberOfStates();
	long detFreeMemory = Runtime.getRuntime().freeMemory();
	 
	*/
    
	/**
	 * Universality che 
	 * Now we have a determinitic automaton. We check for
	 * universality by looking at all states and checking if they
	 * are all accepting states. However, because of the way they
	 * implemented determinization, this is not enough. We also
	 * check if the transition relation is total, that is, if
	 * there is an outgoing edge on each letter from each state.
	 
	long uniCheckStart = System.currentTimeMillis(); // <<--- start the clock
	
	totalLanguage = checkUniversality(a);
	
	long uniCheckDone = System.currentTimeMillis(); // <<--- stop the clock
	long uniFreeMemory = Runtime.getRuntime().freeMemory();
	 
	*/

    
	/**
	 * Minimizing step
	  
	long minStart = System.currentTimeMillis(); // <<--- start the clock
	
	a.minimize(); // the method determinizes (Hop/Huf only, but they are already deterministic)
		
	long minDone = System.currentTimeMillis(); // <<--- stop the clock
	int minSize = a.getNumberOfStates();
	long minFreeMemory = Runtime.getRuntime().freeMemory();
		
	//stop the global clock
	long clockStop = System.currentTimeMillis();
	long endFreeMemory = Runtime.getRuntime().freeMemory();
	*/
	
	/*
	 * END OF MINIMIZING STEP
	 */

	

	
	// prepare the data to be written
	String toWrite = "";
	
	toWrite+="Total: "+totalLanguage                                    + '\n';
	toWrite+="Determinize Time: "+           (detDone -       detStart) + '\n';
	toWrite+="Uni Check Time: "+        (uniCheckDone -  uniCheckStart) + '\n';
	toWrite+="Minimize Time: "+              (minDone -       minStart) + '\n';
	toWrite += '\n';
	toWrite+="Pruned Size: " +       prunedSize + '\n';
	toWrite+="Determinized Size: " +    detSize + '\n';
	toWrite+="Minimized Size: " +       minSize + '\n';
	toWrite += '\n';
	toWrite += "Total memory: "                    + totalMemory + '\n';
	toWrite += "Initially free memory: "   + initiallyFreeMemory + '\n';
	toWrite += "After determinize free memory: " + detFreeMemory + '\n';
	toWrite += "After unicheck free memory: " +    uniFreeMemory + '\n';
	toWrite += "After minimize free memory: " +    minFreeMemory + '\n';
	toWrite += "Finally free memory: " +           endFreeMemory + '\n';
    
	//toWrite+="Complemented Size: " +   compSize + '\n';
	//toWrite+="Resulting Size: " + resultingSize + '\n';
    
	// Create the file and write the data
	if (outputFile != null) {
	    try {
		PrintWriter out = new PrintWriter(new FileWriter(outputFile));
		out.println(toWrite);
		out.close();
	    }
	    catch (IOException e) {
		System.err.println("Error opening the file " + outputFile.toString());
	    }
	}
    
	//    }
	//    catch (IOException e) {
	//      System.err.println(e.toString());
	//    }
    }
  
  
    /*#############################################################################*/
  
  
  
    static boolean checkUniversality(Automaton a) {
    
	Set allStates = a.getStates();
	Iterator i = allStates.iterator();
	boolean universalLanguage = true; // until proven otherwise
    
	while (i.hasNext()) {
	    State q = (State) i.next();
	    if (! q.isAccept() )
		return false;
	    else { 
		// OK, it is an accepting state, but what about the outgoing transitions?
		// we need a total transition relation
		Set allTransitions = q.getTransitions();
		Iterator j = allTransitions.iterator();
		boolean hasA = false;
		boolean hasB = false;
		while (j.hasNext()) {
		    Transition t = (Transition) j.next();
		    if (t.getMin() != t.getMax()) {
			Utils.affirm(t.getMin() == 'a', "Double transition: a");
			Utils.affirm(t.getMax() == 'b', "Double transition: b");
			hasA = true;
			hasB = true;
		    }
		    else { //min == max
			if (t.getMin() == 'a')
			    hasA = true;
			else if (t.getMin() == 'b')
			    hasB = true;
		    }//else
		}//while
		if (!hasA || !hasB)
		    return false;
	    }
	}
	return true;
    }
  
    /*#############################################################################*/
  
    public static boolean[][] extractTransitionMatrix(File filename, String id) {
    
	try {
	    boolean[][] transitionMatrix;
      
	    BufferedReader inFile = new BufferedReader(new FileReader(filename));
	    String currentLine="";
      
	    do {
		currentLine = inFile.readLine();
	    } while (currentLine != null && currentLine.indexOf(id+"Transition matrix follows") == -1);
      
	    if (currentLine==null) {
		System.err.println("There is no "+id +"Transition matrix in this file "+filename.toString()+". Exiting!");
		System.exit(2);
	    }
      
	    // Now follows the transition matrix. Count the total number of 0's and 1's
	    currentLine = inFile.readLine();
	    int charCounter = 0;
	    for (int i=0; i<currentLine.length(); i++) {
		char currentChar = currentLine.charAt(i);
		if (currentChar=='0' || currentChar=='1')
		    charCounter++;
	    }
      
	    //This gives us the size of the square matrix. Create it and fill it up
	    transitionMatrix = new boolean[charCounter][charCounter] ;
	    for (int i=0; i<charCounter; i++) {
		for (int j=0; j<charCounter; j++) {
		    transitionMatrix[i][j] = ( currentLine.charAt(j)=='1' );
		}
		currentLine = inFile.readLine();
	    }
	    // make sure that there is nothing left after the matrix
	    //currentLine = inFile.readLine();
      
	    char nextChar = currentLine.charAt(0);//must be '#' for the final states comment, or some whitespace character.
	    if (nextChar=='0' || nextChar=='1')
		System.out.println("In this file:\n"+filename.getName() + "\n the matrix was followed by a matrix element. The offending line:\n"+currentLine+"\nInvestigate!");
      
	    inFile.close(); //be nice
      
	    return transitionMatrix;
	}//try
	catch (IOException e) {
	    System.err.println("IOException in extractTransitionMatrix()!"+e);
	    System.exit(2);
	}
    
	return null;
    }
  
    /*#############################################################################*/
  
    public static boolean[] extractFinalArray(File filename) {
	boolean[] finalArray = null;
	try {
	    BufferedReader inFile = new BufferedReader(new FileReader(filename));
	    String currentLine="";
      
	    do {
		currentLine = inFile.readLine();
	    } while (currentLine != null && currentLine.indexOf("Final states array") == -1);
      
	    if (currentLine==null) {
		System.err.println("There is no array of final states in this file "+filename.toString()+". Exiting!");
		System.exit(3);
	    }
      
	    // Now follows the array. Count the total number of 0's, a's and b's
	    currentLine = inFile.readLine();
	    int charCounter = 0;
	    for (int i=0; i<currentLine.length(); i++) {
		char currentChar = currentLine.charAt(i);
		if (currentChar=='1' || currentChar=='0')
		    charCounter++;
	    }
      
	    // We know the size. Create the array and fill up the data
	    finalArray = new boolean[charCounter];
	    for (int i=0; i<currentLine.length(); i++)  {
		finalArray[i] = (currentLine.charAt(i)=='1');
	    }
	    inFile.close();
	}//try
	catch (IOException e) {
	    System.out.println("Exception in extractFinalArray: \n"+e.toString());
	}
    
	return finalArray;    
    }
  
    /*#############################################################################*/
  
  
  
  
}
