import java.io.*;
import java.text.DecimalFormat;


/**
 * Create and save random transition matrices and final states arrays
 * to be parsed by AutomataComplementor later
 */
public class AutomataCreator {
  
    public static final int UNSTRUCTURED = 0;
    public static final int LINEAR = 1;
    public static final int MATRIX = 2;

    java.util.Random randy = new java.util.Random();
    double epsilon = 0.001;
    int structType = UNSTRUCTURED;
    float structDistance = 0;

    

    /**
     * A hook to set the structure mode of the automaton. 
     */
    void setStructureType(int st) {
	structType = st;
    }

    



    /**
     * A hook to set the distance in a structured automaton
     */
    void setDistance(float d) {
	if (structType == UNSTRUCTURED)
	    System.out.println("WARNING: Setting a distance has no effect when the automaton is unstructured!");
	structDistance = d;
    }


    


    /**
     * Creates the matrix of accepting states
     */
    public boolean[] generateRandomFinalStates(int numberOfStates, int numberOfFinalStates, boolean initialMustAccept) {

	
	int state = 0;
	boolean[] finalStates = new boolean[numberOfStates];
    
	for (int i=0; i< numberOfFinalStates; i++) {
	    
	    if (i==0 && initialMustAccept)
		finalStates[0] = true;
	    
	    else {
		do {
		    state = randy.nextInt(numberOfStates);
		} while (true == finalStates[state]);
		finalStates[state] = true;
	    }//if
	}//for
    
	return finalStates;
    }
  

  

    /**
     * Creates a transition matrix with @param numberOfStates number
     * of states (that is, a square matrix of that size, and a 
     * @param numberOfTransitions number of transitions. In addition,
     * if @param initialNodeMustLeave is set, the initial state is
     * forced to make a transition.
     */
    public boolean[][] generateRandomTransitions(int numberOfStates, 
						 int numberOfTransitions, 
						 boolean initialNodeMustLeave) {
	
	// Two additions to handle the new structural types
	if (structType == LINEAR)
	    return generateLinearModelTransitions(numberOfStates, numberOfTransitions, initialNodeMustLeave);

	if (structType == MATRIX)
	    return generateMatrixModelTransitions(numberOfStates, numberOfTransitions, initialNodeMustLeave);

	
	// Otherwise create unstructured automaton (default)
	boolean[][] transitionTable = new boolean[numberOfStates][numberOfStates];
	int fromState;
	int toState;

	if (initialNodeMustLeave) {
	    if (numberOfTransitions < 1)
		throw new AssertionError("There should be at least 1 transition!");
	    
	    fromState = 0;
	    toState = randy.nextInt(numberOfStates);
	    if (transitionTable[fromState][toState])
		throw new AssertionError("This is impossible!");

	    transitionTable[fromState][toState] = true;
	    
	    // Since we have just consumed one transition, we decrease their number
	    numberOfTransitions --;
	}
	
	for (int i=0; i < numberOfTransitions; i++) {
	    do {
		fromState = randy.nextInt(numberOfStates);
		toState = randy.nextInt(numberOfStates);
	    } while (true == transitionTable[fromState][toState]);
	    transitionTable[fromState][toState] = true;
	}//for loop
    
	return transitionTable;
    }//generateRandomTransitions





    /**
     * Creates a transition matrix with @param numberOfStates number
     * of states (that is, a square matrix of that size, and a 
     * @param numberOfTransitions number of transitions. In addition,
     * if @param initialNodeMustLeave is set, the initial state is
     * forced to make a transition. The connectiveness of the states
     * is implemented using a linear model, whereas a state cannot
     * talk to a state further than @param numberOfStates times
     * structDistance.
     */
    public boolean[][] generateLinearModelTransitions(int numberOfStates, int numberOfTransitions, boolean initialNodeMustLeave) {
	
	int distance = Math.round(numberOfStates * structDistance); 
	
	if (distance == 0) {
	    System.out.println("The linear distance is 0. Increasing to 1 otherwise no transitions will be possible.");
	    distance = 1;
	}
	
	boolean[][] transitionTable = new boolean[numberOfStates][numberOfStates];
	int fromState;
	int toState;

	if (initialNodeMustLeave) {
	    if (numberOfTransitions < 1)
		throw new AssertionError("There should be at least 1 transition!");
	    
	    fromState = 0;
	    do {
		toState = randy.nextInt(numberOfStates);
	    } while (Math.abs(toState - fromState) > distance);
	    
	    
	    if (transitionTable[fromState][toState])
		throw new AssertionError("This is impossible!");
	    
	    transitionTable[fromState][toState] = true;
	    
	    // Since we have just consumed one transition, we decrease their number
	    numberOfTransitions --;
	}
	
	for (int i=0; i < numberOfTransitions; i++) {
	    do {
		fromState = randy.nextInt(numberOfStates);
		toState = randy.nextInt(numberOfStates);
	    } while (true == transitionTable[fromState][toState] || Math.abs(toState - fromState) > distance);
	    transitionTable[fromState][toState] = true;
	}//for loop
    
	return transitionTable;
    }

    

    /**
     * Returns true if the two nodes are close enough in Manhattan
     * distance, assuming that the size of the automaton is a perfect
     * square.
     */
    boolean isCloseEnough(int node1, int node2, int distance, int states) {

	int square = (int) Math.round(Math.sqrt(states));
	
	int node1x = node1 % square;
	int node1y = node1 / square;

	int node2x = node2 % square;
	int node2y = node2 / square;

	int separation = Math.abs(node1x - node2x) + Math.abs(node1y - node2y);

	return separation <= distance;
    }


    /**
     * Generates the matrix model of the random automata. Each node is
     * only allowed to talk to nodes within some Manhatan distance
     * from it.
     */
    public boolean[][] generateMatrixModelTransitions(int numberOfStates, int numberOfTransitions, boolean initialNodeMustLeave) {
	int distance = Math.round(numberOfStates * structDistance);
	
	if (distance == 0) {
	    System.out.println("The linear distance is 0. Increasing to 1 otherwise no transitions will be possible.");
	    distance = 1;
	}
	
	boolean[][] transitionTable = new boolean[numberOfStates][numberOfStates];
	int fromState;
	int toState;

	if (initialNodeMustLeave) {
	    if (numberOfTransitions < 1)
		throw new AssertionError("There should be at least 1 transition!");
	    
	    fromState = 0;
	    do {
		toState = randy.nextInt(numberOfStates);
	    } while (! isCloseEnough(fromState, toState, distance, numberOfStates));
	    
	    
	    if (transitionTable[fromState][toState])
		throw new AssertionError("This is impossible!");
	    
	    transitionTable[fromState][toState] = true;
	    
	    // Since we have just consumed one transition, we decrease their number
	    numberOfTransitions --;
	}
	
	for (int i=0; i < numberOfTransitions; i++) {
	    do {
		fromState = randy.nextInt(numberOfStates);
		toState = randy.nextInt(numberOfStates);
	    } while (true == transitionTable[fromState][toState] || (! isCloseEnough(fromState, toState, distance, numberOfStates)));
	    transitionTable[fromState][toState] = true;
	}//for loop
    
	return transitionTable;
    }

}
