import java.io.*;
import java.text.DecimalFormat;

public class AutomataCreatorLF extends AutomataCreator {
    /** Create and save random transition matrices and final states arrays
     * to be parsed by AutomataComplementor later
     * Uses linear density of final states
     */
  
    static java.util.Random randy = new java.util.Random();
    static double epsilon = 0.001;
  
    public static void main(String[] args) {
       if (args.length < 8) {
           System.err.println("Not enough parameters!");
           System.err.println("Required parameters: ");
           System.err.println("-rf (from) -rt (to) -rs (step) to indicate the transition density distribution");
           System.err.println("-ff (from) -ft (to) -fs (step) to indicate the final states distribution");
           System.err.println("-sf (from) -st (to) -ss (step) to indicate the size of the automata");
           System.err.println("-c Number of automata that you would like to create ");
           System.err.println("-d Destination directory");
           System.err.println("Optional parameters: ");
           System.err.println("[-initAcpt] indicates that the initial node must accept (default: off) ");
           System.err.println("[-initTrans] indicates that the initial node must transition on both letters (default: off)");
           System.err.println("-struct [LINEAR | MATRIX | NONE] indicates the structural model");
           System.err.println("-distance [float] defines the distance multiplier for the structural model");
           System.err.println("-squarestep [integer] defines the square step. Use 0 to indicate linear steps please!");
           System.err.println("-f filename: Turn on single file mode. This provides the file name.");
           System.err.println("             In SFM the -rf, -ff, and -sf switches control the structural params.");
       }
       
       else {
           // parse the input
           int sFrom=0;
           int sTo = 0;
           int sStep = 0;
           int count=0;
           double rFrom=0;
           double rTo = 0;
           double rStep = 0;
           double fFrom=0;
           double fTo=0;
           double fStep = 0;
           String outputDir="";
           boolean initialNodeMustAccept = false;
           boolean initialNodeMustLeave = false;
           boolean b_squareStep = false;
           int i_squareStep = 0;
           boolean singleFileMode = false;
           String sfmFilename = "";

           int structureType = AutomataCreator.UNSTRUCTURED;
           float distance = 0;
      
           boolean[] requiredParameters = new boolean[] {true, true, true, true, true, true, true, true};
      
           for (int i=0; i<args.length; i++) {
        
              if (args[i].equalsIgnoreCase("-initAcpt"))
                  initialNodeMustAccept = true;

              if (args[i].equalsIgnoreCase("-initTrans"))
                  initialNodeMustLeave = true;
        
              // Transition density
              if (args[i].equalsIgnoreCase("-rf") && i+1 < args.length )
                  rFrom = Double.parseDouble(args[i+1]);

              if (args[i].equalsIgnoreCase("-rt") && i+1 < args.length )
                  rTo = Double.parseDouble(args[i+1]);

              if (args[i].equalsIgnoreCase("-rs") && i+1 < args.length )
                  rStep = Double.parseDouble(args[i+1]);

              // Final state density
              if (args[i].equalsIgnoreCase("-ff") && i+1 < args.length )
                  fFrom = Double.parseDouble(args[i+1]);

              if (args[i].equalsIgnoreCase("-ft") && i+1 < args.length )
                  fTo = Double.parseDouble(args[i+1]);

              if (args[i].equalsIgnoreCase("-fs") && i+1 < args.length )
                  fStep = Double.parseDouble(args[i+1]);
                  
              // Size of the automata
              if (args[i].equalsIgnoreCase("-sf") && i+1 < args.length )
                  sFrom = Integer.parseInt(args[i+1]);

              if (args[i].equalsIgnoreCase("-st") && i+1 < args.length )
                  sTo = Integer.parseInt(args[i+1]);

              if (args[i].equalsIgnoreCase("-ss") && i+1 < args.length )
                  sStep = Integer.parseInt(args[i+1]);
              
              
              // Count of the automata
              if (args[i].equalsIgnoreCase("-c") && i+1 < args.length )
                  count = Integer.parseInt(args[i+1]);
              
              // Output directory
              if (args[i].equalsIgnoreCase("-d") && i+1 < args.length )
                  outputDir = args[i+1];

              // Single file mode
              if (args[i].equalsIgnoreCase("-f") && i+1 < args.length ) {
                  singleFileMode = true;
                  sfmFilename = args[i+1];
              }

              // Structured circuits:
              if (args[i].equalsIgnoreCase("-struct") && i+1 < args.length ) {
                  if (args[i+1].equalsIgnoreCase("LINEAR"))
                     structureType = AutomataCreator.LINEAR;
                  else if (args[i+1].equalsIgnoreCase("MATRIX"))
                     structureType = AutomataCreator.MATRIX;
                  else if (args[i+1].equalsIgnoreCase("NONE"))
                     structureType = AutomataCreator.UNSTRUCTURED;
                  else
                     throw new AssertionError("Unknown structure type: " + args[i+1]);
              }

              
              // Define the distance too
              if (args[i].equalsIgnoreCase("-distance") && i+1 < args.length ) {
                  distance = Float.parseFloat(args[i+1]);
              }

              // Square steps, if specified
              if (args[i].equalsIgnoreCase("-squarestep") && i+1 < args.length ) {
                  i_squareStep = Integer.parseInt(args[i+1]);
                  if (i_squareStep > 0)
                     b_squareStep = true;
              }
              

           }//for
      
           
           // If the size step is 0, use square step instead. Make
           // sure that the starting/ending sizes are squares!

           if (b_squareStep) {
              if ( Math.abs(Math.floor(Math.sqrt(sFrom)) * Math.floor(Math.sqrt(sFrom)) - sFrom) > 0.1 )
                  throw new AssertionError("Starting size is not a square!");
              if ( Math.abs(Math.floor(Math.sqrt(sTo)) * Math.floor(Math.sqrt(sTo)) - sTo) > 0.1 )
                  throw new AssertionError("Ending size is not a square!");
           }
           
           
           System.out.println("Parsed input:");

           if (singleFileMode) {
              System.out.println("Single File Mode...");
              System.out.println("Transition Density: \t"+rFrom);
              System.out.println("Acceptance Density: \t"+fFrom);
              System.out.println("Size: \t\t" + sFrom);
           }

           else {
              System.out.println("size: "+sFrom + " - "+sTo + " by " +sStep);
              System.out.println("count = " + count);
              System.out.println("rho-low = " + rFrom + ",\trho-to = " + rTo + ",\t rho-step = " + rStep + 
                               ", \nfinal-low = " + fFrom + ",\t final-to = " + fTo + ",\t final-step = " + fStep + 
                               ", \noutput dir = " + outputDir + 
                               ", \ninitial node must accept: " + initialNodeMustAccept +
                               ", \ninitial node must have a successor on 'a' and 'b': "+ initialNodeMustLeave);
           }

           System.out.print("Structured Automata: " );
           if (structureType == AutomataCreator.LINEAR)
              System.out.println("LINEAR");
           else if (structureType == AutomataCreator.MATRIX)
              System.out.println("MATRIX");
           else if (structureType == AutomataCreator.UNSTRUCTURED)
              System.out.println("UNSTRUCTURED");
           else
              throw new AssertionError("Unknown structure type: " + structureType);
           
           System.out.println("Structural distance: " + distance);
           System.out.println("Square steps: " + b_squareStep + " (" + i_squareStep + ")");
           
           


           // In Single File Mode, deduce the parameters to make the rest of the code work
           if (singleFileMode) {
              rTo = rFrom;
              rStep = 1; // arbitrary positive number
              fTo = fFrom;
              fStep = 1;
              sTo = sFrom;
              sStep = 1;

              count = 1;

              //deduce the output directory...
              File f = new File(sfmFilename);
              if (f.isDirectory()) {
                  throw new AssertionError("Output file name " + sfmFilename + " should be a file, not a directory!");
              }

              String absolutePath = f.getAbsolutePath();
              int slash = absolutePath.lastIndexOf('/');
              outputDir = absolutePath.substring(0, slash);
              System.out.println("Output directory is " + outputDir);
           }
           


           File f_outputDir = new File(outputDir);
           //Check if the output dir exists
           if (! f_outputDir.exists()) {
              System.out.println("Destination directory doesn't exist.");
              if (f_outputDir.mkdirs())
                  System.out.println("Directory successfully created!");
              else {
                  System.err.println("Couldn't create output directory. Exiting");
                  System.exit(10);
              }
           }
           
           
           // Check for square steps
           if (b_squareStep) {
              int start = (int) Math.round(Math.sqrt(sFrom));
              int finish = (int) Math.round(Math.sqrt(sTo));
              
              int size = start * start;

              do {
                  System.out.println("Size: "+size);
                  AutomataCreatorLF aclf = new AutomataCreatorLF();
                  
                  aclf.setStructureType(structureType);
                  aclf.setDistance(distance);
                  
                  aclf.makeAutomataSeries(size, count, rFrom, rTo, rStep, fFrom, fTo, fStep, f_outputDir, initialNodeMustAccept, initialNodeMustLeave, singleFileMode, sfmFilename);

                  start += i_squareStep;
                  size = start * start;
              } while (size <= finish * finish); 
           }

           else {
              for (int size = sFrom; size<=sTo; size+=sStep) {
                  System.out.println("Size: "+size);
                  AutomataCreatorLF aclf = new AutomataCreatorLF();
                  
                  aclf.setStructureType(structureType);
                  aclf.setDistance(distance);
                  
                  aclf.makeAutomataSeries(size, count, rFrom, rTo, rStep, fFrom, fTo, fStep, f_outputDir, initialNodeMustAccept, initialNodeMustLeave, singleFileMode, sfmFilename);
              }
           }
           
       }//else (enough parameters)
    }//main
  
  
    /*#############################################################################*/
  
  
    public void makeAutomataSeries(int size, int count, double rFrom, double rTo, double rStep,
                               double fFrom, double fTo, double fStep, File outputDir, 
                               boolean initialNodeMustAccept, boolean initialNodeMustLeave,
                               boolean singleFileMode, String sfmFilename) {
    
       for (double r = rFrom; r<=rTo + epsilon; r+=rStep) {
           System.out.println(r);
           for (double f = fFrom; f<=fTo + epsilon; f+=fStep) {
              makeAutomata(size, r, f, count, outputDir, initialNodeMustAccept, initialNodeMustLeave, singleFileMode, sfmFilename);
           }
       }
    }
  
  
 
    /* ############################################################################# */
  
    public void makeAutomata(int size, double rDensity, double fDensity, int count, File outputDir, 
                          boolean initialNodeMustAccept, boolean initialNodeMustLeave,
                          boolean singleFileMode, String sfmFilename) {

       int numTransitions =  (int) Math.round(size * rDensity);
       int numFinal = (int) Math.round(size * fDensity);
       System.out.println("("+numTransitions + ", "+numFinal+")");
    
       for (int i=0; i<count; i++) {
           //the transition matrix
           boolean[][] A_transitionMatrix = generateRandomTransitions(size, 
                                                               numTransitions, 
                                                               initialNodeMustLeave);
           boolean[][] B_transitionMatrix = generateRandomTransitions(size, 
                                                               numTransitions, 
                                                               initialNodeMustLeave);
      
           // the final states array
           boolean[] finalStatesArray = generateRandomFinalStates(size, numFinal, initialNodeMustAccept);
      
           // the file name
           DecimalFormat formatter = new DecimalFormat("0.00");

           String outputFileName = "new-s-"+size+"-r-"+formatter.format(rDensity)+"-f-"+formatter.format(fDensity)+
              "--"+(i+1)+"-of-"+count;
      
           try {
              PrintWriter outputFile;
              if (singleFileMode) {
                  outputFile = new PrintWriter(new FileWriter(new File(sfmFilename)));
              }
              
              else {
                  outputFile = new PrintWriter(new FileWriter(new File(outputDir, outputFileName)));
              }
              
              int aCounter = 0;
              int bCounter = 0;
        
                // print boilerplate
                outputFile.print("#Produced by dk automaton generator\n#Sparse automaton for universality checking\n2\n#Sigma\n2\n#Size\n"+size+"\n\n#Initial:\n0\n");
              // print the transition matrices
              outputFile.println("#List of transitions follows, size = "+size+" rho-density = "+
                               rDensity + " number of transitions = " + numTransitions);
              for (int ii = 0; ii< size; ii++) {
                  for (int j=0; j<size; j++) {
                     if (A_transitionMatrix[ii][j]) {
                         outputFile.println(ii+" 0 "+j);
                         aCounter++;
                     }
                     if (B_transitionMatrix[ii][j]) {
                         outputFile.println(ii+" 1 "+j);
                         bCounter++;
                     }
                  }
              }
        
              //... and the final states array
              outputFile.println("-\n#Final state list follows, size = "+size+" f-density = "+ fDensity);
        
              for (int ii=0; ii<size; ii++) {
                  if (finalStatesArray[ii])
                     outputFile.print(ii+" ");
              }
              outputFile.println("\n#Number of a's: "+aCounter);
              outputFile.println("\n#Number of b's: "+bCounter);
              outputFile.close();
           }//try
           catch (IOException e) {
              System.out.println("Error creating file " + outputFileName + " in directory " + outputDir);
           }
       }//for
    }//makeAutomata
  
}
