package dk.extra;
import dk.extra.Utils;
import dk.brics.automaton.RandomAutomaton;
import dk.brics.automaton.Automaton;
import dk.brics.automaton.UniversalAutomaton;
import java.io.*;
import java.util.Calendar;
import dk.extra.Utils;

public class AutomataComplementor {
  /** A program to investigate the probability of an automaton
   * being the universal automaton.
   */
  private static String _logFile = ""; 
  public static void main(String args[]) {
    /** Main entry to the program */
    
    String logFile="";
    String pathToAutomata="";
    String pathToResults="";
    String idTag="";
    int rank=0;
    int numProcs=1;
    int minAlgorithm = -1;
    
    if (args.length < 5)
      System.err.println("Signature: "+
                         "-[lLogFile] -sSourceDir -dDestinationDir -a[Huffman | Hopcroft | Brzozowski] -rRank -pNumProcessors [-iIDtag]");
    
    
    else {
      boolean[] requiredParameters = new boolean[] {true, true, true, true, true};
      
      for (int i=0; i<args.length; i++) {
        char indicator = args[i].charAt(1);
        switch (indicator) {
          case 'l': logFile = args[i].substring(2); break;
          case 's': 
            pathToAutomata =  args[i].substring(2); 
            requiredParameters[0] = false; 
            break;
          case 'd': 
            pathToResults =  args[i].substring(2); 
            requiredParameters[1] = false; 
            break;
          case 'i': idTag = args[i].substring(2); break;
          case 'r': 
            rank = Integer.parseInt(args[i].substring(2)); 
            requiredParameters[2] = false; 
            break;
          case 'p': 
            numProcs = Integer.parseInt(args[i].substring(2)); 
            requiredParameters[3] = false; 
            break;
          case 'a':
            requiredParameters[4] = false; 
            String algo = args[i].substring(2).toLowerCase();
            if (algo.indexOf("huf")>-1)
              minAlgorithm = Automaton.MINIMIZE_HUFFMAN;
            else if (algo.indexOf("hop")>-1)
              minAlgorithm = Automaton.MINIMIZE_HOPCROFT;
            else if (algo.indexOf("br")>-1)
              minAlgorithm = Automaton.MINIMIZE_BRZOZOWSKI;
            else {
              System.err.println("Unknown algorithm: "+algo+" . I am exiting now!");
              requiredParameters[4] = true; 
              System.exit(6);            
            }
            break;
          default: System.err.println("Unknown parameter " + args[i] + " in AutomataComplementor!");
        }
      }//for
      System.out.println("Parsed parameters: \nlog: "+logFile+", \nsource: "+pathToAutomata+ ", \noutput: "+pathToResults +
                         " \nalgorithm: " + ((minAlgorithm==Automaton.MINIMIZE_HUFFMAN)? "Huffman" :
                                               (minAlgorithm==Automaton.MINIMIZE_HOPCROFT)? "Hopcroft" : 
                                               (minAlgorithm==Automaton.MINIMIZE_BRZOZOWSKI)? "Brzozowski" : "Unknown")
                           + ", \nidTag: -->"+idTag+"<--, \nrank = "+rank+", \nnumber of processors = "+numProcs );
      
      // check both source and output directories for existence, create if necessary
      File source = new File(pathToAutomata);
      File destination = new File(pathToResults);
      
      if (! source.exists()) {
        System.err.println("The source directory doesn't exist! I am exiting now!");
        System.exit (9);
      }
      
      if (rank==0) {
        if (! destination.exists()) {
          System.out.println("This is process 0. Destination directory doesn't exist. I attempting to create it now!");
          if ( destination.mkdirs())
            System.out.println("Destination directory successfully created!");
          else {
            System.err.println("Couldn't create output directory. Exiting");
            System.exit(10);
          }
        }
      }
      else if (rank > 0) {
        //All other processes: wait until the directory has been created!
        do {} while (! destination.exists());
      }
      _logFile = logFile; //need later, don't want to pass along.
      startSimulation(logFile, source, pathToResults, idTag, minAlgorithm, rank, numProcs);
    }//else
  }
  
  /*#############################################################################*/
  
  public static void startSimulation(String logFile, File dir, String pathToResults, 
                                     String idTag, int minAlgo, int rank, int numProcs) {
    /** Processes all of the files in the directory by calling singleAutomaton to handle each automaton */
    
    // This partitions the set of files for individual processors
    File[] simulationFiles = dir.listFiles(Utils.validNames(rank, numProcs));
    if (simulationFiles == null) {
      System.err.println("Source directory doesn't exist or is empty. Exiting simulation!");
      System.exit(1); 
    }
    for (int i=0; i<simulationFiles.length; i++) {
      // extract data
      boolean[][] transitionMatrix_A = extractTransitionMatrix(simulationFiles[i], "A_");
      boolean[][] transitionMatrix_B = extractTransitionMatrix(simulationFiles[i], "B_");
      
      boolean[] finalStatesArray = extractFinalArray(simulationFiles[i]);
      Automaton a = new RandomAutomaton(transitionMatrix_A, transitionMatrix_B, finalStatesArray, minAlgo);
      
      // create the output File object
      File f_pathToResults = new File(pathToResults);      
      File outFile = new File(f_pathToResults, simulationFiles[i].getName()+"-processed-"+idTag);
      analyzeSingleAutomaton(a, outFile);
    }//for
  }
  
  /*#############################################################################*/
  
  public static Automaton createFromFile(String inputFile) {
    /** For debugging purposes... creates the automaton using the 
     * specified file to extract the transition matrix and final states 
     * array. Assumes that the file contains this info ;-).
     */
    
    File in = new File (inputFile);
    boolean[][] transitionMatrix_A = extractTransitionMatrix(in, "A_");
    boolean[][] transitionMatrix_B = extractTransitionMatrix(in, "B_");
    
    boolean[] finalStatesArray = extractFinalArray(in);
    int minAlgo = Automaton.MINIMIZE_HOPCROFT;
    return new RandomAutomaton(transitionMatrix_A, transitionMatrix_B, finalStatesArray, minAlgo);
    
  }
  
  /*#############################################################################*/ 
  
  
  public static void analyzeSingleAutomaton(Automaton a, File outputFile) {
    /** Complements a single automaton, and writes the data to 
     * the output file
     */
    
    //save some time when running on partially processed data
    if (outputFile.exists() && outputFile.length() > 0)
      return;
    
      
      boolean totalLanguage = false;
      //start the clock
      int sizeBeforeMinimize = a.getNumberOfStates();

      long startMinimizing = System.currentTimeMillis();
      
      a.minimize(); // the method determinizes when necessary (Hopcroft and Huffman)
      // stop the clock for minimizing
      long minimizingDone = System.currentTimeMillis();
      a = a.complement();
      // stop the clock for complementing
      long complementingDone = System.currentTimeMillis();
      a = a.intersection(UniversalAutomaton.Singleton);
      if (a.isEmpty()) {
	  totalLanguage=true;
      }
      // stop the clock for reachability check
      long reachabilityCheckDone = System.currentTimeMillis();
      
      // prepare the data to be written
      String toWrite = "";
      toWrite+="Total: "+totalLanguage + '\n';
      toWrite+="Minimize Time: "+(minimizingDone -  startMinimizing)+ '\n';
      toWrite+="Complement Time: "+(reachabilityCheckDone - startMinimizing)+ '\n';
      toWrite+="Size after minimize: "+sizeBeforeMinimize+'\n';
      toWrite+="Reachability Check Time: "+ (reachabilityCheckDone - complementingDone)+ '\n';
      toWrite+="Resulting Rho Density: " + ( (double) a.getNumberOfTransitions() / a.getNumberOfStates() ) + '\n';
      toWrite+="Resulting States: "  + a.getNumberOfStates() + '\n';
      toWrite+="Resulting Transitions: " +  a.getNumberOfTransitions() + '\n';
      
      // stop the clock for information gathering and processing
      long infoProcessingDone = System.currentTimeMillis();
      toWrite+="Info gathered and processed: " +  (infoProcessingDone - reachabilityCheckDone) + '\n';
//      p.println(""+outputFile.getName().toString()+" finished");
//      p.close();
      // Create the file and write the data
      try {
        PrintWriter out = new PrintWriter(new FileWriter(outputFile));
        out.println(toWrite);
        out.close();
      }
      catch (IOException e) {
        System.err.println("Error opening the file " + outputFile.toString());
      }
      
      
//    }
//    catch (IOException e) {
//      System.err.println(e.toString());
//    }
  }
  
  /*#############################################################################*/
  
  public static boolean[][] extractTransitionMatrix(File filename, String id) {
    
    try {
      boolean[][] transitionMatrix;
      
      BufferedReader inFile = new BufferedReader(new FileReader(filename));
      String currentLine="";
      
      do {
        currentLine = inFile.readLine();
      } while (currentLine != null && currentLine.indexOf(id+"Transition matrix follows") == -1);
      
      if (currentLine==null) {
        System.err.println("There is no "+id +"Transition matrix in this file "+filename.toString()+". Exiting!");
        System.exit(2);
      }
      
      // Now follows the transition matrix. Count the total number of 0's and 1's
      currentLine = inFile.readLine();
      int charCounter = 0;
      for (int i=0; i<currentLine.length(); i++) {
        char currentChar = currentLine.charAt(i);
        if (currentChar=='0' || currentChar=='1')
          charCounter++;
      }
      
      //This gives us the size of the square matrix. Create it and fill it up
      transitionMatrix = new boolean[charCounter][charCounter] ;
      for (int i=0; i<charCounter; i++) {
        for (int j=0; j<charCounter; j++) {
          transitionMatrix[i][j] = ( currentLine.charAt(j)=='1' );
        }
        currentLine = inFile.readLine();
      }
      // make sure that there is nothing left after the matrix
      //currentLine = inFile.readLine();
      
      char nextChar = currentLine.charAt(0);//must be '#' for the final states comment, or some whitespace character.
      if (nextChar=='0' || nextChar=='1')
        System.out.println("In this file:\n"+filename.getName() + "\n the matrix was followed by a matrix element. The offending line:\n"+currentLine+"\nInvestigate!");
      
      inFile.close(); //be nice
      
      return transitionMatrix;
    }//try
    catch (IOException e) {
      System.err.println("IOException in extractTransitionMatrix()!"+e);
      System.exit(2);
    }
    
    return null;
  }
  
  /*#############################################################################*/
  
  public static boolean[] extractFinalArray(File filename) {
    boolean[] finalArray = null;
    try {
      BufferedReader inFile = new BufferedReader(new FileReader(filename));
      String currentLine="";
      
      do {
        currentLine = inFile.readLine();
      } while (currentLine != null && currentLine.indexOf("Final states array") == -1);
      
      if (currentLine==null) {
        System.err.println("There is no array of final states in this file "+filename.toString()+". Exiting!");
        System.exit(3);
      }
      
      // Now follows the array. Count the total number of 0's, a's and b's
      currentLine = inFile.readLine();
      int charCounter = 0;
      for (int i=0; i<currentLine.length(); i++) {
        char currentChar = currentLine.charAt(i);
        if (currentChar=='1' || currentChar=='0')
          charCounter++;
      }
      
      // We know the size. Create the array and fill up the data
      finalArray = new boolean[charCounter];
      for (int i=0; i<currentLine.length(); i++)  {
        finalArray[i] = (currentLine.charAt(i)=='1');
      }
      inFile.close();
    }//try
    catch (IOException e) {
      System.out.println("Exception in extractFinalArray: \n"+e.toString());
    }
    
    return finalArray;    
  }
  
  /*#############################################################################*/
  
  
  
  
}
