package dk.extra;
import dk.brics.automaton.*;
import java.util.*;
import java.io.*;
import java.text.DecimalFormat;
import java.util.regex.*;

public class SpinParser extends ADataParser {
    
    /* Make these arrays visible everywhere, to avoid passing them
     * down as arguments*/
    MeasuredTime[] a_complementTime;
    int[] a_depthReached;
    int[] a_universal; // Only needed for the raw data dump. We normally dump the total count.
    
    
    int automataPerGroup;
    int numUniversal = 0;
    int numFinishedComplementations = 0; // Those that didn't time out
    PrintWriter[] af_complementTime = new PrintWriter[4]; // user, system, total, tool
    PrintWriter[] af_ave_complementTime = new PrintWriter[4]; //ditto, but for the averages
    PrintWriter[] af_std_complementTime = new PrintWriter[4]; //standard deviation
    PrintWriter f_universal;
    PrintWriter f_rawData;
    PrintWriter f_maxDepth; // Keep track of the maximum depth reached
    PrintWriter f_nontimeout; // number of automata that didn't time out
    
    
    
    /** 
     * The main entry into the program. Immediately transfer control
     * to the parsing driver, and let him handle the rest of the
     * operation calling the appropriate methods from here.
     */
    public static void main(String[] args) {
	ParsingDriver driver = new ParsingDriver(new SpinParser());
	driver.theMain(args);
    }
    



    /**
     * Print an id to standard output
     */
    public void printMyName() {
	System.out.println("This is the Spin Parser!");
    }
    

    
    
    /**
     * We are using PrintWriter arrays for each major step of the
     * algorithm, and each array index corresponds to some default
     * file name. Here we create the mapping so that we can streamline
     * the creation of the new files.
     */
    String fileIndex2Str(int index) {
	switch(index) {
	case 0: return "user";
	case 1: return "system";
	case 2: return "total";
	case 3: return "spin";
	default: return "wrong_index_value";
	}
    }

    

    /** 
     * Gives actual values to the (globally declared) output files
     * based on the prefix/affix string, and the output directory).
     */
    public PrintWriter[] defineOutputFiles(File output, String prefix, String affix) {
	LinkedList allFiles = new LinkedList();
	try {

	    // Medians
	    for (int i=0; i<4; i++) {
		File comp = new File(output, prefix + fileIndex2Str(i) + "_compl_t" + affix);
		af_complementTime[i] = new PrintWriter(new FileWriter(comp));
		
		allFiles.add(af_complementTime[i]);
	    }

	    // Means
	    for (int i=0; i<4; i++) {
		File ave = new File(output, prefix + fileIndex2Str(i) + "_ave_compl_t" + affix);
		af_ave_complementTime[i] = new PrintWriter(new FileWriter(ave));
		
		allFiles.add(af_ave_complementTime[i]);
	    }

	    // Standard Deviation
	    for (int i=0; i<4; i++) {
		File std = new File(output, prefix + fileIndex2Str(i) + "_std_compl_t" + affix);
		af_std_complementTime[i] = new PrintWriter(new FileWriter(std));
		
		allFiles.add(af_std_complementTime[i]);
	    }
	    
	    
	    File uni = new File(output, prefix + "universal" + affix);
	    f_universal = new PrintWriter(new FileWriter(uni));
	    allFiles.add(f_universal);

	    File raw = new File(output, prefix + "RAW" + affix);
	    f_rawData = new PrintWriter(new FileWriter(raw));
	    allFiles.add(f_rawData);

	    File depth = new File(output, prefix + "max_depth" + affix);
	    f_maxDepth = new PrintWriter(new FileWriter(depth));
	    allFiles.add(f_maxDepth);


	    File nontimeout = new File(output, prefix + "nontimeout" + affix);
	    f_nontimeout = new PrintWriter(new FileWriter(nontimeout));
	    allFiles.add(f_nontimeout);


	}
	catch(IOException e) {
	    System.out.println("Error creating the output files "+e);
	    System.exit(2223);
	}
	PrintWriter[] toReturn = new PrintWriter[allFiles.size()];
	
	return (PrintWriter[]) allFiles.toArray(toReturn);
    }
    
    
    
    
    /** 
     * Writes down a data point for each measurable to the output
     * files. Remember that a_complementTime is globally visible. Its
     * contents are filled-in elsewhere (actually below).
     */
    void printDataToFiles() {
	
	// Data format for the data that goes into the files
	DecimalFormat form = new DecimalFormat("0.000");
	
	
	MeasuredTime med_complementTime = Utils.getMedian(a_complementTime, automataPerGroup); 
	af_complementTime[0].print(" " + form.format(med_complementTime.getUserTime()));
	af_complementTime[1].print(" " + form.format(med_complementTime.getSystemTime()));
	af_complementTime[2].print(" " + form.format(med_complementTime.getTotalTime()));
	af_complementTime[3].print(" " + form.format(med_complementTime.getToolTime()));

	MeasuredTime ave_complementTime = Utils.getMean(a_complementTime, automataPerGroup);
	af_ave_complementTime[0].print(" " + form.format(ave_complementTime.getUserTime()));
	af_ave_complementTime[1].print(" " + form.format(ave_complementTime.getSystemTime()));
	af_ave_complementTime[2].print(" " + form.format(ave_complementTime.getTotalTime()));
	af_ave_complementTime[3].print(" " + form.format(ave_complementTime.getToolTime()));
	
	MeasuredTime std_complementTime = Utils.getMean(a_complementTime, automataPerGroup);
	af_std_complementTime[0].print(" " + form.format(std_complementTime.getUserTime()));
	af_std_complementTime[1].print(" " + form.format(std_complementTime.getSystemTime()));
	af_std_complementTime[2].print(" " + form.format(std_complementTime.getTotalTime()));
	af_std_complementTime[3].print(" " + form.format(std_complementTime.getToolTime()));

	f_universal.print(" " + numUniversal + "/" + numFinishedComplementations);
	f_maxDepth.print(" " + Utils.getMax(a_depthReached, automataPerGroup));
	f_nontimeout.print(" " + numFinishedComplementations + "/" + automataPerGroup);
	
	System.out.println("Timeouts: " + (automataPerGroup - numFinishedComplementations));
	
	// In addition to the usual statistics, here we also add some raw data
	f_rawData.print("User time: \n[");
	for (int i = 0; i<automataPerGroup; i++)
	    f_rawData.print(""+a_complementTime[i].getUserTime() + ((i==automataPerGroup-1)?"]\n":", "));
	
	f_rawData.print("System time: \n[");
	for (int i = 0; i<automataPerGroup; i++)
	    f_rawData.print(""+a_complementTime[i].getSystemTime() + ((i==automataPerGroup-1)?"]\n":", "));

	f_rawData.print("Total  time: \n[");
	for (int i = 0; i<automataPerGroup; i++)
	    f_rawData.print(""+a_complementTime[i].getTotalTime() + ((i==automataPerGroup-1)?"]\n":", "));

	f_rawData.print("Depth reached: \n[");
	for (int i = 0; i<automataPerGroup; i++)
	    f_rawData.print(""+a_depthReached[i] + ((i==automataPerGroup-1)?"]\n":", "));

	f_rawData.print("Universality: \n[");
	for (int i = 0; i<automataPerGroup; i++)
	    f_rawData.print("" + a_universal[i] + ((i==automataPerGroup-1)?"]\n":", "));

	f_rawData.print("Non-timed out: \n");
	f_rawData.print("" + numFinishedComplementations + "/" + automataPerGroup + "\n");
    }
  



  
  
  
  
    /**
     * Parses each file in the group and extract the appropriate data
     * to the arrays. The array @param groupFiles contains the files
     * that match the r,f,s criteria of the group. It is not necessary
     * that the size of @param groupFiles matches the number that is
     * encoded in the file name (this refers to the Y in
     * blah--X-of-Y.ext in the file name). This distinction is
     * important because later we will calculate the medians based on
     * the size of @param groupFiles, not on the number encoded in the
     * file name..
     */
    void parseIntoArrays(File[] groupFiles) {
	
	/* SETUP */
	int numAutomata = groupFiles.length; //previously checked to match numAutomata from the file name
	automataPerGroup = numAutomata; // Starting count, will decrease if necessary
	numFinishedComplementations = numAutomata;  //This is a global constant
	
	// automataPerGroup are those which have been through gatorade before the time ran out
	// numFinishedComplementations are those which didn't time out (for the universality count)
	
	System.out.println("Parsing "+groupFiles.length + " finished automata into arrays");
	

	// Create the arrays based on the maximum number of data points that we expect (from the file name)	
	a_complementTime = new MeasuredTime[automataPerGroup]; // user, system, total, tool
	a_universal = new int[automataPerGroup];
	a_depthReached = new int[automataPerGroup];
	
	numUniversal = 0;
	

	/* WORK */
	for (int i=0; i<numAutomata; i++)  {
      
	    /* I am not sure if this can ever happen.*/
	    if (groupFiles[i] == null) {
		automataPerGroup = i-1;
		System.err.println("Note that the number of automata in this group is " +
				   automataPerGroup + 
				   " while expected number was " + numAutomata);
		break;
	    }
      
	    SpinFileParser parser = makeParser(groupFiles[i]);
	    int status = parser.parse();
      	    
	    if (status < 0) {
		System.err.println("There was an error while parsing file " + 
				   groupFiles[i].getName());
		System.err.println("The error code returned by the parser is " + status);
		System.err.println("The error stream is:\n" + parser.getErrorStream());
		System.exit(12321);
	    }
	    
	    else { // No error; status == 0
		a_complementTime[i] = parser.getComplementTime();
    
		if (parser.timeout()) {
		    numFinishedComplementations --;
		    a_universal[i] = -1;
		}
		else {
		    if (parser.isUniversal()) {
			numUniversal++;
			a_universal[i] = 1; // Need this for the raw data dump
		    }
		    else 
			a_universal[i] = 0;
		}

		a_depthReached[i] = parser.getDepthReached(); // Again, only for the raw data
	    }
	    
	} //for all automata in this group

	// In order to annotate the raw data file, we dump a filename
	// at the time we parse the whole group. This is because later
	// on we have only the arrays with the data, but not the r, f,
	// s, parameters that the data corresponds to. Conveniently,
	// these three parameters are in the file name (any file name
	// from this group).
	f_rawData.println(groupFiles[0].getName());
    
    }//parseIntoArrays
    




    /**
     * Returns an object that can parse Spin output files
     */
    SpinFileParser makeParser(File file) {
	return new SpinFileParser(file);
    }
    
}







/**
 * An object that knows how to parse the files that are the result of
 * Spin's universality check. All data is stored in private fields and
 * can be queried only after parse() has been called to fill in the
 * field.
 */
class SpinFileParser implements IFileParser {
    private File _file;
    private MeasuredTime _complement = new MeasuredTime();
    private MeasuredTime _emptiness = new MeasuredTime();
    private boolean _universal;
    private String _errorStream = "";
    private boolean _universalSet = false;
    private boolean _timeout = false;
    private int _depthReached = -1;
    private boolean _time_parsed = false;
    private boolean _file_done = false;
    private boolean _allow_universality_exception = false;

    /**
     * The constructor
     */
    public SpinFileParser(File f) {
	_file = f;
    }





    /**
     * Deprecated methods, which are still in the IFileParser
     * interface for backwards compatibility reasons. They should not
     * be used, and probably will crash any program that tries to use
     * them.
     */
    public double getUserTime() {return -1;}
		boolean time_parsed = false;    public double getSystemTime() {return -1;}

    public int getNodes() {return -1;}
    public double getHeuristicTime() {return -1;}
    
    
    
    /**
     * Obvious
     */
    public boolean isUniversal() {
	if (_universalSet)
	    return _universal;
	else if (! _allow_universality_exception) {
	    // We should never be able to reach this
	    throw new AssertionError("The status of the automaton was never set, but you should have checked the error code returned by parse() and know this already!");
	}
	
	else {
	    System.err.println("Exception allowed!");
	    System.err.println("Universality data non accurate!");
	    return false;
	}
    }
    



    /**
     * Obvious
     */
    public boolean timeout() {
	return _timeout;
    }
    
    
    /**
     * Obvious: parses the file and stores information about it.
     */
    public int parse() {
	
	int errorCode = 0;
	try {
	    BufferedReader in = new BufferedReader ( new FileReader (_file));
	    String s;
	    

	    // Read the file line by line and look for stuff that we care about
	    while (!_file_done && (s=in.readLine()) != null) {
		if (s==null)
		    throw new AssertionError("This is impossible!");
		
		/*************************/
		/* Checking for sentinel */
		/*************************/
		Pattern sentinel = Pattern.compile("cambridgedancersclub");
		Matcher ma = sentinel.matcher(s);
		if (ma.matches())
		    _file_done = true;


		/*************************/
		/* Checking universality */
		/*************************/
		Pattern spinACLine = Pattern.compile("pan: assertion violated .+ \\(at depth \\d+\\)");
		Matcher m = spinACLine.matcher(s);
		if (m.matches()) {
		    /* If the assertion has been violated, the
		     automaton is not universal (we are asserting that
		     at any point at least one of the accepting states
		     is high, like in the smv model. */
		    Utils.affirm(_universalSet == false, "Line 291 in SpinParser");
		    _universal = false;
		    _universalSet = true;
		}


		Pattern spinACLine2 = Pattern.compile("pan: acceptance cycle \\(at depth \\d+\\)"); 
		
		m = spinACLine2.matcher(s);
		if (m.matches()) {
		    /* If there is an accepting cycle, then the
		     * complementary automaton is not empty, so the
		     * starting automaton was not universal to begin
		     * with. */
		    Utils.affirm(_universalSet == false, "Line 393 in SpinParser " + _file.toString());
		    _universal = false;
		    _universalSet = true;
		}
		


		/* If there is no assertion violation/acceptance
		 cycle, then spin wouldn't say anything, and in this
		 case it will report 0 errors.*/
		Pattern spinStatusLine = Pattern.compile("State-vector (\\d+) byte, depth reached (\\d+), errors: (\\d)");
		m = spinStatusLine.matcher(s);
		if (m.matches()) {
		    // Group 1 --> State vector size
		    // Group 2 --> Depth reached
		    // Group 3 --> Errors
		    
		    /* Need this to make sure we are not getting false
		     negatives because of hitting the depth limit. If
		     it so happens, there is a way of increasing the
		     depth by a command line switch to pan.*/
		     _depthReached = Integer.parseInt(m.group(2));
		    
		    if (m.group(3).equals("0")) {
			Utils.affirm(_universalSet == false, "Line 303 in SpinParser");
			_universal = true;
			_universalSet = true;
		    }
		    // This is subtle. If Spin indicates an error,
		    // then either there was an assertion violation,
		    // or something else went wrong. Usually we expect
		    // a single error to correspond to an assertion
		    // violation, and this should be consistent with
		    // above.
		    else if (! _universalSet  || _universal) {
			_errorStream = _errorStream.concat("Spin error: Number of errors is not zero, but at this point in the file we haven't matched \"assertion violation\" or \"accepting cycle\"!\n");
			_errorStream = _errorStream.concat("Spin error: Status line is: ").concat(m.group(0));
			errorCode -= 10;
		    }
		}
		
		// In case of timeout, the universality result is moot
		Pattern timeout_p = Pattern.compile("toobadIhadtobailout");
		m = timeout_p.matcher(s);
		
		if (m.matches())
		    _timeout = true;
		

		Pattern timeout2_p = Pattern.compile("alarm_handler: timeout detected!");
		m = timeout2_p.matcher(s);
		
		if (m.matches()) {
		    _timeout = true;
		    _file_done = true;
		}
		


		/**********************************/
		/*   Checking return codes        */
		/**********************************/
				
		/* Checking the various retun codes from the compilation of the model */
		Pattern spinStatus = Pattern.compile("Compiling pan.c from spin returned exit code (\\d+)");
		m = spinStatus.matcher(s);
		
		// If everything is OK, Spin->Pan.c should return 0
		if (m.matches() && (! m.group(1).equals("0"))) {
		    
		    _errorStream = _errorStream.concat("Spin to Pan.c compilation error: the return status was ");
		    _errorStream = _errorStream.concat(m.group(1));
		    _errorStream = _errorStream.concat("\n");
		    errorCode -= 100;
		}

		// If everything is OK, pan.c->pan* should also return 0
		spinStatus = Pattern.compile("Compiling pan with gcc returned exit code (\\d+)");
		m = spinStatus.matcher(s);
		if (m.matches() && (! m.group(1).equals("0"))) {
		    
		    _errorStream = _errorStream.concat("pan.c to pan* compilation error: the return status was ");
		    _errorStream = _errorStream.concat(m.group(1));
		    _errorStream = _errorStream.concat("\n");
		    errorCode -= 1000;
		}

		// If everything is OK, pan* itself should return 0
		spinStatus = Pattern.compile("Pan returned status: (\\d+)");
		m = spinStatus.matcher(s);
		if (m.matches() && (! m.group(1).equals("0"))) {
		    
		    _errorStream = _errorStream.concat("pan* execution error: the return status was ");
		    _errorStream = _errorStream.concat(m.group(1));
		    _errorStream = _errorStream.concat("\n");
		    errorCode -= 10000;
		}
		

		/****************************/
		/*    Parsing the times     */
		/****************************/

		
		/* Measured complementation times (system + user) */
		Pattern unicheckUser = Pattern.compile("Child: Universality check: Measured User time: (\\d+\\.\\d+)");
		m = unicheckUser.matcher(s);
		if (m.matches()) {
		    _time_parsed = true;
		    _complement.setUserTime((new Double(m.group(1))).doubleValue());
		}


		Pattern unicheckUser2 = Pattern.compile("Parent: Universality check: Measured User time: (\\d+\\.\\d+)");
		m = unicheckUser2.matcher(s);
		if (m.matches()) {
		    _complement.setUserTime((new Double(m.group(1))).doubleValue());
		    _timeout = true;
		    _time_parsed = true;
		}

		Pattern unicheckSystem = Pattern.compile("Child: Universality check: Measured System time: (\\d+\\.\\d+)");
		m = unicheckSystem.matcher(s);
		if (m.matches()) {
		    _time_parsed = true;
		    _complement.setUserTime((new Double(m.group(1))).doubleValue());	
		}

		Pattern unicheckSystem2 = Pattern.compile("Parent: Universality check: Measured System time: (\\d+\\.\\d+)");
		m = unicheckSystem2.matcher(s);
		if (m.matches()) {
		    _timeout = true;
		    _time_parsed = true;
		    _complement.setSystemTime((new Double(m.group(1))).doubleValue());	
		}
		
		/* currently spin doesn't allow to measure its runnng
		 * time internally, so we leave it to default, which
		 * is 0 */
		
	    } //while
      
	    in.close();
	} //try
	catch (IOException e) {
	    System.err.println("Error opening the file " + _file + e);
	    System.exit(1234);
	}
	
	// Make sure that we have parsed the most important stuff
	if (! _universalSet) {

	    if (! _time_parsed) {
		// Must be a timeout
		System.err.println("Universality could not be determined, nor timing information was parsed!");
		System.err.println("Probably a timeout. Setting the timers accordingly, but check the file!");
		System.err.println(_file.toString());
		
		_timeout = true;
		_complement.setSystemTime(3600);
		_complement.setUserTime(3600);
	    }

	    else {		
		_errorStream = _errorStream.concat("The universality of the automaton could not be determined!\n");
		if (! _timeout) {
		    _allow_universality_exception = true;
		    for (int i=0; i<4; i++)
			System.err.println("Error!!!");

		    System.err.println(_errorStream);
		    //errorCode -= 100000;
		}
	    }
	}

	return errorCode;
    }

    
    public boolean getTimeout() {
	return _timeout;
    }


    public String getErrorStream() {
	return _errorStream;
    }

    public MeasuredTime getComplementTime() {
	return _complement;
    }
    
    public MeasuredTime getEmptynessCheck() {
	return _emptiness;
    }


    public int getDepthReached() {
	return _depthReached;
    }
}
