package dk.extra;
import dk.brics.automaton.*;
import java.util.*;
import java.io.*;
import java.text.DecimalFormat;
import java.util.regex.*;

public class WringParser extends ADataParser {
    
    /* Make these arrays visible everywhere, to avoid passing them
     * down as arguments*/
    MeasuredTime[] a_complementTime;
    MeasuredTime[] a_emptinessCheck;
    int[] a_complementSize;
    
    int automataPerGroup;
    int numUniversal = 0;
    int numFinishedComplementations = 0; // Those that didn't time out
    PrintWriter[] af_complementTime = new PrintWriter[4]; // user, system, total, tool
    PrintWriter[] af_emptinessCheck = new PrintWriter[4]; // ditto
    PrintWriter f_universal;
    PrintWriter f_complementSize, f_aveComplementSize, f_nontimeout;
    
    public static final int WRING_TIMEOUT = 39424;
    public static final double EPSILON = 0.0001;


    
    /** 
     * The main entry into the program. Immediately transfer control
     * to the parsing driver, and let him handle the rest of the
     * operation calling the appropriate methods from here.
     */
    public static void main(String[] args) {
	ParsingDriver driver = new ParsingDriver(new WringParser());
	driver.theMain(args);
    }
    



    /**
     * Print an id to standard output
     */
    public void printMyName() {
	System.out.println("This is the Wring Parser!");
    }
    

    
    
    /**
     * We are using PrintWriter arrays for each major step of the
     * algorithm, and each array index corresponds to some default
     * file name. Here we create the mapping so that we can streamline
     * the creation of the new files.
     */
    String fileIndex2Str(int index) {
	switch(index) {
	case 0: return "user";
	case 1: return "system";
	case 2: return "total";
	case 3: return "wring";
	default: return "wrong_index_value";
	}
    }

    

    /** 
     * Gives actual values to the (globally declared) output files
     * based on the prefix/affix string, and the output directory).
     */ 
    public PrintWriter[] defineOutputFiles(File output, String prefix, String affix) {
	LinkedList allFiles = new LinkedList();
	try {
	    for (int i=0; i<4; i++) {
		File comp = new File(output, prefix + fileIndex2Str(i) + "_compl_t_" + affix);
		af_complementTime[i] = new PrintWriter(new FileWriter(comp));
		File empt = new File(output, prefix + fileIndex2Str(i) + "_empt_ck_" + affix);
		af_emptinessCheck[i] = new PrintWriter(new FileWriter(empt));

		allFiles.add(af_complementTime[i]);
		allFiles.add(af_emptinessCheck[i]);
	    }
	    
	    File uni = new File(output, prefix + "universal" + affix);
	    f_universal = new PrintWriter(new FileWriter(uni));
	    allFiles.add(f_universal);

	    File size = new File(output, prefix + "compl_size" + affix);
	    f_complementSize = new PrintWriter(new FileWriter(size));
	    allFiles.add(f_complementSize);
	    
	    File ave_size = new File(output, prefix + "ave_compl_size" + affix);
	    f_aveComplementSize = new PrintWriter(new FileWriter(ave_size));
	    allFiles.add(f_aveComplementSize);

	    File nontimeout = new File(output, prefix + "nontimeout" + affix);
	    f_nontimeout = new PrintWriter(new FileWriter(nontimeout));
	    allFiles.add(f_nontimeout);
	    

	    
	}
	catch(IOException e) {
	    System.out.println("Error creating the output files "+e);
	    System.exit(2223);
	}
	PrintWriter[] toReturn = new PrintWriter[allFiles.size()];
	
	return (PrintWriter[]) allFiles.toArray(toReturn);
    }
    
    
    
    
    /** 
     * Writes down a data point for each measurable to the output
     * files.
     */
    void printDataToFiles() {
	
	// Data format for the data that goes into the files
	DecimalFormat form = new DecimalFormat("0.000");
	
	
	MeasuredTime complementTime = Utils.getMedian(a_complementTime, automataPerGroup); 
	af_complementTime[0].print(" " + form.format(complementTime.getUserTime()));
	af_complementTime[1].print(" " + form.format(complementTime.getSystemTime()));
	af_complementTime[2].print(" " + form.format(complementTime.getTotalTime()));
	af_complementTime[3].print(" " + form.format(complementTime.getToolTime()));

	MeasuredTime emptinessCheck = Utils.getMedian(a_emptinessCheck, automataPerGroup);
	af_emptinessCheck[0].print(" " + form.format(emptinessCheck.getUserTime()));
	af_emptinessCheck[1].print(" " + form.format(emptinessCheck.getSystemTime()));
	af_emptinessCheck[2].print(" " + form.format(emptinessCheck.getTotalTime()));
	af_emptinessCheck[3].print(" " + form.format(emptinessCheck.getToolTime()));
	
	f_universal.print(" " + numUniversal + "/" + numFinishedComplementations);
	f_complementSize.print(" " + Utils.getMedian(a_complementSize, automataPerGroup));
	f_aveComplementSize.print(" " + Utils.getMean(a_complementSize, automataPerGroup));
	f_nontimeout.print(" " + numFinishedComplementations + "/" + automataPerGroup);
    }
  



  
  
  
  
    /**
     * Parse each file in the group and extract the appropriate data
     * to the arrays. The array @groupFiles contains the files that
     * match the r,f,s criteria of the group. It is not necessary that
     * the number of files in the group matches the number that is
     * encoded in the file name (this refers to the Y in
     * blah--X-of-Y.ext in the file name). This is important because
     * later we will calculate the medians based on this number.
     */
    void parseIntoArrays(File[] groupFiles) {
	
	/**/
	int numAutomata = groupFiles.length; //previously checked to match numAutomata from the file name
	automataPerGroup = numAutomata;
	numFinishedComplementations = numAutomata; 
	
	System.out.println("Parsing "+groupFiles.length + " automata into arrays");
	
	a_complementTime = new MeasuredTime[automataPerGroup]; // user, system, total, tool
	a_emptinessCheck = new MeasuredTime[automataPerGroup]; // same
	a_complementSize = new int[automataPerGroup];
	numUniversal = 0;
	
	for (int i=0; i<numAutomata; i++)  {
      
	    /* I am not sure if this can ever happen.*/
	    if (groupFiles[i] == null) {
		automataPerGroup = i-1;
		System.err.println("Note that the number of automata in this group is " +
				   automataPerGroup + 
				   " while expected number was " + numAutomata);
		break;
	    }
      
	    WringFileParser parser = makeParser(groupFiles[i]);
	    int status = parser.parse();
      	    
	    if (status < 0) {
		// There was an error while parsing
		if (parser.getTimeout()) {
		    // Handle timeouts internally
		    a_complementTime[i] = MeasuredTimeTimeout.Singleton;
		    a_emptinessCheck[i] = MeasuredTimeTimeout.Singleton;
		    numFinishedComplementations --;
		}
		else {
		    System.err.println("There was an error while parsing file " + 
				       groupFiles[i] + "\n");
		    System.err.println("The error code returned by the parser is " + status);
		    System.err.println("The error stream is:\n" + parser.getErrorStream());
		    System.exit(12321);
		}
	    }
	    
	    else { // No error; status == 0
		a_complementTime[i] = parser.getComplementTime();
		a_emptinessCheck[i] = parser.getEmptynessCheck();
	    
	    
		if (parser.isUniversal()) {
		    numUniversal++;
		}
		a_complementSize[i] = parser.getComplementSize();
	    }
	    
	} //for all automata in this group
    
    }//parseIntoArrays
    
    /**
     * Returns an object that can parse Wring output files
     */
    WringFileParser makeParser(File file) {
	return new WringFileParser(file);
    }
    
}







/**
 * An object that knows how to parse the files that are the result of
 * Wring's complementation and Spin's emptiness check. All data is
 * stored in private fields and can be queried only after parse() has
 * been called to fill in the field.
 */
class WringFileParser implements IFileParser {
    private File _file;
    private MeasuredTime _complement = new MeasuredTime();
    private MeasuredTime _emptiness = new MeasuredTime();
    private boolean _universal;
    private String _errorStream = "";
    private boolean _universalSet = false;
    private boolean _timeout = false;
    private int _complementSize = 0;
    
    /**
     * The constructor
     */
    public WringFileParser(File f) {
	_file = f;
    }

    /**
     * Deprecated methods, which are still in the IFileParser
     * interface for back compatibility reasons. They should not be
     * used, and hopefully will crash any program that tries to use
     * them.
     */
    public double getUserTime() {return -1;}
    public double getSystemTime() {return -1;}

    public int getNodes() {return -1;}
    public double getHeuristicTime() {return -1;}
    
    
    
    public int getComplementSize() {
	return _complementSize;
    }


    /**
     * Obvious
     */
    public boolean isUniversal() {
	if (_universalSet)
	    return _universal;
	else {
	    // We should never be able to reach this
	    throw new AssertionError("The status of the automaton was never set, but you should have checked the error code returned by parse() and know this already!");
	}
    }
    

    /**
     * Obvious
     */
    public boolean timeout() {
	return _timeout;
    }
    
    
    /**
     * Obvious: parses the file and stores information about it.
     */
    public int parse() {
	
	int errorCode = 0;
	try {
	    BufferedReader in = new BufferedReader ( new FileReader (_file));
	    String s;
	    

	    // Read the file line by line and look for stuff that we care about
	    while ( (s=in.readLine()) != null) {
		if (s==null)
		    throw new AssertionError("This is impossible!");
		
		/**
		 * Checking the size of the automaton
		 */
		Pattern size_p = Pattern.compile("Number of states in complemented: (\\d+)");
		Matcher m = size_p.matcher(s);
		if (m.matches()) {
		    _complementSize = (new Integer(m.group(1))).intValue();
		    //System.out.println(s + "---> " + _complementSize);
		}




		/*************************/
		/* Checking universality */
		/*************************/
		Pattern spinACLine = Pattern.compile("\\w+: acceptance cycle \\(at depth (\\d+)\\)");
		m = spinACLine.matcher(s);
		if (m.matches()) {
		    /* There is an accepting cycle, therefore the
		     * automaton has an accepting cycle. The
		     * complement of the automaton is not
		     * universal. */
		    Utils.affirm(_universalSet == false, "_universal == null in WringParser");
		    _universal = false;
		    _universalSet = true;
		}
		
		Pattern spinStatusLine = Pattern.compile("State-vector (\\d+) byte, depth reached (\\d+), errors: (\\d+)");
		m = spinStatusLine.matcher(s);
		if (m.matches()) {
		    // Group 1 --> State vector size
		    // Group 2 --> Depth reached
		    // Group 3 --> Errors
		    if (m.group(3).equals("0")) {
			Utils.affirm(_universalSet == false, "_universal == null in WringParser");
			_universal = true;
			_universalSet = true;
		    }
		    // This is subtle. If Spin indicates an error, then either there is an accepting cycle,
		    // or something went wrong. If we have parsed already "accepting cycle", then it is 
		    // the former. In all other cases we complain bitterly.
		    else if (! _universalSet  || _universal) {
			_errorStream = _errorStream.concat("Spin error: Number of errors is not zero.\n");
			_errorStream = _errorStream.concat("Spin error: Status line is: ").concat(m.group(0));
			errorCode -= 10;
		    }
		}
		
		//System.out.println("Parsing the tool time");
		
		/* Tool Time */
		Pattern wringTime = Pattern.compile("CPU time: (\\d+\\.\\d+)");
		m = wringTime.matcher(s);
		if (m.matches()) {
		    _complement.setToolTime((new Double(m.group(1))).doubleValue());
		}
		
		//System.out.println("Tool time parsed, parsing the wring status");

		/* Wring status and timeout*/
		Pattern wringStatus = Pattern.compile("Wring returned status: (\\d+)");
		m = wringStatus.matcher(s);
		
		// If everything is OK, Wring returns 0
		if (m.matches() && (! m.group(1).equals("0"))) {
		    
		    // A timeout is a type of error that we expect to
		    // happen from time to time.
		    if (m.group(1).equals(""+WringParser.WRING_TIMEOUT)) {
			_timeout = true;
			in.close();
			return -1;
		    }
		    else {
			_errorStream = _errorStream.concat("Wring error: the return status of wring was ");
			_errorStream = _errorStream.concat(m.group(1));
			_errorStream = _errorStream.concat("\n");
			errorCode -= 100;
		    }
		}

		/* Measured complement time (user + system)*/
		Pattern complementUser = Pattern.compile("Complement: Measured User time: (\\d+\\.\\d+)");
		m = complementUser.matcher(s);
		if (m.matches()) {
		    _complement.setUserTime((new Double(m.group(1))).doubleValue());
		}

		Pattern complementSystem = Pattern.compile("Complement: Measured System time: (\\d+\\.\\d+)");
		m = complementSystem.matcher(s);
		if (m.matches()) {
		    _complement.setSystemTime((new Double(m.group(1))).doubleValue());
		}

		/* Measured emptiness check times (system + user) */
		Pattern emptycheckUser = Pattern.compile("Emptiness check: Measured User time: (\\d+\\.\\d+)");
		m = emptycheckUser.matcher(s);
		if (m.matches()) {
		    _emptiness.setUserTime((new Double(m.group(1))).doubleValue());
		}

		Pattern emptycheckSystem = Pattern.compile("Emptiness check: Measured System time: (\\d+\\.\\d+)");
		m = emptycheckSystem.matcher(s);
		if (m.matches()) {
		    _emptiness.setSystemTime((new Double(m.group(1))).doubleValue());
		}
		
		/* Currently spin doesn't allow to measure its runnng
		 * time internally, so we leave it to default, which
		 * is 0 */
		
	    } //while
	    
	    // Deian March 30, 2007 

	    // According to Fabio, Wring should *always* produce an
	    // empty automaton if the the starting automaton is
	    // universal. This is an extra check to see if this is the
	    // case.
	    if (_universalSet && _universal && _complementSize > 0) {
		System.err.println("Found universal automaton whose complement is non-empty!");
		_errorStream = _errorStream.concat("Found universal automaton whose complement is non-empty!");
		errorCode -= 1000000;
	    }
	       


	    in.close();
	} //try
	catch (IOException e) {
	    System.err.println("Error opening the file " + _file + e);
	    System.exit(1234);
	}
	
	// Make sure that we have parsed the most important stuff
	if (! _universalSet) {
	    _errorStream = _errorStream.concat("The universality of the automaton could not be determined!\n");
	    errorCode -= 1000;
	}

	if (_emptiness.getUserTime() < WringParser.EPSILON) {
	    _errorStream = _errorStream.concat("The emptiness check user time is too small: "
					       + _emptiness.getUserTime() + " !\n");
	    //errorCode -= 10000;
	}

	if (_complement.getUserTime() < WringParser.EPSILON) {
	    _errorStream = _errorStream.concat("The complementation user-time is too small!\n");
	    errorCode -= 100000;
	}
	
	if (_complement.getToolTime() < WringParser.EPSILON) {
	    _errorStream = _errorStream.concat("The complementation tool-time is too small!\n");
	    //errorCode -= 1000000;
	}

	return errorCode;
    }

    
    public boolean getTimeout() {
	return _timeout;
    }


    public String getErrorStream() {
	return _errorStream;
    }

    public MeasuredTime getComplementTime() {
	return _complement;
    }
    
    public MeasuredTime getEmptynessCheck() {
	return _emptiness;
    }
    
}
