
/***************************************************************************************************

 The containment analyzer is useful for checking the containment of two Buechi
 automata. Based on a SCT analyzer by: 
 Copyright (C) 2004  Chin Soon Lee
 Further copyright (C) 2009 Seth Fogarty

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

 The authors can be contacted by e-mail at cslee_sg@hotmail.com and sfogarty@gmail.com.

***************************************************************************************************/


#include <stdio.h>
#include <stdlib.h>

#include "std.h"
#include "sgraph.h"
#include "astate_graph.h"

/**************************************************************************************************
This file contains operations on the graph obtain from automaton A (with
transitions as edges). It's
operations are thus among components of astates.
**************************************************************************************************/

/* "astate_rdf" visits any unvisited vertex v (v->_rdf_flag==false) in the current CG subgraph from 
 vertex "astate", and return these vertices in reverse d.f.s. order. Vertices are marked with "nr" 
 (non-zero) as they are visited; if direction is FORWARD, the _succ fields are used to traverse 
 the CG subgraph; else the _pred fields are used to traverse the transpose graph. */

AStateList astate_rdf(AState astate,AStateList sofar,Direction d,int nr); 

/* "astate_rdfs" visits all unvisited vertices from "astates" using astate_rdf, and returns them. */

AStateList astate_rdfs(AStateList astates,AStateList sofar,Direction d,int nr); 


void init_astates(AStateList astates); 		/* clear _rdf_flag, _pred, _succ fields */
void setup_astates(SGraphList sgraphs); 		/* set _pred, _succ fields from _out_sgraphs fields */
void clear_astate_flags(AStateList astates); 	/* clear _rdf_flag fields */

int unique_astatecomp_nr(void);		/* for labelling s.c.g. SCCs */


/* Returns whether every component of "astates" (vertex set) satifies "prc". 
   The current astate subgraph is specified by the _out_sgraphs field of "astates". */

bool every_astate_comp(prc,astates,sgraphs) 
  bool (*prc)(AStateList); 
  AStateList astates; 
	SGraphList sgraphs; {
  AStateList ordered,fs,gs;
  int res; 

  init_astates(astates); 		/* clear _rdf_flag, _succ, _pred fields */
  setup_astates(sgraphs);  		/* set _succ, _pred (for astate_rdf and astate_rdfs) from _out_sgraphs */

  fs= ordered= astate_rdfs(astates,(AStateList)NULL,FORWARD,true); /* uniquified + rev d.f.s. ordered */
  clear_astate_flags(fs);
  
  res= true;
  while (res && fs!=(AStateList)NULL) {
    if (!fs->head->_rdf_flag) {
      gs= astate_rdf(fs->head,(AStateList)NULL,BACKWARD,unique_astatecomp_nr());
      				/* collect component gs; mark gs members with unique number */ 
      res= (*prc)(gs);
      free_astates(gs);
    }
    fs= fs->tail;
  }
  free_astates(ordered);
  return res;
}


/* "astate_rdf" visits any unvisited vertex v (v->_rdf_flag==false) in the current CG subgraph from 
 * vertex "astate", and return these vertices in reverse d.f.s. order. Vertices are marked with "nr" 
 * (non-zero) as they are visited; if direction is FORWARD, the _succ fields are used to traverse 
   the CG subgraph; else the _pred fields are used to traverse the transpose graph. */
/* PRE: _succ, _pred fields have been set. */

AStateList astate_rdf(astate,sofar,d,nr)
  AState astate;
  AStateList sofar;
  Direction d; 
  int nr; {
  AStateList fs,next;

  astate->_rdf_flag= nr; 
  fs= (AStateList)malloc(sizeof(struct AStateListRec));
  fs->head= astate;
  next= d==FORWARD? astate->_succ: astate->_pred;
  fs->tail= astate_rdfs(next,sofar,d,nr);
  return fs;
}   

/* apply "astate_rdf" to all members of "astates", and return all (currently) unvisited vertices. */

AStateList astate_rdfs(astates,sofar,d,nr)
  AStateList astates,sofar;
  Direction d; 
  int nr; {

  while (astates != (AStateList)NULL) {
    if (!astates->head->_rdf_flag) 
      sofar= astate_rdf(astates->head,sofar,d,nr); /* for single astate */
    astates= astates->tail;
  }
  return sofar;
}

int unique_astatecomp_nr() {
  static int astatecomp_nr= 1;

  if (astatecomp_nr==MAXINT) astatecomp_nr= 1;
  return ++astatecomp_nr;
}

/* For each astate in "astates", clear _rdf_flag, _succ and _pred fields. */

void init_astates(astates) 
  AStateList astates; {
  AState astate;

  while (astates!=(AStateList)NULL) {
    astate= astates->head;
    astate->_rdf_flag= false;

    free_astates(astate->_succ);
    free_astates(astate->_pred);
    astate->_succ= astate->_pred= (AStateList)NULL;

    astates= astates->tail;
  }
}

/* For each astate in "astates", clear _rdf_flag. */

void clear_astate_flags(astates)
  AStateList astates; {

  while (astates!=(AStateList)NULL) {
    astates->head->_rdf_flag= false;
    astates= astates->tail;
  }
}


/* Specify the current CG subgraph by setting the _out_sgraphs field of "astates". */

void setup_out_sgraphs(astates,sgraphs)
  AStateList astates;
  SGraphList sgraphs; {
  AState astate;
  SGraph sgraph;
  SGraphList out_sgraphs;

  while (astates!=(AStateList)NULL) {	/* clear _out_sgraphs field of astates */
    astate= astates->head;
    free_sgraphs(astate->_out_sgraphs);
    astate->_out_sgraphs= (SGraphList)NULL;
    astates= astates->tail;
  }
  while (sgraphs!=(SGraphList)NULL) { /* set _out_sgraphs field of astates */
    sgraph= sgraphs->head;
    astate= sgraph->src;

    out_sgraphs= (SGraphList)malloc(sizeof(struct SGraphListRec));
    out_sgraphs->head= sgraph;
    out_sgraphs->tail= astate->_out_sgraphs;
    astate->_out_sgraphs= out_sgraphs;
    sgraphs= sgraphs->tail;
  }
}

/* Assign the _succ and _pred fields of "astates" for the current CG subgraph. 
   These have been initialized by init_astates(). */

void setup_astates(sgraphs)
  SGraphList sgraphs; {
  SGraph sgraph;
  AState src, dst;
  AStateList fs;

  //while (astates!=(AStateList)NULL) 
  //  sgraphs= astates->head->_out_sgraphs;

    while (sgraphs!=(SGraphList)NULL) {
      sgraph= sgraphs->head;
      src= sgraph->src;
      dst= sgraph->dst;

      fs= (AStateList)malloc(sizeof(struct AStateListRec));
      fs->head= dst;
      fs->tail= src->_succ;
      src->_succ= fs;	/* may have duplicates */

      fs= (AStateList)malloc(sizeof(struct AStateListRec));
      fs->head= src;
      fs->tail= dst->_pred;
      dst->_pred= fs;	/* may have duplicates */

      sgraphs= sgraphs->tail;
    }
}


/* Return the intra-component s.c.g.'s with source astatection in the input "fs". The component of 
   a astatection is identified by the number of its _rdf_flag field. If "fs" is uniquified, so is 
   the output. */
