
/***************************************************************************************************

 The containment analyzer is useful for checking the containment of two Buechi
 automata. Based on a SCT analyzer by: 
 Copyright (C) 2004  Chin Soon Lee
 Further copyright (C) 2009 Seth Fogarty

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

 The authors can be contacted by e-mail at cslee_sg@hotmail.com and sfogarty@gmail.com.

***************************************************************************************************/


#include <stdio.h>
#include <stdlib.h>

#include "std.h"
#include "sgraph.h"
#include "bstate_graph.h"

/**************************************************************************************************
This file contains operations on graphs with bstates as nodes, i.e. sgraphs.

The current bstate subgraph getting attention is specified by the _out_edges field of nodes (SGraph nodes).  
**************************************************************************************************/

/* "bstate_rdf" visits any unvisited vertex v (v->_rdf_flag==false) in the current DG subgraph from 
 vertex "bstate", and return these vertices in reverse d.f.s. order. Vertices are marked with "nr" 
 (non-zero) as they are visited; if direction is FORWARD, the _succ fields are used to traverse 
 the DG subgraph; else the _pred fields are used to traverse the transpose graph. */

BStateList bstate_rdf(BState bstate,BStateList sofar,Direction d,int nr); 

/* Visit all of "bstates" using "bstate_rdf", and return the visited vertices. */

BStateList bstate_rdfs(BStateList bstates,BStateList sofar,Direction d,int nr); 

void clear_bstate_flags(BStateArray bstates, int bsize);   /* clear _rdf_flag field of bstates */

int unique_bstatecomp_nr(void);    /* for labelling DG components */

/* Returns whether each DG-component of "bstates" satisfies prc */

/* pass in a reduced list of bstates to only examine reachable SCC's */

/* Applies prc to every DG-component of bstates */

/* "bstate_rdf" visits any unvisited vertex v (v->_rdf_flag==false) in the current DG subgraph from 
 vertex "bstate", and return these vertices in reverse d.f.s. order. Vertices are marked with "nr" 
 (non-zero) as they are visited; if direction is FORWARD, the _succ fields are used to traverse 
 the DG subgraph; else the _pred fields are used to traverse the transpose graph. 
PRE: _succ, _pred fields specify the current DG subgraph receiving attention. */

BStateList bstate_rdf(bstate,sofar,d,nr)
  BState bstate;
  BStateList sofar;
  Direction d; 
  int nr; {
  BStateList vs,next;

  bstate->_rdf_flag= nr; 
  vs= (BStateList)malloc(sizeof(struct BStateListRec));
  vs->head= bstate;
  next= d==FORWARD? bstate->_succ: bstate->_pred;
  vs->tail= bstate_rdfs(next,sofar,d,nr);
  return vs;
}   

/* Visit all members of "bstates" using "bstate_rdf", and return (currently) unvisited vertices. */

BStateList bstate_rdfs(bstates,sofar,d,nr)
  BStateList bstates,sofar;
  Direction d; 
  int nr; {

  while (bstates != (BStateList)NULL) {
    if (!bstates->head->_rdf_flag) 
      sofar= bstate_rdf(bstates->head,sofar,d,nr); 
    bstates= bstates->tail;
  }
  return sofar;
}

int unique_bstatecomp_nr() {
  static int bstatecomp_nr= 1;

  if (bstatecomp_nr==MAXINT) bstatecomp_nr= 1;
  return ++bstatecomp_nr;
}


/* For each member of "bstates", clear _rdf_flag. */

void clear_bstate_flags(bstates, bsize)
  BStateArray bstates; 
  int bsize;{
  int i;

  for(i=0;i<bsize;i++) {
    (&bstates[i])->_rdf_flag= false;
  }
}

/* For each size edgeendency edge= node1->node2, add edge to the _out_edge field
 * of node1; if node1 corresponds to bstate1, and node2 corresponds to bstate2, add
 * node1 to the bstate1->_src_nodes, and node2 to bstate2->_dst_nodes . */

void add_nodes_and_edges(sgraph) 
  SGraph sgraph; {
  NodeArray dstnodes,srcnodes;
  NodeList xs;
  Node dstnode,srcnode;
  BState dstbstate,srcbstate,v;
  EdgeList edges,ds;
  Edge edge;
  BStateArray bstates;
  BStateList vs;
  BState srcv,dstv;
  int i,bsize;

  /* init from old setup_dg  */
  bstates=sgraph->cp->bstate_array;
  bsize=sgraph->cp->bsize;
  for(i=0;i<bsize;i++) {
    v=&bstates[i];
    free_bstates(v->_pred);
    free_bstates(v->_succ);
    v->_pred= v->_succ= (BStateList)NULL;
    v->_rdf_flag= false;
  } 

  dstnodes= sgraph->dst_node_array;
  for(i=0;i<sgraph->bsize;i++) {
    dstnode=&dstnodes[i];
    dstbstate= dstnode->bstate;
    xs= (NodeList)malloc(sizeof(struct NodeListRec));
    xs->head= dstnode;
    xs->tail= dstbstate->_dst_nodes;
    dstbstate->_dst_nodes= xs;
  }
  srcnodes= sgraph->src_node_array;   
  for(i=0;i<sgraph->bsize;i++) {
    srcnode=&srcnodes[i];
    srcbstate= srcnode->bstate;
    xs= (NodeList)malloc(sizeof(struct NodeListRec));
    xs->head= srcnode;
    xs->tail= srcbstate->_src_nodes;
    srcbstate->_src_nodes= xs;
    edges= srcnode->edges;
    while (edges!=(EdgeList)NULL) {
      edge= edges->head;
      if (!edge->_cln) {
        ds= (EdgeList)malloc(sizeof(struct EdgeListRec));
        ds->head= edge;
        ds->tail= srcnode->_out_edges;
        srcnode->_out_edges= ds;
       
        /* this used to be done in setup_dg, but for simplicity I am doing it
       * here */
        srcv=edge->src->bstate;/* to avoid possible collision with srcbstate
        above */
        dstv=edge->dst->bstate;
        vs= (BStateList)malloc(sizeof(struct BStateListRec));
        vs->head= dstv; /*adding the dstv to the srcv's successors */
        vs->tail= srcv->_succ;
        srcv->_succ= vs;
      
        vs= (BStateList)malloc(sizeof(struct BStateListRec));
        vs->head= srcv;
        vs->tail= dstv->_pred;
        dstv->_pred= vs;
        /* since we already have the edges */
      }
      edges= edges->tail;
    }
    
  }
}
  

/* Set up the current DG subgraph. */

void setup_out_edges(bstates,bsize,sgraph)
  BStateArray bstates;
  int bsize;
  SGraph sgraph; {

  clear_bstates(bstates,bsize);
  clear_out_edges(sgraph);
  add_nodes_and_edges(sgraph);
}


/**************************************************************************************************
Aux funs to do with the DG                                                                
**************************************************************************************************/


/* Are all intra-component arcs deq-labeled? */

bool is_deq_bstatecomp(bstates) 
  BStateList bstates; {
  EdgeList edges;
  NodeList nodes;
  Edge edge;

  while (bstates!=(BStateList)NULL) {
    nodes= bstates->head->_src_nodes;

    while (nodes!=(NodeList)NULL) {
      edges= nodes->head->_out_edges;

      while (edges!=(EdgeList)NULL) {
        edge= edges->head;
        if (edge->accept==DEC && 
                 edge->dst->bstate->_rdf_flag==edge->src->bstate->_rdf_flag)
          return false;
        edges= edges->tail;
      }
      nodes= nodes->tail;
    }
    bstates= bstates->tail;
  }
  return true;
}

void clear_anchor_flags(sgraphs)
  SGraphList sgraphs; {
  SGraph sgraph;
  
  while (sgraphs!=(SGraphList)NULL) {
    sgraph= sgraphs->head;
    sgraph->_anchor= false;
    sgraphs= sgraphs->tail;
  }
}

/* Return the members of "sgraphs" not flagged as anchors */

SGraphList get_unflagged_sgraphs(scc)
  SGraphList scc; {
  SGraphList sgraphs, dummy;
  SGraph sgraph;

  dummy= sgraphs= (SGraphList)malloc(sizeof(struct SGraphListRec));

  while (scc!=(SGraphList)NULL) {
    sgraph= scc->head;
    if (!sgraph->_anchor) {
      sgraphs= sgraphs->tail= (SGraphList)malloc(sizeof(struct SGraphListRec));
      sgraphs->head= sgraph;
    }
    scc= scc->tail;
  }
  sgraphs->tail= (SGraphList)NULL;
  sgraphs= dummy->tail;
  free(dummy);
  return sgraphs;
}

BStateList in_one_scc(sgraph)
  SGraph sgraph; {
  BStateList bstates,ordered,vs,scc,accumulator,temp;
  accumulator=(BStateList)NULL;
  bstates=sgraph->cp->bstates; /*looking through all graphs */
  vs= ordered= bstate_rdfs(bstates,(BStateList)NULL,FORWARD,true); /* uniquified + rev d.f.s. ordered */
  clear_bstate_flags(sgraph->cp->bstate_array, sgraph->cp->bsize);      
  while (vs!=(BStateList)NULL) {
    if (!vs->head->_rdf_flag) {
      scc= bstate_rdf(vs->head,(BStateList)NULL,BACKWARD,unique_bstatecomp_nr());
      if(is_deq_bstatecomp(scc)) {
        free_bstates(scc);
      }
      else {
         temp=scc;
         assert(temp != (BStateList)NULL);
         while(temp->tail !=(BStateList)NULL) {
            temp=temp->tail;
         }
         temp->tail=accumulator;
         accumulator=scc;
      }
    }
    vs= vs->tail;
  }
  free_bstates(ordered);
  return accumulator;
}

BStateList reaching_one_scc(sgraph)
  SGraph sgraph; {
  BStateList inscc, retval;
  inscc = in_one_scc(sgraph);
  clear_bstate_flags(sgraph->cp->bstate_array,sgraph->cp->bsize);
  retval = bstate_rdfs(inscc,(BStateList)NULL,BACKWARD,true);
  retval = bstates_listsort(retval);
  free_bstates(inscc);
  return retval;
}

void annotate_sgraph(sgraph) 
  SGraph sgraph; {

  setup_out_edges(sgraph->cp->bstate_array,sgraph->cp->bsize,sgraph);        /* specify the current DG subgraph */
  BStateList bstates;
  bstates= reachable_from_init(sgraph);  /* reduce this list of bstates to limit the SCC's you reach */
  sgraph->reachable_from_init=bstates;
  bstates = reaching_one_scc(sgraph);
  sgraph->reaching_one_scc=bstates;
}

BStateList reachable_from_init(SGraph g) {
   BStateList accumulator=NULL;
   NodeArray nodes;
   Node node;
   EdgeList edges;
   Edge edge;
   int i;
  
   nodes=g->src_node_array;
   for(i=0;i<g->bsize;i++) {
      node =&nodes[i];
      if(node->bstate->initial) {
         edges = node->edges;
         while(edges != (EdgeList)NULL) {
            assert(edges->head != (Edge)NULL);
            edge = edges->head;
            accumulator = add_bstate_to_set(edge->dst->bstate, accumulator);
            edges = edges->tail;
        }
     }
  }
  return accumulator;
}
