
/***************************************************************************************************

 The containment analyzer is useful for checking the containment of two Buechi
 automata. Based on a SCT analyzer by: 
 Copyright (C) 2004  Chin Soon Lee
 Further copyright (C) 2009 Seth Fogarty

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

 The authors can be contacted by e-mail at cslee_sg@hotmail.com and sfogarty@gmail.com.

***************************************************************************************************/


#include <stdio.h>
#include <stdlib.h>

#include "std.h"
#include "sgraph.h"
#include "astate_graph.h"
#include "bstate_graph.h"
#include "containment.h"

/***************************************************************************************************
 This file contains the procedures for containment analysis.
***************************************************************************************************/

bool containment_scc(AStateList astates);         /* Containment for an SCC of astates */


/* The subgraph of the Call Graph (CG) considered by containment_scc() is specified by the _out_sgraphs field 
   of the input astates. */ 

void restrict_out_sgraphs(AStateList astates);      /* restrict _out_sgraphs to inter-comp sgraphs */
SGraphList get_out_sgraphs(AStateList astates);      /* retrieve all _out_sgraphs of astates */

void clear_clo_sgraphs(AStateList astates);      /* clear and free _clo_sgraphs of astates */
bool insert_sgraph_closure(SGraph sgraph);      /* insert sgraph into _clo_sgraphs if it is not
                 already subsumed; true if sgraph inserted */ 

/* Containment maintains a worklist (WL) of sgraphs. */

SGraphList wl_insert(SGraph sgraph,SGraphList wl);      /* insert element as top of wl */
SGraphList wl_reduce(SGraphList wl);        /* remove top of wl, freeing memory */
SGraphList wl_extend(SGraph sgraph,SGraphList wl);      /* extend sgraph by 1 step; insert new sgraphs
                 into wl; return new wl */
#define wl_top(wl)   (wl->head)
#define wl_empty(wl)  (wl==(SGraphList)NULL)
#define wl_init()  ((SGraphList)NULL)


/* SGraph operations */

SGraph sgraph_copy(SGraph sgraph);          /* make a copy of given sgraph */
SGraph sgraph_compose(SGraph G,SGraph H);        /* compose 2 sgraphs */

SGraphRelation sgraph_compare(SGraph G, SGraph H); /* G <= H? H <= G */

bool critical_sgraph(SGraph sgraph);        /* does sgraph satisfy Ramsey failure cond? */
bool critical_pair(SGraph g,SGraph h);        /* does (g,h) satisfy Buechi failure cond? */
bool fails_lasso_test(SGraph sgraph);        /* is this part of a (g,h) that fails critical_pair? */


/* Other functions */

SGraphList reverse_sgraphs(SGraphList sgraphlist);      /* compute the reverse of an SGraphList */ 
int closure_size(AStateList astates);        /* size of closure computed */


SGraphList FAILURE_PATH;           /* on failure, holds critical multipath */
SGraphList FAILURE_PREFIX;        /* on failure, holds critical multipath */

Containment check_containment(cg) 
  CProb cg; {
  Containment ret;

  FAILURE_PATH= (SGraphList)NULL;  
  FAILURE_PREFIX= (SGraphList)NULL;  
  setup_out_sgraphs(cg->astates,cg->sgraphs);

  ret= (Containment)malloc(sizeof(struct ContainmentRec));
  ret->result = containment_scc(cg->astates);
  //ret->result= every_astate_comp(&containment_scc,cg->astates,cg->sgraphs); no
  //longer clear that it is safe to only go by components.
  ret->failure_path= FAILURE_PATH;
  ret->failure_prefix= FAILURE_PREFIX;
  ret->closure_size= closure_size(cg->astates);
  return ret;
}

/* Containment for an SCC of astates: See Algorithm 7.1 in paper. */

bool containment_scc(astates)
  AStateList astates; {
  SGraphList worklist,out_sgraphs,temp;
  SGraph sgraph;

  restrict_out_sgraphs(astates);        /* restrict _out_sgraphs field of astates to */  
  out_sgraphs= get_out_sgraphs(astates);       /*   inter-component sgraphs */
  temp=out_sgraphs;

  /* STEP 1 of Algorithm 7.1 in paper */
  
  clear_clo_sgraphs(astates);          /* clear _clo_sgraphs field of funs */
  worklist= wl_init();          /* initialize worklist */
  while (out_sgraphs!=(SGraphList)NULL) {
    
    sgraph= out_sgraphs->head;
    worklist= wl_insert(sgraph_copy(sgraph),worklist);
    out_sgraphs= out_sgraphs->tail;
  }
  //free_sgraphs(out_sgraphs); always empty
  free_sgraphs(temp);

  /* STEP 2 */

loop:
  if (wl_empty(worklist)) {
      //dispose_sgraphs(temp); 
      return true;      /* if worklist empty, return true */
  }
  sgraph= wl_top(worklist);        /* else remove some sgraph from worklist */
  worklist= wl_reduce(worklist);
  
  /* STEP 3 */
  
  if (!insert_sgraph_closure(sgraph)) {       /* if sgraph already subsumed, goto STEP 2 */ 
      dispose_sgraph(sgraph);
      goto loop;
  }
  /* STEP 4 */
  
  if (fails_lasso_test(sgraph)) {         /* if failure cond holds, fail */
      dispose_sgraphs(worklist); 
      //FAILURE_PATH= reverse_sgraphs(sgraph->_composition);
      /* set critical LOOP, we don't yet include the prefix: now e do do, in
       * fails_lasso_test*/
      return false;
  }
  /* STEP 5 and 6 */
  
  worklist= wl_extend(sgraph,worklist);       /* extend sgraph 1 step, and add new sgraphs 
                 to worklist */
  /* STEP 7 */

  goto loop;            /* goto STEP 2 */
}


/***************************************************************************************************
 Supporting operations for Containment.
***************************************************************************************************/

/* Clear the _clo_sgraphs fields of funs: the _clo_sgraphs specify the composition closure so far. */

void clear_clo_sgraphs(astates)
  AStateList astates; {
  AState astate;

  while (astates!=(AStateList)NULL) {
    astate= astates->head;
    dispose_sgraphs(astate->_out_clo_sgraphs);
    dispose_sgraphs(astate->_in_clo_sgraphs);
    astate->_in_clo_sgraphs= (SGraphList)NULL;
    astate->_out_clo_sgraphs= (SGraphList)NULL;
    astates= astates->tail;
  }
}
      
/* Restrict a given list of sgraphs to intra-component sgraphs. Input SGraphList is modified destructively.
   In a call to every_cg_comp(), whether an sgraph is intra-component is checked as follows: */
#define intra_comp_sgraph(sgraph)  (sgraph->src->_rdf_flag == sgraph->dst->_rdf_flag)

SGraphList restrict_sgraphs(sgraphs) /* to intra-comp sgraphs */ 
  SGraphList sgraphs; {
  SGraphList res, current_sgraphs;
  SGraph sgraph;
  
  while (sgraphs!=(SGraphList)NULL) {        /* look for first intra-component sgraph */
    sgraph= sgraphs->head;
    if (intra_comp_sgraph(sgraph)) break;      /* on finding one, break out of loop */
    current_sgraphs= sgraphs;
    sgraphs= sgraphs->tail;
    free(current_sgraphs);          /* else free current sgraph cell */
  }
  if (sgraphs==(SGraphList)NULL)
    return (SGraphList)NULL;

  res= current_sgraphs= sgraphs;        /* res points at the sgraph found above */
  sgraphs= sgraphs->tail;

  while (sgraphs!=(SGraphList)NULL) {        /* for each remaining sgraph */
    sgraph= sgraphs->head;
    if (intra_comp_sgraph(sgraph)) {        /* keep it */
      current_sgraphs= sgraphs;
      sgraphs= sgraphs->tail;
    } else {            /* or throw it away */
      current_sgraphs->tail= sgraphs->tail;
      free(sgraphs);
      sgraphs= current_sgraphs->tail;
    }
  }
  return res;
}

/* Restrict the _out_sgraphs field of funs to inter-component sgraphs. The _out_sgraphs field specifies the 
   subgraph of the CG currently receiving attention. */

void restrict_out_sgraphs(astates)
  AStateList astates; {
  AState astate;
  
  while (astates!=(AStateList)NULL) {
    astate= astates->head;
    astate->_out_sgraphs= restrict_sgraphs(astate->_out_sgraphs);
    astates = astates->tail;
  }
}

/* Retrieve the sgraphs in the _out_sgraphs field of astates. */

SGraphList get_out_sgraphs(astates)
  AStateList astates; {
  SGraphList dummy, sgraphs, sgraphset;
  
  dummy= sgraphs= (SGraphList)malloc(sizeof(struct SGraphListRec));
  
  while (astates!=(AStateList)NULL) {
    sgraphset= astates->head->_out_sgraphs;
    while (sgraphset!=(SGraphList)NULL) {

      sgraphs= sgraphs->tail= (SGraphList)malloc(sizeof(struct SGraphListRec));
      sgraphs->head= sgraphset->head;
      sgraphset= sgraphset->tail;
    }
    astates= astates->tail;
  }
  sgraphs->tail= (SGraphList)NULL;
  sgraphs= dummy->tail;
  free(dummy);
  return sgraphs;
}

/* Try to insert given sgraph into the composition closure: If sgraph is already subsumed by some sgraph in
   the _clo_sgraphs field of its src fun, return false without making any insertion; else perform the
   insertion and return true. */

bool insert_sgraph_closure(sgraph) 
  SGraph sgraph; {
  SGraphList sgraphs;
  AState src, dst;
  SGraphRelation relation;
  SGraphList tmp;
  SGraph remove;

  src= sgraph->src;
  dst= sgraph->dst;
  sgraphs= src->_out_clo_sgraphs;
  while (sgraphs!=(SGraphList)NULL) {        /* for each elt of src astate's _clo_sgraphs */
    if(subsumption) {
      relation=sgraph_compare(sgraph,sgraphs->head);
      if ((relation==EQUAL) || (relation==LESSER))
      {
      /* if sgraph is subsumed by it */
        return false;          /* just return false */
      }
      if (relation==GREATER)
      {
        //remove sgraphs->head
        //tricky here, because we have to remove it both from
        //src->_out_clo_sgraphs and from
        //dst->_in_clo_sgraphs
        remove=sgraphs->head;
        assert(dst->_in_clo_sgraphs != NULL);
        if(remove->_in_clo_pred==(SGraphList)NULL) {
          //this means we are teh head of dst->->in_clo_sgraphs.
          wassert(dst->_in_clo_sgraphs->head==remove);
          tmp = dst->_in_clo_sgraphs;
          dst->_in_clo_sgraphs=dst->_in_clo_sgraphs->tail;
          //now we have to fix up our successor.
          free(tmp);
          dst->_in_clo_sgraphs->head->_in_clo_pred=(SGraphList)NULL;
        }
        else {
          //we have a predecessessor in the in_clo_sgraphs.
          assert(remove->_in_clo_pred->tail != NULL);
          //which should point to us
          tmp = remove->_in_clo_pred->tail;
          //this should be the linked list
          //node pointing to remove
          wassert(tmp->head==remove);

          if(tmp->tail!=NULL)
          {
             //if we have something pointing back to us:
             tmp->tail->head->_in_clo_pred=remove->_in_clo_pred; 
          }
          //since we're getting rod of tmp->head
          remove->_in_clo_pred->tail=tmp->tail; 
          //and we want to skip over this node and free it
          free(tmp);
        }
        if(sgraphs->tail==NULL) {
        //we are at the end of the list, lets just swap.
          dispose_sgraph(remove);
          sgraphs->head=sgraph;
        } else {
          //we actually have to remove a node in the list
          tmp=sgraphs->tail;
          sgraphs->head=tmp->head;
          sgraphs->tail=tmp->tail;
          free(tmp);
          dispose_sgraph(remove);
        }
      }
      sgraphs= sgraphs->tail;
    } else {
      if (sgraph_compare(sgraph,sgraphs->head)==EQUAL)
        return false;          /* just return false */
      sgraphs= sgraphs->tail;
    }
  }
  if((sgraphs == NULL) || (sgraphs->head != sgraph)) { 
     sgraphs= (SGraphList)malloc(sizeof(struct SGraphListRec));  /* otherwise perform insertion */
     sgraphs->head= sgraph;
     sgraphs->tail= src->_out_clo_sgraphs;
     src->_out_clo_sgraphs= sgraphs;
  }
  
  //Not checking for subsumption, as we already have considered all possible
  //graphs
  sgraphs= (SGraphList)malloc(sizeof(struct SGraphListRec));  /* otherwise perform insertion */
  sgraphs->head= sgraph;
  sgraphs->head->_in_clo_pred=(SGraphList)NULL;
  tmp=dst->_in_clo_sgraphs;
  sgraphs->tail=tmp; 
  if(tmp != (SGraphList)NULL)
  {
    assert(tmp->head->_in_clo_pred==NULL);
    tmp->head->_in_clo_pred=sgraphs;
  }
  dst->_in_clo_sgraphs= sgraphs;
  
  return true;
}


/***************************************************************************************************
 SGraph operations.
***************************************************************************************************/

/* Return whether given sgraph satisfies the containment failure condition. */


bool critical_pair(g,h)
  SGraph g,h; {
  if ((!(g->src->initial)) || (!(h->src->accepting)))
    return false;
  if ((g->dst!=h->src) || (h->src!=h->dst))
    return false;
  if(bstate_set_intersect(g->reachable_from_init, h->reaching_one_scc))
    return false;
  else
    return true;
  }

bool fails_lasso_test(sgraph)
  SGraph sgraph; {
  SGraphList sgraphs;
  AState src, dst; /* sgraph is a supergraph with an arc src->dst */
  annotate_sgraph(sgraph);

  dst= sgraph->dst; /* first we look at graphs that go out of the dst, dst->foo */
  sgraphs= dst->_out_clo_sgraphs;
  while (sgraphs!=(SGraphList)NULL) {        
      if (critical_pair(sgraph, sgraphs->head)) {
        //FAILURE_PREFIX= reverse_sgraphs(sgraph->_composition);
        //FAILURE_PATH= reverse_sgraphs(sgraphs->head->_composition);
        return true;
      }
      sgraphs= sgraphs->tail;
  } 
  src = sgraph->src; /* Then we look at graphs that go in to the src, foo -> src */
    sgraphs= src->_in_clo_sgraphs;
    while (sgraphs!=(SGraphList)NULL) {        
        if(sgraphs->head != sgraph) { /*we have already checked <g,g> above*/
          if (critical_pair(sgraphs->head, sgraph)) {
            //FAILURE_PREFIX= reverse_sgraphs(sgraphs->head->_composition);
            //FAILURE_PATH= reverse_sgraphs(sgraph->_composition);
            return true;
          }
        }
        sgraphs= sgraphs->tail;
    }
  return false;
}


//typedef enum {EQUAL,GREATER,LESSER,INCOMPARABLE} SGraphRelation;

SGraphRelation SGR_join(SGraphRelation a, SGraphRelation b) {
  switch(a) {
    case EQUAL: 
      return b;
    case INCOMPARABLE: 
      return INCOMPARABLE;
    case GREATER: 
      if ((b==INCOMPARABLE) || (b==LESSER)) {
        return INCOMPARABLE;
      } else { 
        return GREATER;
      }
    default: //case LESSER: 
      if ((b==INCOMPARABLE) || (b==GREATER)) {
        return INCOMPARABLE;
      } else { 
        return LESSER;
      }
  }
}


/* The greater graph (and thus node) is the one with FEWER edges.  Thus if I
 * detect that A has an edge B does not, B has FEWER edges, and I know that A is
 * LESSER than B or incomparable.
 * If I detect that B has an edge that A does not, A has FEWER edges, and I know
 * that A is GREATER than B or incomprable.
*/

SGraphRelation node_compare(Node A, Node B) {
  SGraphRelation retval=EQUAL; //actually take in the current retval
  EdgeList edges_A, edges_B;
  Edge edge_A, edge_B;
  //wassert(A->type==SRC);
  //wassert(B->type==SRC);
  edges_A = A->edges;
  edges_B = B->edges;

  while((edges_A != (EdgeList)NULL) &&(edges_B != (EdgeList)NULL) 
        &&(retval != INCOMPARABLE)) 
  {
    edge_A = edges_A->head;
    edge_B = edges_B->head; //unecessary assignment here sometimes
    assert(edge_A != NULL);
    assert(edge_B != NULL);
    if ((int)edge_A->dst->bstate == (int)edge_B->dst->bstate) {
      if (edge_A->accept > edge_B->accept) { 
        //edge_A ->accept == 1, edge_B ->accept==0
        //This is equivalent to A having an edge that B does not
        retval = SGR_join(retval, LESSER);
      } else if (edge_A->accept < edge_B->accept) {
        //edge_A ->accept == 0, edge_B ->accept==1
        //This is equivalent to B having an edge that A does not
        retval = SGR_join(retval, GREATER);
      }
      edges_A=edges_A->tail;
      edges_B=edges_B->tail;
    } else if ((int)edge_A->dst->bstate < (int)edge_B->dst->bstate) {
      //A has an edge that B does not, because we are sorted.
      retval = SGR_join(retval, LESSER);
      edges_A=edges_A->tail;
    } else { //((int)edge_A->dst->bstate > (int)edge_B->dst->bstate) 
      //B has an edge that A does not
      retval = SGR_join(retval, GREATER);
      edges_B=edges_B->tail;
    }
  }
  if(retval != INCOMPARABLE) {
    if((edges_A != (EdgeList)NULL) && (edges_B == (EdgeList)NULL)) {
      //A has an edge B doesn't, so B cannot be greater than A
      retval = SGR_join(retval, LESSER);
    } else if((edges_A == (EdgeList)NULL) && (edges_B != (EdgeList)NULL)) {
      //B has an edge A doesn't, so B cannot be less than A
      retval = SGR_join(retval, GREATER); 
    }
  }
  return retval;
}

/* Return whether G <= H, for sgraphs G and H, i.e., for every arc of H, is there an arc of G that
   matches or is stricter? */

bool sgraph_lte(G,H) 
  SGraph G,H; {
  //write_sgraph(G);
  //write_sgraph(H);
  NodeArray nodes_H, nodes_G;
  EdgeList edges_H, edges_G, ds;
  Node node_H, node_G;
  Edge edge_H, d;
  BState bstate_H;
  bool found;
  int i;

  if (G->src != H->src || G->dst != H->dst)
    return false;

  nodes_H= H->src_node_array;
  nodes_G= G->src_node_array;
  for(i=0;i<H->bsize;i++) {
    node_G=(&nodes_G[i]);
    node_H=(&nodes_H[i]);

    wassert(node_G->bstate==node_H->bstate);

    
    edges_G= node_G->edges;
    edges_H= node_H->edges;
    passert (assert_edges_sorted(edges_G));
    passert (assert_edges_sorted(edges_H));
    while (edges_H!=(EdgeList)NULL) {      /* for each arc (edge) from this node */
      
      assert(edges_H->head!= NULL);
      edge_H= edges_H->head;
      assert(edge_H->dst->bstate != NULL);
      bstate_H= edge_H->dst->bstate;         /* look for corresponding dst bstate v among 
                       G's arcs */
      found= false;
      
      ds= edges_G;
      while (ds != (EdgeList)NULL) {      /* so: for each of G's arcs */
        assert (ds->head != NULL);
        d= ds->head;
        if (d->dst->bstate == bstate_H) {      /* if its dst bstate matches v */
          found= edge_H->accept==DEQ || d->accept==DEC;  /* decide whether we have found some */ 
          break;          /*    matching or stricter arc */
        }    
        ds= ds->tail;          /* else look on */
      }
      if (!found) return false;        /* if nothing appropriate found, fail */
      edges_H= edges_H->tail;        /* else continue */
    }
  }
  /*
  if(sgraph_compare(G,H) != EQUAL) {
     assert(sgraph_compare(G,H)==LESSER);
     assert(sgraph_compare(H,G)==GREATER);
  }
  */
  return true;            /* success if every arc of H checks out */
}


bool sgraph_eq(G,H) 
  SGraph G,H; {
  return (sgraph_lte(G,H)&&sgraph_lte(H,G));
}


SGraphRelation sgraph_compare(SGraph G, SGraph H) {
  NodeArray nodes_H, nodes_G;
  Node node_H, node_G;
  int i;
  SGraphRelation retval = EQUAL;
  if (G->src != H->src || G->dst != H->dst)
    return INCOMPARABLE;
  
  nodes_H= H->src_node_array;
  nodes_G= G->src_node_array;
  for(i=0;i<H->bsize && retval != INCOMPARABLE;i++) {
    node_G=(&nodes_G[i]);
    node_H=(&nodes_H[i]);
    retval = SGR_join(retval, node_compare(node_G, node_H));
  }
  if (retval==EQUAL) {
    passert(sgraph_eq(G,H));
  }
  else if (retval==LESSER) {
    passert(sgraph_lte(G,H));
    passert(sgraph_lte(H,G)==false);
  }
  else if (retval==GREATER) {
    passert(sgraph_lte(G,H)==false);
    passert(sgraph_lte(H,G));
  }
  else if (retval==INCOMPARABLE) {
    passert(sgraph_lte(G,H)==false);
    passert(sgraph_lte(H,G)==false);
  }
  return retval;
}


/***************************************************************************************************
 sgraph_compose() and sgraph_copy() involve making new SGraph objects. 
***************************************************************************************************/

/* Create new Node object (SGraph node) x for BState v. 
   If source node x corresponds to v, update v->_src_node ("current" source node for v) to x. 
   If target node x corresponds to v, update v->_dst_node ("current" target node for v) to x. */


/* Create new Node objects for BStateList vs. */


/* Given SGraph nodes x,y and bool r, create new edge, and insert it into the edges of x and y. */

void fill_new_edge(x,y,r) 
  Node x,y;
  bool r; {
  Edge d;
  EdgeList ds;

  d= (Edge)malloc(sizeof(struct EdgeRec));
  d->src= x;
  d->dst= y;
  d->accept= r;
  //d->sgraph= x->sgraph;
  
  //d->_ndg_interior= r==DEQ;        /* initialize all working bstateiables */
  d->_cln= false;

  ds= (EdgeList)malloc(sizeof(struct EdgeListRec));  /* insert d into x->edges */
  ds->head= d;
  ds->tail= x->edges;
  x->edges= ds;

  ds= (EdgeList)malloc(sizeof(struct EdgeListRec));  /*  insert d into y->edges */
  ds->head= d;
  ds->tail= y->edges;
  y->edges= ds;
}

/* Having created new Node objects for the nodes of "sgraph", fill_new_edges() is called to create new 
   Edge objects corresponding to the arcs of "sgraph".    

   The new Node objects are accessed via the _src_node and _dst_node fields of the bstates of "sgraph". */

void fill_new_edges(sgraph) 
  SGraph sgraph; {
  NodeArray xs;
  EdgeList ds;
  Edge d;
  Node x,y;
  bool r;
  int i;

  //we don't have to sort, because we are directly copying.
  xs= sgraph->src_node_array;
  for(i=0;i<sgraph->bsize;i++) {
    
    ds=(&xs[i])->edges;
    while (ds!=(EdgeList)NULL) {        /* for each arc from this node */
      
      d= ds->head;
      x= d->src->bstate->_src_node;        /* get the newly created source Node */
      y= d->dst->bstate->_dst_node;        /* get the newly created target Node */
      r= d->accept;
      fill_new_edge(x,y,r);        /* make a new Edge object for them */
      ds= ds->tail;
    } 
  }
}  

/* To make a copy of "sgraph", first make new Node objects according to the nodes of SGraph, then call
   fill_new_edges() to create Edge objects corres. to the arcs of "sgraph" (but for the new nodes). */

SGraph sgraph_copy(sgraph) 
  SGraph sgraph; {
  SGraph sgraph1;
  AState src, dst;
  int i;
  int bsize=sgraph->cp->bsize;
  BStateArray bstate_array=sgraph->cp->bstate_array;

  src= sgraph->src;
  dst= sgraph->dst;
  
  sgraph1= (SGraph)malloc(sizeof(struct SGraphRec));
  sgraph1->reachable_from_init=NULL;
  sgraph1->reaching_one_scc=NULL;
  sgraph1->src= src;
  sgraph1->dst= dst;
  sgraph1->index= -1;
  sgraph1->cp=sgraph->cp;
  sgraph1->bsize=sgraph->bsize;

  sgraph1->src_node_array= new_node_array(bsize,bstate_array,SRC);
  sgraph1->dst_node_array= new_node_array(bsize,bstate_array,DST);
  fill_new_edges(sgraph);
  for(i=0;i<sgraph->bsize;i++) {
    sgraph1->src_node_array[i].edges=edges_listsort(sgraph1->src_node_array[i].edges);
    sgraph1->dst_node_array[i].edges=edges_listsort(sgraph1->dst_node_array[i].edges);
  }

  sgraph1->_anchor= false;          /* initialize working bstateiables */

  //sgraph1->_composition=          /* when containment fails, this contains the */ 
  //  (SGraphList)malloc(sizeof(struct SGraphListRec));    /*    path causing failure, in reverse; */
  //sgraph1->_composition->head= sgraph;      /* initialize it to a list with just sgraph */
  //sgraph1->_composition->tail= (SGraphList)NULL;
  passert(assert_consistent_nodes(sgraph1));
  sgraph1->_in_clo_pred=(SGraphList)NULL;          /* initialize working bstateiables */
  sgraph1->reachable_from_init=NULL;          /* initialize working bstateiables */
  sgraph1->reaching_one_scc=NULL;          /* initialize working bstateiables */
  return sgraph1;
}

/* fill_new_edge(x,y,r): 
   Given SGraph nodes x,y and bool r, create new edge, and insert it into the edges of x and y. 
   
   fill_composed_edge(x,y,r) first checks that (x,y,_) has not already been created. If it has,
   update the size label accordingly, else call fill_new_edge(x,y,r). */

void fill_composed_edge(x,y,r)
  Node x,y;
  bool r; {
  EdgeList ds;
  Edge d;

  ds= x->edges;
  while (ds!=(EdgeList)NULL) {        /* for each arc from x */
    d= ds->head;
    if (d->dst==y) {          /* if it ends in y */
      if (r==DEC) d->accept= DEC;        /*    make it strict if r==DEC */
      return;            /*    and return */
    }
    ds= ds->tail;
  }
  fill_new_edge(x,y,r);          /* else call fill_new_edge(x,y,r) */ 
}

/* For each newly created SGraph node in xs, update the corresponding BState object's _join_node field to 
   point to it. This is applied to sgraph H in prenodeation for computing G;H. */

void fill_join_nodes(xs,bsize) 
  NodeArray xs;
  int bsize; {
  Node x;
  int i;
  
  for(i=0;i<bsize;i++) {
    x=&xs[i];
    x->bstate->_join_node= x;
    //xs= xs->tail;
  }
}

/* Having created new Node objects for the source nodes of "G" and target nodes of "H", 
   fill_composed_edges() is called to create new Edge objects corresponding to the arcs of G;H.
   It is modelled after fill_new_edges() above.
   
   The new Node objects are accessed via the _src_node and _dst_node fields of the bstates of G;H. */

void fill_composed_edges(G,H)   
  SGraph G,H; {
  NodeArray xs_G;
  EdgeList ds, es;
  Edge d, e;
  Node x_G, y_H, x, y;
  bool r;
  int i;
  
  fill_join_nodes(H->src_node_array, H->bsize);        /* for each src node x of H,
                 set x->bstate->_join_node to x */
  xs_G= G->src_node_array;
  for(i=0;i<G->bsize;i++) {
    
    //x_G= xs_G->head;
    x_G=&xs_G[i]; 
    x= x_G->bstate->_src_node;         /* x is corresponding NEW node for G;H */
    
    ds= x_G->edges;
    while (ds != (EdgeList)NULL) {      /* for each arc d from x_G */
      
      d= ds->head;
      y_H= d->dst->bstate->_join_node;       /* find H src node corres. to d's dst */ 

      es= y_H->edges;          /* for each arc e from this node */
      while (es != (EdgeList)NULL) {
        
        e= es->head;
        y= e->dst->bstate->_dst_node;       /* y is the NEW node corres. to e's dst */
              /* new edge will be from x to y */
        r= d->accept==DEC? DEC: e->accept;      /* work out its size label from d and e */
        fill_composed_edge(x,y,r);      /* make new edge */
        es= es->tail;
      }
      ds= ds->tail;
    }
    //xs_G= xs_G->tail;
  } 
}

/* To compute composition G;H, first make new Node objects according to the source nodes of G and 
   target nodes of H, then call fill_composed_edges(G,H) to create Edge objects corresponding to the
   arcs of G;H (connecting the new Node objects).
   
   sgraph_compose(G,H) is modelled after sgraph_copy(sgraph). 
   
Usage: G is always a newly-created sgraph, while H is an sgraph from the input set. */

SGraph sgraph_compose(G,H) 
  SGraph G, H; {
  SGraph res;
  int bsize=G->cp->bsize;
  BStateArray bstate_array=G->cp->bstate_array;
  int i;

  if (G->dst != H->src) 
    error("sgraph_compose: Uncomposable sgraphs.");

  res= (SGraph)malloc(sizeof(struct SGraphRec));
  res->reachable_from_init=NULL;
  res->reaching_one_scc=NULL;
  res->src= G->src;
  res->dst= H->dst;
  res->cp=G->cp;
  res->bsize=G->bsize;
  res->index= -1;

  res->src_node_array= new_node_array(bsize,bstate_array,SRC);   /* make source nodes for G;H */
  res->dst_node_array= new_node_array(bsize,bstate_array,DST);   /* make source nodes for G;H */

  fill_composed_edges(G,H);        /* make new arcs for these nodes--
                    found in the _src_nodes and _dst_nodes
                    fields of the corresponding bstates */
                    /*we now need to sort these edges */
  for(i=0;i<G->bsize;i++) {
    res->src_node_array[i].edges=edges_listsort(res->src_node_array[i].edges);
    res->dst_node_array[i].edges=edges_listsort(res->dst_node_array[i].edges);
  }

  res->_anchor= false;          /* initialize working bstateiables */
  res->_in_clo_pred=(SGraphList)NULL;          /* initialize working bstateiables */
  res->reachable_from_init=NULL;          /* initialize working bstateiables */
  res->reaching_one_scc=NULL;          /* initialize working bstateiables */

  //res->_composition=          /* when containment fails, this contains the */ 
  //  (SGraphList)malloc(sizeof(struct SGraphListRec));    /*    path causing failure, in reverse */
  //res->_composition->head= H;        /* for res, this path starts with H, */
  //res->_composition->tail= copy_sgraphs(G->_composition);   /*    followed by G->_composition */
  passert(assert_consistent_nodes(res));
  return res;
}


/***************************************************************************************************
 Worklist operations.
***************************************************************************************************/

SGraphList wl_reduce(sgraphstack)
  SGraphList sgraphstack; {
  SGraphList sgraphs;

  sgraphs= sgraphstack->tail;
  free(sgraphstack);
  return sgraphs;
}

SGraphList wl_insert(sgraph,sgraphstack)
  SGraph sgraph;
  SGraphList sgraphstack; {
  SGraphList sgraphs;
  
  sgraphs= (SGraphList)malloc(sizeof(struct SGraphListRec));
  sgraphs->head= sgraph;
  sgraphs->tail= sgraphstack;
  return sgraphs;
}

/* Extend sgraph by 1 step, and add new sgraphs to worklist */

SGraphList wl_extend(sgraph,worklist)
  SGraph sgraph; 
  SGraphList worklist; {
  SGraphList sgraphs;

  sgraphs= sgraph->dst->_out_sgraphs;

  while (sgraphs!=(SGraphList)NULL) {
    worklist= wl_insert(sgraph_compose(sgraph,sgraphs->head),worklist);
    sgraphs= sgraphs->tail;
  }
  return worklist;
}


/***************************************************************************************************
 Miscellaneous operations.
***************************************************************************************************/

SGraphList reverse_sgraphs(sgraphlist)
  SGraphList sgraphlist; {
  SGraphList sgraphs, current_sgraphs;

  sgraphs= (SGraphList)NULL;
  while (sgraphlist!=(SGraphList)NULL) {
    current_sgraphs= sgraphs;
    sgraphs= (SGraphList)malloc(sizeof(struct SGraphListRec));
    sgraphs->head= sgraphlist->head;
    sgraphs->tail= current_sgraphs;
    sgraphlist= sgraphlist->tail;
  }
  return sgraphs;
}

int closure_size(astates)
  AStateList astates; {
  int len= 0;

  while (astates!=(AStateList)NULL) {
    len+=len_sgraphs(astates->head->_out_clo_sgraphs);
    astates= astates->tail;
  }
  return len;
}

void free_containment(ret_cont)
  Containment ret_cont; {

  free_sgraphs(ret_cont->failure_path);
  free(ret_cont);
}

void report_cont_failure(prefix, loop)
  SGraphList prefix, loop; {

  printf("\nCritical lasso prefix:\n");
  //write_sgraphs(prefix);
  printf("\nCritical lasso loop:\n");
  //write_sgraphs(loop);
}

