
/***************************************************************************************************

 The containment analyzer is useful for checking the containment of two Buechi
 automata. Based on a SCT analyzer by: 
 Copyright (C) 2004  Chin Soon Lee
 Further copyright (C) 2009 Seth Fogarty

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

 The authors can be contacted by e-mail at cslee_sg@hotmail.com and sfogarty@gmail.com.

***************************************************************************************************/

#include <stdio.h>
#include <stdlib.h>

#include "std.h"
#include "sgraph.h"

/***************************************************************************************************
This file contains operations on the data structures supporting the call graph (CG) [now annotated
call graph (ACG)], defined in sgraph.h
***************************************************************************************************/

/* Write out data structures in sgraph.h */

void write_cg(cg) /* Call Graph is an annotated digraph */
  CProb cg; {
  AStateList astates;
  SGraphList sgraphs;
 
  astates= cg->astates;
  printf("Vertices:\n");
  while (astates!=(AStateList)NULL) {      /* cg's vertices are subject program astates */
    printf("  %s",astates->head->id);
    astates= astates->tail;
  }
  printf("\n\nArcs: \n");
  astates= cg->astates;
  while (astates!=(AStateList)NULL) {      /* cg's arcs are size-change graphs */
    sgraphs= astates->head->out_sgraphs;
    while (sgraphs!=(SGraphList)NULL) {
      write_sgraph(sgraphs->head);
      sgraphs= sgraphs->tail;
    }
    astates= astates->tail;
  }
  printf("\n");
}

void write_sgraphs(sgraphs) 
  SGraphList sgraphs; {

  while (sgraphs!=(SGraphList)NULL) { 
    write_sgraph(sgraphs->head);
    sgraphs= sgraphs->tail;
  }
}

void write_sgraph(sgraph)
  SGraph sgraph; {
  NodeArray xs;
  EdgeList ds;
  string str_x,str_y,str_r;
  Edge d;
  int i;

  printf("(%s %s { ",sgraph->src->id,sgraph->dst->id); /* write out sgraph's src and dst astate */
  xs= sgraph->src_node_array;
  //while (xs!=(NodeList)NULL) {      /* for each s.c.g. node */
  for(i=0;i<sgraph->bsize;i++) {
    ds=(&xs[i])->edges;
    while (ds!=(EdgeList)NULL) {      /* for each edge leaving this node */
      d= ds->head;
      str_x= d->src->bstate->id;      /* note the src node's bstate */
      str_y= d->dst->bstate->id;      /* note the dst node's bstate */
      str_r= d->accept==DEC? "dec": "deq";    /* note the size-change label */
      printf("(%s %s %s) ",str_x,str_y,str_r);  /* write it all out */
      ds= ds->tail;
    }
   // xs= xs->tail;
  }
  printf("})\n");
}

void write_edges(ds)
  EdgeList ds; {
  string str_x,str_y,str_r;
  Edge d;
  while (ds!=(EdgeList)NULL) {      /* for each edge leaving this node */
    d= ds->head;
    str_x= d->src->bstate->id;      /* note the src node's bstate */
    str_y= d->dst->bstate->id;      /* note the dst node's bstate */
    str_r= d->accept==DEC? "dec": "deq";    /* note the size-change label */
    printf("(%s %s %s) ",str_x,str_y,str_r);  /* write it all out */
    ds= ds->tail;
  }
  printf("})\n");
 }


/***************************************************************************************************
 Disposing recursive structures                                                                            
***************************************************************************************************/

void dispose_edge(edge)
  Edge edge; {
  free(edge);
}

void dispose_edges(edges)
  EdgeList edges; {
  EdgeList current_edges;
  
  while (edges!=(EdgeList)NULL) {
    dispose_edge(edges->head);
    current_edges= edges;
    edges= edges->tail;
    free(current_edges);
  }
}

void dispose_bstate(bstate)
  BState bstate; {

  free(bstate->id);

  free_nodes(bstate->_src_nodes);
  free_nodes(bstate->_dst_nodes);

  free_bstates(bstate->_succ);
  free_bstates(bstate->_pred);

  //free(bstate);
}

void dispose_src_par(par)
  Node par; {
    
  dispose_edges(par->edges);
  
  free_edges(par->_out_edges);
  //free(par);
}

void dispose_dst_par(par)
  Node par; {

  free_edges(par->edges);
  //free(par);
}

void dispose_src_nodes(nodes,bsize)
  NodeArray nodes; {
  int i;

  for(i=0; i<bsize; i++) {
    dispose_edges(nodes[i].edges);
    free_edges(nodes[i]._out_edges);
  }
  free(nodes);
}

void dispose_dst_nodes(nodes,bsize)
  NodeArray nodes; {
  int i;

  for(i=0; i<bsize; i++) {
     free_edges(nodes[i].edges);
  }
  free(nodes);
}
   

void dispose_sgraph(sgraph)
  SGraph sgraph; {

  //free_sgraphs(sgraph->_composition);
  if(sgraph->src_node_array!=(NodeArray)NULL)
     dispose_src_nodes(sgraph->src_node_array, sgraph->bsize);
  if(sgraph->dst_node_array!=(NodeArray)NULL) 
     dispose_dst_nodes(sgraph->dst_node_array, sgraph->bsize);
  free_bstates(sgraph->reachable_from_init); 
  free_bstates(sgraph->reaching_one_scc);
  free(sgraph);
}

void dispose_sgraphs(sgraphs)
  SGraphList sgraphs; {
  SGraphList current_sgraphs;

  while (sgraphs!=(SGraphList)NULL) {
    dispose_sgraph(sgraphs->head);
    current_sgraphs= sgraphs;
    sgraphs= sgraphs->tail;
    free(current_sgraphs);
  }
}

void dispose_bstates(bstates)
  BStateList bstates; {
  BStateList current_bstates;

  while (bstates!=(BStateList)NULL) {
    dispose_bstate(bstates->head);
    current_bstates= bstates;
    bstates= bstates->tail;
    free(current_bstates);
  }
}

void release_bstates(bstates)
  BStateList bstates; {
  BStateList current_bstates;

  while (bstates!=(BStateList)NULL) {
    free(bstates->head);
    current_bstates= bstates;
    bstates= bstates->tail;
    free(current_bstates);
  }
}
               

void dispose_astate(astate) 
  AState astate; {

  free(astate->id);

  free_sgraphs(astate->in_sgraphs);
  dispose_sgraphs(astate->out_sgraphs);
    
  free_astates(astate->_succ);
  free_astates(astate->_pred);
  free_sgraphs(astate->_out_sgraphs);
  dispose_sgraphs(astate->_in_clo_sgraphs);
  free_sgraphs(astate->_out_clo_sgraphs);

  free(astate);
}

void dispose_astates(astates)
  AStateList astates; {
  AStateList current_astates;

  while (astates!=(AStateList)NULL) {
    dispose_astate(astates->head);
    current_astates= astates;
    astates= astates->tail;
    free(current_astates);
  }
}

void dispose_cp(cg)
  CProb cg; {
  BStateArray barray;
  SGraphList iter;
  int i;
  clear_bstates(cg->bstate_array,cg->bsize);
  barray=cg->bstate_array;

  iter = cg->sgraphs;
  while(iter != NULL) {
     clear_out_edges(iter->head);
     iter=iter->tail;
  }


  for(i=0;i<cg->bsize;i++) {
    free_bstates(barray[i]._pred);
    free_bstates(barray[i]._succ);
    barray[i]._pred= barray[i]._succ= (BStateList)NULL;
    barray[i]._rdf_flag= false;
  } 


  dispose_astates(cg->astates);
  dispose_bstates(cg->bstates);
  free_sgraphs(cg->sgraphs);
  free(cg->bstate_array);
  free(cg);
}

void free_astates(astates)
  AStateList astates; {
  AStateList current_astates;

  while (astates!=(AStateList)NULL) {
    current_astates= astates;
    astates= astates->tail;
    free(current_astates);
  }
}


/***************************************************************************************************
 Non-recursive disposal
***************************************************************************************************/

void free_sgraphs(sgraphs)
  SGraphList sgraphs; {
  SGraphList current_sgraphs;

  while (sgraphs!=(SGraphList)NULL) {
    current_sgraphs= sgraphs;
    sgraphs= sgraphs->tail;
    free(current_sgraphs);
  }
}

void free_bstates(bstates)
  BStateList bstates; {
  BStateList current_bstates;

  while (bstates!=(BStateList)NULL) {
    current_bstates= bstates;
    bstates= bstates->tail;
    free(current_bstates);
  }
}

void free_edges(edges)
  EdgeList edges; {
  EdgeList current_edges;
  
  while (edges!=(EdgeList)NULL) {
    current_edges= edges;
    edges= edges->tail;
    free(current_edges);
  }
}

void free_nodes(nodes)
  NodeList nodes; {
  NodeList current_nodes;

  while (nodes!=(NodeList)NULL) {
    current_nodes= nodes;
    nodes= nodes->tail;
    free(current_nodes);
  }
}


/***************************************************************************************************
 Misc
***************************************************************************************************/

int len_sgraphs(sgraphs) 
  SGraphList sgraphs; { /* number of elements of "sgraphs" */
  int len= 0;

  while (sgraphs!=(SGraphList)NULL) {
    len++;
    sgraphs= sgraphs->tail;
  }
  return len;
}

int len_astates(astates)
  AStateList astates; {
  int len= 0;

  while (astates!=(AStateList)NULL) {
    len++;
    astates= astates->tail;
  }
  return len;
}

int len_bstates(bstates)
  BStateList bstates; {
  int len= 0;

  while (bstates!=(BStateList)NULL) {
    len++;
    bstates= bstates->tail;
  }
  return len;
}

int len_nodes(nodes)
  NodeList nodes; {
  int len= 0;

  while (nodes!=(NodeList)NULL) {
    len++;
    nodes= nodes->tail;
  }
  return len;
}

int len_edges(edges)
  EdgeList edges; {
  int len= 0;

  while (edges!=(EdgeList)NULL) {
    len++;
    edges= edges->tail;
  }
  return len;
}


/* top-level copy of "sgraphs" */

SGraphList copy_sgraphs(sgraphset) 
  SGraphList sgraphset; {
  SGraphList dummy, sgraphs;

  dummy= sgraphs= (SGraphList)Malloc(sizeof(struct SGraphListRec));

  while (sgraphset!=(SGraphList)NULL) {
    sgraphs= sgraphs->tail= (SGraphList)Malloc(sizeof(struct SGraphListRec));
    sgraphs->head= sgraphset->head;
    sgraphset= sgraphset->tail;
  }
  sgraphs->tail= (SGraphList)NULL;
  sgraphs= dummy->tail;
  free(dummy);
  return sgraphs;
}


/* lists as database functions */
  
void astate_add(string id, bool init, bool accept, CProb cprob) {
  AStateList prev;
  AState astate;

  astate = (AState)Malloc(sizeof(struct AStateRec));
  astate->id=id;
  astate->initial=init;
  astate->accepting=accept;
  astate->in_sgraphs=NULL;
  astate->out_sgraphs=NULL;
  astate->_in_clo_sgraphs=NULL;
  astate->_out_clo_sgraphs=NULL;
  astate->_out_sgraphs=NULL;
  astate->_succ=NULL;
  astate->_pred=NULL;
  
  prev=cprob->astates;
  cprob->astates = (AStateList)Malloc(sizeof(struct AStateListRec));
  cprob->astates->tail=prev;
  cprob->astates->head= astate;
  return;
}


void bstate_add(string id, bool init, CProb cprob) {
  BStateList prev;
  BState bstate;
  bstate= (BState)Malloc(sizeof(struct BStateRec));
  bstate->id=id;
  bstate->initial=init;
  bstate->_succ=NULL;
  bstate->_pred=NULL;
  bstate->_src_node=NULL;
  bstate->_dst_node=NULL;
  bstate->_join_node=NULL;
  bstate->_src_nodes=NULL;
  bstate->_dst_nodes=NULL;
  cprob->bsize++;

  prev=cprob->bstates;
  cprob->bstates = (BStateList)Malloc(sizeof(struct BStateListRec));
  cprob->bstates->tail=prev;
  cprob->bstates->head=bstate;
  return;
}

AState astate_lookup(string id, CProb cprob) {
  AStateList astates=cprob->astates;
  if(astates==(AStateList)NULL) {
    printf("Tried to look up %s in empty AStateList\n", id);
    exit(1);
  }
  if(astates->head==(AState)NULL) {
    printf("Tried to look up %s in singleton AStateList\n", id);
    exit(1);
  }
  while(astates!=(AStateList)NULL) {
    if(strcmp(astates->head->id, id)==0) {
       return astates->head;
    }
    astates=astates->tail;
  }
  printf("Failed to find %s in populated AStateList\n", id);
  exit(1);
}

BState bstate_lookup(string id, CProb cprob) {
  BStateList bstates=cprob->bstates;
  if(bstates==(BStateList)NULL) {
    printf("Tried to look up %s in empty BStateList\n", id);
    exit(1);
  }
  if(bstates->head==(BState)NULL) {
    printf("Tried to look up %s in singleton BStateList\n", id);
    exit(1);
  }
  while(bstates!=(BStateList)NULL) {
    if(strcmp(bstates->head->id, id)==0) {
       return bstates->head;
    }
  }
  printf("Failed to find %s in populated BStateList\n", id);
  exit(1);
}

Node node_array_lookup(string id, NodeArray narray, CProb cprob) {
  int i;
  for(i=0; i<cprob->bsize; i++) {
    if(strcmp(narray[i].bstate->id, id)==0) {
      return &(narray[i]);
    }
  }
  fprintf(stderr,"Attempted to look up %s but could not find it. CP dump:\n",id);
  write_cg(cprob);
  exit(1);
  return NULL;
}

void sgraph_add(SGraph sgraph, CProb cprob) {
  SGraphList prev;
  prev=cprob->sgraphs;
  cprob->sgraphs= (SGraphList)Malloc(sizeof(struct SGraphListRec));
  cprob->sgraphs->tail=prev;
  cprob->sgraphs->head=sgraph;
  return;
}

NodeList new_nodelist() {
  NodeList nlist;
  nlist=(NodeList)Malloc(sizeof(struct NodeListRec));
  nlist->head=NULL;
  nlist->tail=NULL;
  return nlist;
}

BStateList add_bstate_to_set(BState bstate, BStateList set) {
  BStateList pointer, prev, newnode;
  pointer=prev=set;
  if(set == (BStateList)NULL || (((int)(set->head)) > ((int)bstate))) {
  /*we have an empty set, or the lowest element in the set */
    newnode = (BStateList)Malloc(sizeof(struct BStateListRec));
    newnode->head = bstate;
    newnode->tail=set;
    return newnode;
  }
  /*otherwise we have a non-empty set, so find our place */
  while(pointer!=(BStateList)NULL && (((int)(pointer->head)) < ((int)bstate))) {
    wassert(strcmp(pointer->head->id, bstate->id)!=0);
    prev=pointer;
    pointer=pointer->tail;
  }
  if(pointer==(BStateList)NULL || pointer->head != bstate) { 
  //we found something greater than ourselves (or the end of the list)
    wassert(prev->tail==pointer);
    newnode = (BStateList)Malloc(sizeof(struct BStateListRec));
    newnode->head = bstate;
    newnode->tail=pointer;
    prev->tail=newnode;
    return set;
  }
  else {
    wassert(pointer->head == bstate);
    wassert(strcmp(pointer->head->id, bstate->id)==0);
    return set;
  }
}

bool list_is_sorted_set(BStateList bstates)
{
   BState current=NULL;
   while(bstates != (BStateList)NULL) {
      assert(bstates->head != (BState)NULL);
      assert(((int)current) < ((int)(bstates->head)));
      current=bstates->head;
      bstates=bstates->tail;
   }
   return true;
}


bool assert_edges_sorted(EdgeList edges)
{
   if(edges == (EdgeList)NULL) {
    return true;
   }
   Edge current=edges->head;
   edges=edges->tail;
   while(edges != (EdgeList)NULL) {
      assert(edges->head != (Edge)NULL);
      assert(edges->head->dst != (Node)NULL);
      assert(edges->head->src!= (Node)NULL);
      if(((int)(current->src->bstate)) == ((int)(edges->head->src->bstate))) {
         assert(((int)(current->dst->bstate)) < ((int)(edges->head->dst->bstate)));
      } else { 
         assert(((int)(current->src->bstate)) < ((int)(edges->head->src->bstate)));
      }
      current=edges->head;
      edges=edges->tail;
   }
   return true;
}

bool bstate_set_intersect(BStateList set1, BStateList set2) {
  passert(list_is_sorted_set(set1));
  passert(list_is_sorted_set(set2));
  bool res=false;
  while((res==false) && (set1 != (BStateList)NULL) && (set2 != (BStateList)NULL)) {
     if((int)(set1->head) == (int)(set2->head)) {
       wassert(strcmp(set1->head->id, set2->head->id)==0);
       res=true;
     }
     else if ((int)(set1->head) < (int)(set2->head)) {
       wassert(strcmp(set1->head->id, set2->head->id)!=0);
       set1=set1->tail;
     }
     else /* set1->head > set2->head */ {
       wassert(strcmp(set1->head->id, set2->head->id)!=0);
       set2=set2->tail;
     }
  }
  return res;
}

SGraphList sgraph_prepend(SGraph graph, SGraphList rest) {
  SGraphList newlist;
  newlist = (SGraphList)Malloc(sizeof(struct SGraphListRec));
  newlist->head=graph;
  newlist->tail=rest;
  return newlist;
}

EdgeList edge_prepend(Edge graph, EdgeList rest) {
  EdgeList newlist;
  newlist = (EdgeList)Malloc(sizeof(struct EdgeListRec));
  newlist->head=graph;
  newlist->tail=rest;
  return newlist;
}

NodeList node_prepend(Node graph, NodeList rest) {
  NodeList newlist;
  newlist = (NodeList)Malloc(sizeof(struct NodeListRec));
  newlist->head=graph;
  newlist->tail=rest;
  return newlist;
}

void finish_sgraph(SGraph sgraph) {
  int i;
  //EdgeList eiter;

  for(i=0;i<sgraph->cp->bsize;i++) {
    //sgraph->src_node_array[i].sgraph=sgraph;
    //sgraph->dst_node_array[i].sgraph=sgraph;
    sgraph->src_node_array[i].edges=edges_listsort(sgraph->src_node_array[i].edges);
    sgraph->dst_node_array[i].edges=edges_listsort(sgraph->dst_node_array[i].edges);
    //eiter = sgraph->src_node_array[i].edges;
    /*while(eiter != NULL)
    {
        eiter->head->sgraph=sgraph;
        eiter = eiter->tail;
    }*/
  }

  return;
}

//This is WRONG, but not yet changed -- SF
//You should only make nodes that have incoming (outgoing) edges.
/*
Node new_untied_node(v,ty)
  BState v;
  NodeType ty; {
  Node x;

  x= (Node)malloc(sizeof(struct NodeRec));
  x->bstate= v;
  //x->type= ty;
  x->edges= (EdgeList)NULL;        // to be assigned 
  
  //x->_flag= false;
  x->_out_edges= (EdgeList)NULL;

  if (ty==SRC) v->_src_node= x;        // make x the "current" src node for v 
  if (ty==DST) v->_dst_node= x;        // make x the "current" dst node for v 
  return x;
}
*/

Node new_node(v,ty)
  BState v;
  NodeType ty; {
  Node x;

  x= (Node)malloc(sizeof(struct NodeRec));
  x->bstate= v;
  //x->type= ty;
  x->edges= (EdgeList)NULL;        /* to be assigned */
  //x->sgraph=sgraph;
  
  //x->_flag= false;
  x->_out_edges= (EdgeList)NULL;

  if (ty==SRC) v->_src_node= x;        /* make x the "current" src node for v */
  if (ty==DST) v->_dst_node= x;        /* make x the "current" dst node for v */
  return x;
}

/*
NodeList new_untied_nodes_from_array(int bsize, BStateArray vs, NodeType ty) {
  int i;
  NodeList xs, dummy;

  dummy= xs= (NodeList)malloc(sizeof(struct NodeListRec));
  
  for(i=0;i<bsize;i++) {
    xs= xs->tail= (NodeList)malloc(sizeof(struct NodeListRec));
    xs->head= new_untied_node(&vs[i],ty);
  }
  xs->tail= (NodeList)malloc(sizeof(struct NodeListRec));
  xs->tail->head= (Node)NULL;
  xs->tail->tail= (NodeList)NULL;
  xs= dummy->tail;
  free(dummy);
  return xs;
}
*/

/*
NodeArray new_untied_node_array(int bsize, BStateArray bstate_array, NodeType ty) {
  NodeArray retval;
  BStateArray xs;
  int i;
  retval = (NodeArray)malloc(bsize*(sizeof(struct NodeRec)));
  xs = bstate_array;
  i=0;
  for(i=0; i<bsize; i++) {
    retval[i].bstate=&xs[i];
    //retval[i].type=ty;
    retval[i].edges= (EdgeList)NULL;        // to be assigned 
    //retval[i]._flag= false;
    retval[i]._out_edges= (EdgeList)NULL;
  }
  return retval;
}
*/

NodeArray new_node_array(int bsize, BStateArray bstate_array,  NodeType ty) {
  NodeArray retval;
  BStateArray xs;
  int i;
  retval = (NodeArray)malloc(bsize*(sizeof(struct NodeRec)));
  xs = bstate_array;
  i=0;
  for(i=0; i<bsize; i++) {
    retval[i].bstate=&xs[i];
    //retval[i].type=ty;
    retval[i].edges= (EdgeList)NULL;        /* to be assigned */
    //retval[i]._flag= false;
    retval[i]._out_edges= (EdgeList)NULL;
    if (ty==SRC) (&xs[i])->_src_node=&retval[i];
    if (ty==DST) (&xs[i])->_dst_node=&retval[i];

  }
  return retval;
}



NodeList new_nodes_from_array(bsize,vs,ty) 
  int bsize;
  BStateArray vs; 
  NodeType ty; {
  NodeList xs, dummy;
  int i;

  dummy= xs= (NodeList)malloc(sizeof(struct NodeListRec));
  
  for(i=0;i<bsize;i++) {
    xs= xs->tail= (NodeList)malloc(sizeof(struct NodeListRec));
    xs->head= new_node(&vs[i],ty);
  }
  xs->tail= (NodeList)NULL;
  xs= dummy->tail;
  free(dummy);
  return xs;
}

/*
 * The next two functions are copyright 2001 Simon Tatham.
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL SIMON TATHAM BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

int bstatelist_cmp(BStateList a, BStateList b) {
  return ((int)a->head) - ((int)b->head);
}

int edgelist_cmp(EdgeList a, EdgeList b) {
  wassert(a->head->src->bstate==b->head->src->bstate || a->head->dst->bstate==b->head->dst->bstate );
  if(((int)a->head->src->bstate)==((int)b->head->src->bstate)) {
     return ((int)a->head->dst->bstate) - ((int)b->head->dst->bstate);
  } else { 
     return ((int)a->head->src->bstate) - ((int)b->head->src->bstate);
  }
}

/*
 * This is the actual sort function. Notice that it returns the new
 * head of the list. (It has to, because the head will not
 * generally be the same BStateList after the sort.) So unlike sorting
 * an array, where you can do
 * 
 *   sort(myarray);
 * 
 * you now have to do
 * 
 *   list = listsort(mylist);
 */
BStateList bstates_listsort(BStateList list) {
  BStateList p, q, e, tail, oldhead;
  int insize, nmerges, psize, qsize, i;
  oldhead=list;

  /*
   * Silly special case: if `list' was passed in as NULL, return
   * NULL immediately.
   */
  if (!list)
    return NULL;

  insize = 1;

  while (1) {
    p = list;
    list = NULL;
    tail = NULL;

    nmerges = 0;  /* count number of merges we do in this pass */

    while (p) {
      nmerges++;  /* there exists a merge to be done */
      /* step `insize' places along from p */
      q = p;
      psize = 0;
      for (i = 0; i < insize; i++) {
        psize++;
        q = q->tail;
        if (!q) break;
      }

      /* if q hasn't fallen off end, we have two lists to merge */
      qsize = insize;

      /* now we have two lists; merge them */
      while (psize > 0 || (qsize > 0 && q)) {

        /* decide whether next BState of merge comes from p or q */
        if (psize == 0) {
          /* p is empty; e must come from q. */
          e = q; q = q->tail; qsize--;
        } else if (qsize == 0 || !q) {
          /* q is empty; e must come from p. */
          e = p; p = p->tail; psize--;
        } else if (bstatelist_cmp(p,q) <= 0) {
          /* First BState of p is lower (or same);
           * e must come from p. */
          e = p; p = p->tail; psize--;
        } else {
          /* First BStateList of q is lower; e must come from q. */
          e = q; q = q->tail; qsize--;
        }

        /* add the next BStateList to the merged list */
        if (tail) {
          tail->tail = e;
        } else {
          list = e;
        }
        tail = e;
      }

      /* now p has stepped `insize' places along, and q has too */
      p = q;
    }
    tail->tail = NULL;

    /* If we have done only one merge, we're finished. */
    if (nmerges <= 1)   /* allow for nmerges==0, the empty list case */
      return list;

    /* Otherwise repeat, merging lists twice the size */
    insize *= 2;
  }
}

EdgeList edges_listsort(EdgeList list) {
  EdgeList p, q, e, tail, oldhead;
  int insize, nmerges, psize, qsize, i;
  oldhead=list;

  /*
   * Silly special case: if `list' was passed in as NULL, return
   * NULL immediately.
   */
  if (!list)
    return NULL;

  insize = 1;

  while (1) {
    p = list;
    list = NULL;
    tail = NULL;

    nmerges = 0;  /* count number of merges we do in this pass */

    while (p) {
      nmerges++;  /* there exists a merge to be done */
      /* step `insize' places along from p */
      q = p;
      psize = 0;
      for (i = 0; i < insize; i++) {
        psize++;
        q = q->tail;
        if (!q) break;
      }

      /* if q hasn't fallen off end, we have two lists to merge */
      qsize = insize;

      /* now we have two lists; merge them */
      while (psize > 0 || (qsize > 0 && q)) {

        /* decide whether next Edge of merge comes from p or q */
        if (psize == 0) {
          /* p is empty; e must come from q. */
          e = q; q = q->tail; qsize--;
        } else if (qsize == 0 || !q) {
          /* q is empty; e must come from p. */
          e = p; p = p->tail; psize--;
        } else if (edgelist_cmp(p,q) <= 0) {
          /* First Edge of p is lower (or same);
           * e must come from p. */
          e = p; p = p->tail; psize--;
        } else {
          /* First EdgeList of q is lower; e must come from q. */
          e = q; q = q->tail; qsize--;
        }

        /* add the next EdgeList to the merged list */
        if (tail) {
          tail->tail = e;
        } else {
          list = e;
        }
        tail = e;
      }

      /* now p has stepped `insize' places along, and q has too */
      p = q;
    }
    tail->tail = NULL;

    /* If we have done only one merge, we're finished. */
    if (nmerges <= 1) {   /* allow for nmerges==0, the empty list case */
      passert(assert_edges_sorted(list));
      return list;
    }

    /* Otherwise repeat, merging lists twice the size */
    insize *= 2;
  }
}

BStateArray array_of_bstatelist(BStateList list, int bsize) {
  BStateArray array;
  int i;
  array = (BStateArray)malloc(bsize*sizeof(struct BStateRec));

  for(i=0;i<bsize;i++) {
    assert(list!=(BStateList)NULL);
    array[i].id=list->head->id;
    array[i].index=i;
    array[i].initial=list->head->initial;
    array[i]._succ=list->head->_succ;
    array[i]._pred=list->head->_pred;
    array[i]._src_node=list->head->_src_node;
    array[i]._dst_node=list->head->_dst_node;
    array[i]._join_node=list->head->_join_node;
    array[i]._src_nodes=list->head->_src_nodes;
    array[i]._dst_nodes=list->head->_dst_nodes;
    list=list->tail;
  }
  return array;
}

BStateList list_of_bstatearray(BStateArray array, int bsize) {
  BStateList list=NULL;
  BStateList head=NULL;
  int i;

  for(i=0;i<bsize;i++) {
    if(i==0) {
      list = (BStateList)malloc(sizeof(struct BStateListRec));
      head=list;
    } else {
      list->tail = (BStateList)malloc(sizeof(struct BStateListRec));
      list=list->tail;
    }
    list->head=&array[i];
    list->tail=(BStateList)NULL;
    assert(list!=(BStateList)NULL);
  }
  return head;
}


bool assert_consistent_bstate(BState a, BState b) {
    assert(a != (BState)NULL);
    assert(b != (BState)NULL);
    assert(a->id==b->id);
    assert(a->initial==b->initial);
    return true;
}



bool assert_consistent_cprob(CProb cp) {
  int i;
  SGraphList sgraphs=cp->sgraphs;
  BState ar;
  BState ls;
  BStateList iter=cp->bstates;
  for(i=0; i<cp->bsize; i++) {
    ar=&(cp->bstate_array[i]);
    ls=iter->head;
    assert(ls != NULL);
    assert_consistent_bstate(ar, ls);
    iter=iter->tail; 
  }
  while(sgraphs!=(SGraphList)NULL) {
    assert(sgraphs->head != (SGraph)NULL);
    assert_consistent_nodes(sgraphs->head);
    sgraphs=sgraphs->tail;
  }
  return true;
}

bool assert_consistent_nodes(SGraph sgraph) {
  int i,j;
  Node ar;
  NodeArray array;
  EdgeList ar_edges;

  for(i=0;i<2;i++) {
    if(i==0) {
      //iter=sgraph->src_nodes;
      array=sgraph->src_node_array;
    } else {
      //iter=sgraph->dst_nodes;
      array=sgraph->dst_node_array;
    }
      
    for(j=0; i<sgraph->cp->bsize; i++)  {
      ar=&array[j];
      ar_edges=ar->edges;
      assert_edges_sorted(ar_edges);
     // while(ar_edges != (EdgeList)NULL) {
        //ar_edge=ar_edges->head;
        //assert(ar_edge->sgraph==sgraph);
        //ar_edges=ar_edges->tail;
     // }
    }
  }
  return true;
}

/* The current DG subgraph is specified by the _out_edges fields of nodes (SGraph nodes).
 * To initialize the current subgraph, update the _src_nodes and _dst_nodes fields of
 * "bstates"; then the _out_edge field of these nodes. */

void clear_bstates(bstates,bsize) /* clear the _src_node and _dst_node fields of "bstates" */
  BStateArray bstates; 
  int bsize; {
  BState bstate;
  int i;
 
  for(i=0;i<bsize;i++) {
    bstate=&bstates[i];
 
    free_nodes(bstate->_src_nodes);
    bstate->_src_nodes= (NodeList)NULL;

    free_nodes(bstate->_dst_nodes);
    bstate->_dst_nodes= (NodeList)NULL;
    //bstates=bstates->tail;
  }
}

/* Clear _out_edges field of s.c.g. bstates (nodes) */

void clear_out_edges(sgraph) 
  SGraph sgraph; {
  NodeArray nodes;
  Node node;
  int i;
  nodes= sgraph->src_node_array;

  for(i=0;i<sgraph->bsize;i++) { 
    node=&nodes[i];// nodes->head;
    free_edges(node->_out_edges);
    node->_out_edges= (EdgeList)NULL;
  }
}
