module Abs where

import Set

{--

  Size-change arc.

  The set of sizechanges is represented by the powerset of SCabs.
  The lattice is given by the subset relation.

  in this context: 
    top = [Inc, Dec, Eq]
    bot = []


       Top
     /  |   \
 *  IE  DE  NE=*
  \ / \ / \ /
  Inc  Eq  Dec
    \  |   /
      Bot


--}

-- Define the elements for the abstract value
data SCabs = Inc
             | Dec
             | Eq
             deriving (Eq)

instance Ord SCabs where
  (<)  = scabsLT
  (<=) = (\x->(\y->(scabsLT x y)||x==y))

instance Show SCabs where
  show = (\x->case x of
           Inc -> "+"
           Eq  -> "="
           Dec -> "-")
         

scabsLT Dec x   = (not (x==Dec))
scabsLT Eq  Inc = True
scabsLT _   _   = False

-- Size change over a single variable
--   Since the set of abstract values is the power set of Scarrow
--   all ordinary operations are given by the Set implementation

type SCarc = Set SCabs

scarcTop :: SCarc
scarcTop = makeSet [Dec, Eq, Inc]

scarcDecEq = makeSet [Dec, Eq]

scarcIncEq = makeSet [Eq, Inc]

scarcBot :: SCarc
scarcBot = empty

scarcInc :: SCarc -> SCarc
scarcInc xs = union (if memSet xs Dec
                    then makeSet [Eq, Dec]
                    else empty)
                    (if memSet xs Eq || memSet xs Inc
                    then sing Inc
                    else empty)

scarcDec :: SCarc -> SCarc
scarcDec xs = union (if memSet xs Inc
                    then makeSet [Inc, Eq]
                    else empty)
                    (if memSet xs Eq || memSet xs Dec
                    then sing Dec
                    else empty)


{-- 
  Size change over a number of variables

  Let xs = [x_1, ..., x_n] be the list of variables for a given function f:
  Any SC-value over the function f must have the same arity as xs.
  Let a = [a_1, ..., a_n] be a SC-value over f,
  then a_i descripes the size-change relative to the variable x_i in f.
  The a_i corresponds to a size-change arc from the variable x_i to
  the value described by a, annotated with a_i.

  Ex. 
    let xs = [x, y], the SC-value [{Inc, Dec}, {Eq, Dec}] describing the value
    v corresponds to the size-change graph:

       -+
    x -------> v      
            / 
       -=  /
    y-----/

    Where -+ is short hand notation for {Dec, Inc}, etc.
            
  
--}
type SCval = [SCarc]

makeBot :: Int -> SCval
makeBot n = replicate n empty

makeTop :: Int -> SCval
makeTop n = replicate n (makeSet [Inc, Eq, Dec])

-- Meet is GLB
meet :: SCval -> SCval -> SCval
meet v1 v2 = map (uncurry inter) (zip v1 v2)

-- Join is LUB
join :: SCval -> SCval -> SCval
join v1 v2 = map (uncurry union) (zip v1 v2)

inc :: SCval -> SCval
inc v = map scarcInc v

dec :: SCval -> SCval
dec v = map scarcDec v

fix f v = 
  let fixit f v o = if v==o then v else fixit f (f v) v
  in fixit f (f v) v

-- Take fixpoint since fx. a Dec in a compose may represent one or more Dec's
scabsComposeBase arg Inc = fix inc arg
scabsComposeBase arg Eq  = arg
scabsComposeBase arg Dec = fix dec arg

scarcCompose n s arg = 
  foldSet join (makeBot n)  (mapSet (scabsComposeBase arg) s)

-- compose n f as: computes the size-change graph for the 
--  call to a function with SCG f 
--  given the SCG's for the arguments in as
--  n is the arity of the SCval (used for computing top, bot)
compose :: Int -> SCval -> [SCval] -> SCval
compose n f as = 
  foldl
    join
    (makeBot n)
    (map
      (\(s, arg)->scarcCompose n s arg)
      (zip f as))



ppScarc a v =
  if      subSet (makeSet [Dec, Eq, Inc]) a then v++"T "
  else if subSet (makeSet [Dec, Eq     ]) a then v++"-= "
  else if subSet (makeSet [Dec,     Inc]) a then v++"-+ "
  else if subSet (makeSet [Dec         ]) a then v++"-  "
  else if subSet (makeSet [     Eq, Inc]) a then v++"+= "
  else if subSet (makeSet [     Eq     ]) a then v++"=  "
  else if subSet (makeSet [         Inc]) a then v++"+  "
  else                                           v++"_  "

ppScval v xs = 
  "[" ++concat (map (uncurry ppScarc) (zip v xs)) ++ "]"




