module Graphs where

import List
import Maybe
import Env

type Param = Int


type Arc = (Param, R, Param)

--Nodes are vertexes of the hopefully contained graph.
type Arcs = [Arc]

-- calls are given numbers
-- as they appear in the program, left-right, top-down.
--
type Letter = Int

-- A sequence of such numbers will identify
-- loops that may cause non-termination 
--
type Word = [Letter] -- to trace the way a graph is composed

data R = Gt
       | Gte
       | Lt
	 deriving (Eq, Ord)

composR Gte Gte = Gte
composR Gte Gt  = Gt
composR Gt  Gte = Gt
composR Gt  Gt  = Gt

instance Show R where {
show r = case r of 
	      Gt -> "->"
	      Gte -> "=>"
              Lt  -> "<"}

class Eq a => Composable a where
     composG :: a -> a -> a
     normaliseG :: a -> a
     canCompose :: a -> a -> Bool
     loopingArc :: a -> Bool

normalise :: Composable a => [a] -> [a]
normalise = map normaliseG

-- all possible compositions between graphs in gs1 and gs2
allcomps ::  Composable a => [a] -> [a] -> [a]
allcomps gs1 gs2 = 
         [composG g1 g2 | g1<-gs1, g2<-gs2, canCompose g1 g2]


-- fixedpoint iteration:
-- add elements to a set emerging from s0 with f on the last change.
-- s0 U f(s0) U f(f(s0)) U ... U (f^n)(s0)
-- such that (f^(n-1))(s0) equals (f^n)(s0).
-- To use ++ below is like `union` because new and acc has no common elements
fixedpoint :: Eq a => ([a] -> [a]) -> [a] -> [a]
fixedpoint f s0 = let af acc last = case nub(f last) \\ acc of
                                    [] -> acc
                                    new -> af (new++acc) new
                      in af s0 s0



--                            Compute S
--            i.e. all the distinct finite compositions 
--              of automata graphs in the program.
s :: Composable a => [a] -> [a]
s gs = let gs' = (nub.normalise) gs
           in fixedpoint (normalise.allcomps gs') gs'

--useful for printing
combine sep strlst = foldr (\ x acc -> x ++ sep ++ acc) [] strlst
interleave sep (x:xs) = foldr (\ x acc -> x ++ sep ++ acc) x xs
interleave sep [] = []

rank lst a = case elemIndex a lst of 
                   Nothing -> error ("Rank error:"++show a++" not found in "++show lst)
                   Just i -> i

showArc (s, r, d) = (show s)++' ':(show r)++' ':(show d) 
showArcs :: Arcs -> String
showArcs [] = "-\n"
showArcs (a:b) = (showArc a)++'\n':(showArcs b)

arcOfLine :: String -> Arc
arcOfLine str = (source, r, dest) where
          r = case rstr of
                  "->" -> Gt
                  "=>" -> Gte
          source = read sstr
          dest = read dstr
          [sstr, rstr, dstr] = words str

arcsOfLines :: [String] -> Arcs
arcsOfLines ls = map arcOfLine ls

verifyArc :: (Int, Int) -> Arc -> Arc
verifyArc (n,m) arc@(a, c, d) = if (a < n) && d < m then arc else error "Invalid arc!"

--reading and printing out of graphs using the set notation used by Amir Ben Amram's SCT/P analyzer.
takeParenMatch :: String -> (String, String)
takeParenMatch str =
    let aux str count acc =
          case (str, count) of
            ('(':xs,_) -> aux xs (count+1) ('(':acc)
            (')':xs,1) -> (reverse (')':acc), xs)
            (')':xs,_) -> aux xs (count-1) (')':acc)
            (x:xs,_) -> aux xs count (x:acc)
            ([],_) -> error "Ran out of string before seeing a closen parenthesis"
    in case str of 
        '(':_-> aux str 0 []
        _ -> error "takeParenMatch expects a paren at the start"

breakByParens str =
    case str of
      [] -> []
      '(':xs -> let (a, rest) = takeParenMatch str in
                    a:(breakByParens rest) 
      _:xs -> breakByParens xs

arcOfACL :: String -> (String, R, String)
arcOfACL str =
    case words str of
      ('(':s):d:"DEQ)":[] -> (s, Gte, d) --the final value
      ('(':s):d:"DEC)":[] -> (s, Gt, d)
      --_ -> error (concat words)

breakACLGraph str =
    let aux str acc =
          case str of
            '(':xs -> (reverse acc, xs)
            "NIL)" -> (reverse acc, [])
            x:xs -> aux xs (x:acc)
            [] ->  (reverse acc, [])
        in case str of
            '(':rest -> 
                let (header, footer) = aux rest [] 
                    arcs = breakByParens footer in
                    case words header of 
                        f:g:[] -> Just (f,g,map arcOfACL arcs)
                        _ -> Nothing
            _ -> Nothing


arcsOfSet :: [String] -> [(String, R, String)]
arcsOfSet words =
    case words of
      ('{':x):xs -> arcsOfSet (x:xs) -- Sentinal for the start of a set.
      ('(':s):d:"deq)":xs -> (s, Gte, d):(arcsOfSet xs)
      ('(':s):d:"dec)":xs -> (s, Gt, d):(arcsOfSet xs)
      ('(':s):d:"deq)})":[] -> [(s, Gte, d)] --the final value
      ('(':s):d:"dec)})":[] -> [(s, Gt, d)]
      ["nil)"] -> []
      ["})"] -> []
      --_ -> error (concat words)


--namesOfArcs :: [String] -> ([String], [String])
namesOfArcs arcs  = 
    --let arcs = arcsOfSet x in
    (nub (map (\ (a,_,_) -> a) arcs),nub (map (\ (_,_,a) -> a) arcs))


paramsOfLine :: [String] -> Maybe ((String, [String]), (String, [String]))
paramsOfLine (('(':f):g:('{':arc):arcs) = Just ((f, nub fparms), (g, nub gparms)) 
            where 
               (fparms, gparms) = (namesOfArcs . arcsOfSet) (('{':arc):arcs)
paramsOfLine (('(':f):g:d:('{':arc):arcs) = Just ((f, nub fparms), (g, nub gparms)) 
            where 
               (fparms, gparms) = (namesOfArcs . arcsOfSet) (('{':arc):arcs)
paramsOfLine (('(':f):g:('(':arc):arcs) = Just ((f, nub fparms), (g, nub gparms)) 
            where 
               (fparms, gparms) = (namesOfArcs . arcsOfSet) (('{':arc):arcs)

paramsOfLine _ = Nothing

paramsOfACLLine :: String -> Maybe ((String, [String]), (String, [String]))
paramsOfACLLine str = 
    case breakACLGraph str of 
        Nothing -> Nothing
        Just (f,g,arcs) -> 
            let (fparms, gparms) = namesOfArcs arcs in
            Just ((f, nub fparms), (g, nub gparms))

paramsOfACLSet :: [String] -> [(String, [String])]
paramsOfACLSet set =
     let (sParams, tParams) = unzip (mapMaybe paramsOfACLLine  set)
         paramsMap = union (nub sParams) (nub tParams)
         foldfun (s, params) acc = case lookup s acc of
              Nothing -> (s, params):acc
              Just a -> update s (union a params) acc in
         foldr foldfun [] paramsMap

paramsOfSet :: [String] -> [(String, [String])]
paramsOfSet set =
     let (sParams, tParams) = unzip (mapMaybe (paramsOfLine . words) set)
         paramsMap = union (nub sParams) (nub tParams)
         foldfun (s, params) acc = case lookup s acc of
              Nothing -> (s, params):acc
              Just a -> update s (union a params) acc in
         foldr foldfun [] paramsMap

--paramsOfACL :: [String] -> [(String, [String])]
--paramsOfACL set =
 --   let (sParams, tParams) = unzip (mapMaybe (paramsOfACLLine) set)
          
convertParam ::  [(String, [(String)])] -> String -> String -> Param
convertParam nm f var =
      case lookup f nm of
        Just pmap ->  (rank pmap var) 
        _ -> error ("Tried to look up a "++f++" that didn't exist!")

convertNames :: [(String, [(String)])] -> String -> Param
convertNames nm f =
      case findIndices ((==f).fst) nm of 
        [] -> error ("convertName error:"++show f++" not found in "++show nm)
        [i] -> i
        lst -> error ("convertName error:"++show f++" found too often in "++show nm++" namely "++show lst)

setStringOfArcs as = combine " " (map (setStringOfArc) as)

setStringOfArc (s, d, t) = ("("++show s++" "++show t++if d == Gte then " deq)" else " dec)")


nocomments strs =
     filter f strs
     where
       f str = case dropWhile (`elem` delim) str of
                "" -> False
                '#':_ -> False
                _ -> True
       delim = [' ', '\t']

intercalate x lst = concat (intersperse x lst)


revdetTarget :: Param -> Arcs -> Bool
revdetTarget target arcs = 2 > (length (filter (\ (a,b,c) -> c==target) arcs))

revdetArcs :: Arcs -> Bool
revdetArcs arcs = 
    let targets = nub (map (\ (a,b,c) -> c) arcs) in
    all (\tar -> revdetTarget tar arcs) targets
