module MCGraphs where

import List
import Asyntax
import Graphs
import Maybe
import Data.List.Split
import Scgraph
import Control.Exception
import Debug.Trace

--type Funid = String
--type Callid = Letter
--type Path = Word  -- to track the way a graph is composed

--data Scgraph = G Int Int Arcs Path
--data NamedScgraph = NG Funid Funid Arcs Path

--data Scgraphs =   Vec ([Scgraph], [(Int, Int)]) 
--                | Limited ([Scgraph], [(Int, Int)], [Int]) 
--                | Named ([NamedScgraph], (Program, NCallgraph))
--                | NamedLimited ([NamedScgraph], [(String, Int)], [String])
--                | NamedVec ([NamedScgraph], [(String, Int)])
--                deriving (Show, Eq)


data MCR = EqualTo
        | GreaterEqual
        | GreaterThan
     deriving (Eq, Ord)


instance Show MCR where {
show r = case r of 
            GreaterThan -> ">"
            GreaterEqual -> ">="
            EqualTo  -> "="}


type MCArc = (String, MCR, String)
type MCArcs = [MCArc]


compareArcs (s, r, d) (s', r', d') = compare (r, s, d) (r', s', d')

showMCArc (s, r, d) = (s)++(show r)++(d) 
showMCArcs :: MCArcs -> String
showMCArcs arcs = "["++(intercalate ", " (map showMCArc arcs))++"]"
displayMCArcs = putStrLn.showMCArcs

--assert True a = a
--assert False _ = error "Failure."

composeMCR r EqualTo = r
composeMCR EqualTo r = r 
composeMCR GreaterEqual GreaterEqual = GreaterEqual
composeMCR GreaterEqual GreaterThan = GreaterThan
composeMCR GreaterThan GreaterEqual = GreaterThan
composeMCR GreaterThan GreaterThan = GreaterThan

addSymmetricToMCArcs :: MCArcs -> MCArcs
addSymmetricToMCArcs arcs =
    (nub) (arcs ++ [(target, EqualTo, source) | (source, r, target) <- arcs, r==EqualTo])

isReflexive (source, r, target) = source == target
    --if source == target then (if (r /= GreaterThan) then id else trace (show (source, r, target))) True else False
        
-- The fixedpoint in Graphs.hs requires a piecewise function. This does not.
safeFixedpoint :: Eq a => ([a] -> [a]) -> [a] -> [a]
safeFixedpoint f s0 = let af acc last = case nub(f last) \\ acc of
                                    [] -> acc
                                    new -> af (new++acc) (new++acc)
                      in af s0 s0

addEqImpsToMCArcs :: MCArcs -> MCArcs
addEqImpsToMCArcs  arcs =
    let aux arcs =
            let symArcs = addSymmetricToMCArcs arcs 
                equalArcs = map (\ (a,_,b) -> (a,b)) (filter (\ (_,r,_) -> r==EqualTo) symArcs)
                leftReplace = [(eqB, r, target) | (source, r, target) <- symArcs, (eqA, eqB) <- equalArcs, source==eqA]
                rightReplace = [(source, r, eqB) | (source, r, target) <- symArcs, (eqA, eqB) <- equalArcs, target==eqA]
            in nub (symArcs++leftReplace++rightReplace)
        replaced = safeFixedpoint aux arcs 
        reflexive = filter isReflexive replaced
    in (nub . (sortBy compareArcs)) ((arcs ++ replaced) \\ reflexive)

instance Composable MCArc where
   composG (source, rel, target) (source', rel', target') =
      if target == source' then (source, composeMCR rel rel', target')
                           else error ("incompatible monotonicity constraint arcs:\n " ++
                                      (show (source, rel, target)) ++
                                      " can't be composed with " ++
                                      (show (source', rel', target')) ++ "\n")
   normaliseG a = a
   canCompose (source, rel, target) (source', re', target') = target == source'
   loopingArc (source, rel, target) = source == target

addTrImpsToMCArcs :: MCArcs -> MCArcs
addTrImpsToMCArcs arcs =
    let aux arcs =
            let symArcs = addSymmetricToMCArcs arcs
                composed = [composG a b | a <- symArcs, b <- symArcs, canCompose a b]
            in composed
        implicants = safeFixedpoint aux arcs
        reflexive = filter isReflexive implicants
    in (nub . (sortBy compareArcs)) ((arcs ++ implicants) \\ reflexive)

testImplications x = 
    let arcs=[("x",EqualTo,"y"),("z",GreaterThan,"x"), ("y",EqualTo,"s"), ("s", GreaterEqual, "t"), ("t",EqualTo,"r"), ("q",EqualTo,"r"), ("w",EqualTo,"q")]
        symArcs = addSymmetricToMCArcs arcs
        eqImpArcs = addEqImpsToMCArcs arcs
        arcsb = [("x",EqualTo,"y"),("z",GreaterThan,"x"), ("y",GreaterEqual,"t")]
        trImpArcs = addTrImpsToMCArcs arcsb
   in (assert (elem ("z",GreaterThan,"t") trImpArcs)) (assert (elem ("x", GreaterEqual, "w") eqImpArcs)) x


--instance Show MCR where {
--show r = case r of 
 --          GreaterThan -> ">"
  --         GreaterEqual -> ">="
   --        EqualTo -> "="}

data MCGraph = MC (Funid,[String]) (Funid,[String]) MCArcs deriving Eq

addTrImpsToMCGraphs :: [MCGraph] -> [MCGraph]
addTrImpsToMCGraphs mcgs = 
    let aux (MC src dst arcs) = (MC src dst (addTrImpsToMCArcs arcs)) in
    (map aux mcgs)

addEqImpsToMCGraphs :: [MCGraph] -> [MCGraph]
addEqImpsToMCGraphs mcgs = 
    let aux (MC src dst arcs) = (MC src dst (addEqImpsToMCArcs arcs)) in
    map aux mcgs

instance Show MCGraph where { -- we don't show their paths by default
show (MC (f,fparms) (g,gparms) as) =
    let showParms parms = "("++(intercalate ", " parms) ++ ")" 
    in f++(showParms fparms)++" :- "++g++(showParms gparms)++": "++showMCArcs as++"\n" }

cleanStr :: String -> String
cleanStr str = filter (\x -> not (elem x "( )[];.")) str

cleanSplitOn :: String -> String -> [String]
cleanSplitOn a b = filter (\x -> not (x=="")) (splitOn a b)

functionOfMcString :: String -> (String, [String])
functionOfMcString str =
    let (name, rest) = break (== '(') str
        params = cleanSplitOn "," (cleanStr rest)
    in (cleanStr name, params)
        

arcsOfMcString :: String -> MCArcs
arcsOfMcString str = 
    let arcOfStr str =
            let (target, rest) =  break (\ x -> or [x == '>', x=='=']) str
                (rel, source) = 
                    case rest of
                        '>':'=':name -> (GreaterEqual, name)
                        '>':name -> (GreaterThan, name)
                        '=':name -> (EqualTo, name)
            in (target, rel, source)
        arcStrs = cleanSplitOn "," (cleanStr str)
    in  map arcOfStr arcStrs

mcGraphOfLine :: String -> MCGraph
mcGraphOfLine str = 
    let callerStr:rest:suffix = splitOn ":-" str
        (arcsStr, calleeStr) = (assert (suffix == [])) break (== ';') rest
        (caller, arcs, callee) = (functionOfMcString callerStr, arcsOfMcString arcsStr, functionOfMcString calleeStr)
    in MC caller callee arcs

mcGraphsOfLines :: [String] -> [MCGraph]
mcGraphsOfLines sts =
    map mcGraphOfLine (nocomments sts)

saneParams fparams gparams = and
    [(intersect fparams gparams) == [],
     sort (nub fparams) == sort fparams,
     sort (nub gparams) == sort gparams]
     

scGraphOfMCGraph :: Callid -> MCGraph -> NamedScgraph
scGraphOfMCGraph cid (MC (f,fparams) (g,gparams) arcs) =
    let convRel EqualTo = Gte 
        convRel GreaterEqual = Gte 
        convRel GreaterThan = Gt
        convArc (s, r, t) = 
            case (elemIndex s fparams, elemIndex t gparams) of
                (Just si, Just ti) -> Just (si, convRel r, ti)
                _ -> Nothing
        res = assert ((saneParams fparams gparams)) (NG f g (sort (mapMaybe convArc arcs)) [cid]) 
    in res

saneMap mapping = 
    let names = map fst (nub mapping)
        allOccurs = sort names
        onceOccurs = (nub.sort) names
    in allOccurs == onceOccurs

scGraphsOfMCGraphs :: [MCGraph] -> Scgraphs
scGraphsOfMCGraphs mcgs =
    let gather (MC (f,fp) (g,gp) _) = [(f, length fp),(g, length gp)]
        functions = (nub.concat) (map gather mcgs)
        aux cid mcg = (cid+1, scGraphOfMCGraph cid mcg)
        graphs = snd (mapAccumL aux 0 mcgs)
    in assert (saneMap functions) (NamedVec (graphs, functions))


--addEqImpsToMCArcs :: MCArcs -> MCArcs
--addTrImpsToMCArcs :: MCArcs -> MCArcs
--scGraphsOfMCGraphs :: [MCGraph] -> Scgraphs
--mcGraphsOfLines :: [String] -> [MCGraph]
