module Redox where

import Automata
import SVWGraphs 
import Data.List
import Control.Monad

 -- type State = String
 -- type Start = [State]
 -- type Rho = [(State, Int, State)]
 -- type Final = [State]
 -- data Automaton =  Automaton Int [State] Start Rho Final deriving Show
 -- data Format = Raskin | Deian deriving Eq
 -- type Acgraphs = ([Scgraph], [Funid], [Param])

-- add elements to a set emerging from s0 with f on the last change.
-- s0 U f(s0) U f(f(s0)) U ... U (f^n)(s0)
-- such that (f^(n-1))(s0) equals (f^n)(s0).
-- To use ++ below is like `union` because new and acc has no common elements
mufix :: Eq a => ([a] -> [a]) -> [a] -> [a]
mufix f s0 = let af acc last = case nub(f last) \\ acc of
                                    [] -> acc
                                    new -> af (new++acc) new
                      in af s0 s0

-- This is easier, we simply take a set s0 and compute f^n(s0) such that f^n+1(s0) = f^n(s0)
nufix :: Eq a => ([a] -> [a]) -> [a] -> [a]
nufix f s0 = let new = nub (f s0) in 
             if new \\ s0 == [] then new else nufix f new


pre :: Automaton -> [State] -> [State]
pre (Automaton sigma states start rho final) ss =
                         nub [state | (state, _, s2) <- rho, s2 <- ss]


post :: forall s. (Eq s) => PAutomaton s -> [s] -> [s]
post (Automaton sigma states start rho final) ss =
                         nub [state | (s1, _, state) <- rho, s1 <- ss]

allpreds :: Automaton -> [State] -> [State]
allpreds aut ss =  mufix (pre aut) ss

allposts :: forall a. (Eq a) => PAutomaton a -> [a] -> [a]
allposts aut ss =  mufix (post aut) ss


equal :: Automaton -> Automaton -> Bool
equal aut1 aut2 = (contains aut1 aut2) && (contains aut2 aut1)
    

alphapreds :: Automaton -> [State] -> [State]
alphapreds aut@(Automaton sigma states start rho final) y = 
      let py = pre aut y  in
      let apy = intersect py final in
      allpreds aut apy

isEmpty :: Automaton -> Bool
isEmpty aut@(Automaton sigma states start rho final) = 
    (intersect start (nufix (alphapreds aut) states)) == []

removeDead :: Automaton -> Automaton
removeDead aut@(Automaton sigma states start rho final) = 
   Automaton sigma livestates livestart liverho livefinal
   where
       livestates = [state | state <- states, not (isEmpty (Automaton sigma states [state] rho final))]
       livestart = intersect start livestates
       livefinal = intersect final livestates
       liverho = [(f, c, g) | (f, c, g) <- rho, elem f livestates, elem g livestates]

removeUnreachable :: forall t. (Eq t) => PAutomaton t -> PAutomaton t
removeUnreachable aut@(Automaton sigma states start rho final) = 
   Automaton sigma livestates livestart liverho livefinal
   where
       livestates = allposts aut start
       livestart = intersect start livestates
       livefinal = intersect final livestates
       liverho = [(f, c, g) | (f, c, g) <- rho, elem f livestates, elem g livestates]

testRemove :: Automaton -> Bool
testRemove aut = (equal aut (removeDead aut)) && (equal aut (removeUnreachable aut)) && (equal aut ((removeDead . removeUnreachable) aut))

--To find maxR we need to do calculations on A2d and A1. It seems to me that
--this should be doable using graphs. Graphs are overkill, though: we keep N sets
--of sets. We need only one.

--Lets define A2d directly.

powerset :: [a] -> [[a]]
powerset = filterM (const [True, False])


subsetConstruction :: Automaton -> (PAutomaton [State])
subsetConstruction aut@(Automaton sigma states start rho final) =
        (Automaton sigma pstates [start] rho_d []) 
        where
           pstates = powerset states
           unionRho c ps = foldr (\s acc ->union acc [d | (s, c, d) <- rho]) [] ps
           rho_d = [(s, c, unionRho c s) | s <- pstates, c <- [0..(sigma-1)]]

maxR :: Automaton -> Int
maxR aut =
    let (Automaton _ states _ _ _) = (removeUnreachable . subsetConstruction) aut in
    maximum [length state | state <- states]

-- This is one way of computing maxR(a2|a1). I think, however, it is much larger than necessary.
-- Here we compute the whole product construction.. but we don't care WHICH state in aut1 we are in, only that we are in one.
subsetProductConstruction :: Automaton -> Automaton -> (PAutomaton (State, [State]))
subsetProductConstruction aut1@(Automaton sigma1 states1 start1 rho1 final1)
                          aut2@(Automaton sigma2 states2 start2 rho2 final2) =
        (Automaton sigma1 pstates [(s1, start2) | s1 <- start1] rho_d []) 
        where
           pstates = [(s1, s2) | s1 <- states1, s2 <- powerset states2]
           unionRho c (s,ps) = 
                       let ps' = foldr (\s acc ->union acc [d | (s, c, d) <- rho2]) [] ps 
                       in [(s', ps') | s' <- [state | (s, c, state) <- rho1]] 
           rho_d = [(s, c, s') | c <- [0..(sigma1-1)], s <- pstates, s' <- (unionRho c s)]

maxRP :: Automaton -> Automaton -> Int
maxRP aut1 aut2 =
    let (Automaton _ states _ _ _) = removeUnreachable (subsetProductConstruction aut1 aut2) in
    maximum [length (snd state) | state <- states]

--This has many more states, but we care about many fewer of them.
subsetDualConstruction :: Automaton -> Automaton -> (PAutomaton ([State], [State]))
subsetDualConstruction aut1@(Automaton sigma1 states1 start1 rho1 final1)
                       aut2@(Automaton sigma2 states2 start2 rho2 final2) =
        (Automaton sigma1 pstates [(start1, start2)] rho_d []) 
        where
           pstates = [(s1, s2) | s1 <- powerset states1, s2 <- powerset states2]
           unionRho c (ps1,ps2) = 
                       let ps1' = foldr (\s acc ->union acc [d | (s, c, d) <- rho1]) [] ps1
                           ps2' = foldr (\s acc ->union acc [d | (s, c, d) <- rho2]) [] ps2
                       in (ps1', ps2')
           rho_d = [(s, c, unionRho c s) | c <- [0..(sigma1-1)], s <- pstates]

maxRD :: Automaton -> Automaton -> Int
maxRD aut1 aut2 =
    let (Automaton _ states _ _ _) = removeUnreachable (subsetDualConstruction aut1 aut2) in
    maximum [length (snd state) | state <- states]

--The above is very slow becuase we have to genreate all of rho_d, and thus
--cover all of the powerset. Below inline everything from post to reachableStates to avoid overcomputing.
maxRO :: Either (Automaton, Automaton) Automaton -> Int
maxRO (Left (aut1@(Automaton sigma1 states1 start1 rho1 final1),
             aut2@(Automaton sigma2 states2 start2 rho2 final2))) =
    maximum [length ((snd state)\\final2) | state <- reachableStates]
    where --here we calculate only the needed portions of the dual automaton.
        sigma= max sigma1 sigma2
        start = [(start1, start2)]
        unionRho c (ps1,ps2) = 
                    let ps1' = foldr (\s acc ->union acc [d | (s, c, d) <- rho1]) [] ps1
                        ps2' = foldr (\s acc ->union acc [d | (s, c, d) <- rho2]) [] ps2 
                    in if ps1 == [] then ([], []) else (ps1', ps2') --If we cannot reach any states in aut1, we don't care what we can reach in aut2
        post ss = [unionRho c s | c <- [0..(sigma-1)], s <- ss]
        allposts ss = mufix post ss
        reachableStates = allposts start 

maxRO (Right (aut@(Automaton sigma states start rho final))) = 
    maximum [length (state\\final) | state <- reachableStates]
    where --here we calculate only the needed portions of the dual automaton.
        fRho c ps = [s | (ps, c, s) <- rho]
        post ss = [fRho c s | c <- [0..(sigma-1)], s <- ss]
        allposts ss = mufix post ss
        reachableStates = allposts [start]


simplifyD :: Either (Automaton, Automaton) Automaton -> Either (Automaton, Automaton) Automaton
simplifyD (Left (desc, flow)) = Left (removeDead desc, removeDead flow)
simplifyD (Right aut) = Right (removeDead aut)

simplifyU :: Either (Automaton, Automaton) Automaton -> Either (Automaton, Automaton) Automaton
simplifyU (Left (desc, flow)) = Left (removeUnreachable desc, removeUnreachable flow)
simplifyU (Right aut) = Right (removeUnreachable aut)


simplifyDU :: Either (Automaton, Automaton) Automaton -> Either (Automaton, Automaton) Automaton
simplifyDU (Left (desc, flow)) = Left ((removeDead . removeUnreachable) desc, (removeDead . removeUnreachable) flow)
simplifyDU (Right aut) = Right ((removeDead . removeUnreachable) aut)

simplifyUD :: Either (Automaton, Automaton) Automaton -> Either (Automaton, Automaton) Automaton
simplifyUD (Left (desc, flow)) = Left ((removeUnreachable .removeDead) desc, (removeUnreachable .removeDead) flow)
simplifyUD (Right aut) = Right ((removeUnreachable . removeDead) aut)
