module SVWGraphs where

import List
import Automata
import Graphs
import Env
import Scgraph

--- Reducing automata to graphs.
type SVWGraphs = ([Scgraph], [State], [State], [State])

svwGraphsOfAutomata :: Automaton -> Automaton -> SVWGraphs
svwGraphsOfAutomata (Automaton sigmaA statesA startA rhoA finalA) (Automaton sigmaB statesB startB rhoB finalB) = 
       (graphs, startA, finalA, map num startB)
       where
         graphs = [G source dest (arcs c) [c] | (source, c, dest) <- rhoA]
         arcs c = [(num a, (descB b), num b) | (a, i, b) <- rhoB, i == c] 
         descB state = if elem state finalB then Gt else Gte
         num = rank statesB


maxcount lst = maximum (0:lst)


svwGraphsOfString :: String -> SVWGraphs
svwGraphsOfString str =
       if (format == "3") then (svwGraphsOfAutomata a b) else error "Only arc list containment supported"
       where
         format:lsize:lns = nocomments (lines str)
         (sigma :: Int) = read lsize
         index = case elemIndex "-" lns of
                  Nothing -> error "Invalid file format"
                  Just i -> i+2
         (astr, bstr) = splitAt index lns
         a = automatonOfString sigma astr
         b = automatonOfString sigma bstr


--                   Detect containment


check_lasso :: SVWGraphs -> Bool
check_lasso (gs, a0, af, b0) = let 
                 desc_param xi (n,rel,n') = (rel == Gt) && (n==n') && (any (\(b,_,b')->elem b b0 && b'==n) xi)
                 has_desc_param (xi, xj) = any (desc_param xi) xj 
                 a = s gs
                 b = filter (\(G d _ _ _) -> elem d a0) a
                 graphs = [(ds, fs)| g@(G f f' fs _)<-a, 
                                     c@(G d d' ds _)<-b, elem f af,
                                     f==f', g == normaliseG (composG g g),
                                     f==d', c == normaliseG (composG c g)]
                           in
                           all has_desc_param graphs
                 

-- Here we trade out recaulcating s gs in exchange for, hopefully, not having
-- to keep all of s gs around. Should be slower and use less space.
-- There is also the intermediate option of recomputing s gs once, and keeping
-- only possible starting points (which should be smaller) around. 
-- Inline a in the above to do this.

check_lasso_in_place :: SVWGraphs -> Bool
check_lasso_in_place (gs, a0, af, b0) = let
                     desc_param xi (n,rel,n') = (rel == Gt) && (n==n') && (any (\(b,_,b')->elem b b0 && b'==n) xi)
                     has_desc_param (xi, xj) = any (desc_param xi) xj 
                     graphs = [ (ds, fs) | g@(G f f' fs _)<-s gs, elem f af,
                                           f==f', g == normaliseG (composG g g),
                                           c@(G d d' ds _)<-s gs, elem d a0,
                                           d'==f, c == normaliseG (composG c g)]
                     in
                     all has_desc_param graphs

check_loop :: [Scgraph] -> [State] -> Bool
check_loop gs af = let
                desc_param (n,rel,n') = rel == Gt && n==n'
                has_desc_param = any desc_param
            in all has_desc_param 
                [ as | g@(G f f' as _)<-s gs, f==f', elem f af, g == normaliseG (composG g g) ]
 

contains :: Automaton -> Automaton -> Bool
aut1 `contains` aut2 =  check_lasso (svwGraphsOfAutomata aut1 aut2)
aut1 `test_contains` aut2 =  let (gs,_,af,_) =  (svwGraphsOfAutomata aut1 aut2) in  check_loop gs af

--showing and reading

vec_of_svwgraphs :: [Scgraph] -> [(State, Int)]
vec_of_svwgraphs gs =
             let fnames as = map (\(a,_,_) -> a) as
                 gnames as = map (\(_,_,c) -> c) as
                 rank_of_graph (G f g as _) = 
                     [(f, (length.nub.fnames) as), (g, (length.nub.gnames) as)]
                 allranks = concatMap rank_of_graph gs
                 foldfun (f, rank) acc = case lookup f acc of
                     Just i -> update f (max i rank) acc
                     Nothing -> (f, rank):acc
                 vec = sort (foldr foldfun [] allranks)
             in vec 


stringOfSVWGraphs :: SVWGraphs -> String
stringOfSVWGraphs (gs,as,af,bs) = 
        let numgraphs = length gs
            fParams = vec_of_svwgraphs gs
            fnames = map (\ (x, _) -> x) fParams
            numstates = length fParams 
            parvector = map (\ (_, params) -> params) fParams
            parvecstr = combine " " (map show parvector)
            astartstr = combine " " (map show as)
            afinalstr = combine " " (map show af)
            bstartstr = combine " " (map show bs)
            header = "#SVW Graph Problem\n6\n#Number of states\n" ++ 
                     (show numstates) ++ "\n#No start state\n-1\n#State to node count vector\n" ++ 
                     parvecstr ++ "\n\n#Number of graphs\n" ++ (show numgraphs) ++ 
                     "\n#Starting states in A1\n" ++ astartstr ++
                     "\n#Accepting states in A1\n" ++ afinalstr ++
                     "\n#Starting states in A2\n" ++ bstartstr ++ "\n\n"
            body = showGraphs gs
            in (header ++ body)

setOfSVWGraphs _ = error "SVW Graphs cannot be printed in set notation.\n"

setShow Gt  = " dec "
setShow Gte = " deq "

setStringOfSVWGraph (G f g as _) =
     "("++show f++" "++show g++"{"++setStringOfArcs as++"})"
 

--reading

desc "deq" = Gte
desc "dec" = Gt

svwgraphsOfSet :: [String] -> [Scgraph]
svwgraphsOfSet set = 
  let names = paramsOfSet set 
      param_map = convertParam names
      name_map = convertNames names 
      getArcs f g arcStr =
        let arcs = arcsOfSet arcStr in
        map (\ (a, b, c) -> (param_map f a, b, param_map g c)) arcs
      aux lines i = 
        case lines of
          ('{':_):lns -> aux lns i
          ['}':_] -> []
          x:xs -> case words x of
                    ('(':f):g:d:arcs -> 
                      (G (name_map f) (name_map g) (getArcs f g arcs) [i]):(aux xs (i+1))
                    _ -> error ("Illegal line: " ++ x)
          [] -> error "Graphs ended without seeing closing brace."
  in aux set 0

svwGraphsOfLines :: [String] -> SVWGraphs
svwGraphsOfLines sts = 
  let sts'     = nocomments sts
      getgraphs [] (acc, _) = reverse acc
      getgraphs ("-":strs) (acc, cid) = getgraphs strs (acc, cid)
      getgraphs strs (acc, cid) = 
              let (next, rest) = break (== "-") strs 
                  newgraph = scgraphOfLines next cid
              in getgraphs (rest) ((newgraph:acc), cid+1)
  in case sts' of
      "6":_ ->
         let (["6", nsstr, start, paramstr, ngstr, astr, afstr, bstr], graphs) = splitAt 7 sts'  
             as = map read (words astr)
             af = map read (words afstr)
             bs = map read (words bstr)
         in (getgraphs graphs ([], 0), as, af, bs)
      ('{':_):lines ->
         let 
             paramMap = paramsOfSet sts
             maxP = maximum (map (\ (_, prms) -> length prms) paramMap)
             graphs = (svwgraphsOfSet sts)
             astates =  nub (concatMap (\ (G f g _ _) -> [f,g]) graphs) 
             bstates = nub (concatMap (\ (G _ _ arcs _) -> concatMap (\ (a,_,c) -> [a,c]) arcs) graphs)
         in (graphs, astates, astates, bstates)
      _ ->error "Invalid header"



