module Scgraph where

import List
import Asyntax
import Graphs
import Maybe

type Funid = String

-- calls are given numbers
-- as they appear in the program, left-right, top-down.
--
type Callid = Letter

-- A sequence of such numbers will identify
-- loops that may cause non-termination 
--
type Path = Word  -- to trace the way a graph is composed

data Scgraph = G Int Int Arcs Path
data NamedScgraph = NG Funid Funid Arcs Path

data Scgraphs =   Vec ([Scgraph], [(Int, Int)]) 
                | Limited ([Scgraph], [(Int, Int)], [Int]) 
                | Named ([NamedScgraph], (Program, NCallgraph))
                | NamedLimited ([NamedScgraph], [(String, Int)], [String])
                | NamedVec ([NamedScgraph], [(String, Int)])
                deriving (Show, Eq)

revdetGraphs (Limited (gs,_,_ )) = all revdetGraph gs
revdetGraphs (Vec (gs,_)) =  all revdetGraph gs
revdetGraphs (Named (ngs, _)) = all revdetNGraph ngs
revdetGraphs (NamedLimited (ngs, _, _)) =all revdetNGraph ngs
revdetGraphs (NamedVec (ngs,_)) =all revdetNGraph ngs

toVecGraphs (Limited (gs, parm, fa)) = Vec (gs, parm)
toVecGraphs (Vec a) = Vec a
toVecGraphs (Named (ngs, pgm)) = NamedVec (ngs, vecOfNScgraphs ngs)
toVecGraphs (NamedLimited (gs, parm, fa)) = NamedVec (gs, parm)
toVecGraphs (NamedVec a) = NamedVec a


-- Two graphs are equal iff
-- they have the same sets of vertices and the same arcs.
-- We assume the lists of arcs to be sorted, thereby comparable with (==).
-- Their paths are not compared.
--
instance Eq Scgraph where { 
(==) (G f f' as _) (G g g' as' _) = f == g && f' == g' && as == as' }

instance Eq NamedScgraph where { 
(==) (NG f f' as _) (NG g g' as' _) = f == g && f' == g' && as == as' }

instance Ord Scgraph where { 
(<) (G f _ _ _) (G g _ _ _) = f < g }

instance Ord NamedScgraph where { 
(<) (NG f _ _ _) (NG g _ _ _) = f < g }

--                          Composition of graphs

-- composes the arcs from two sets of arcs
-- duplicate arcs may be created if some of the arc sets are taken from
-- non-size-change graphs - eg if it is a union of arcs between graphs
-- as in G+.

composAs xs ys = let
          as = nub (composAs2 xs ys)
          in sort (filter (keepA as) as)

composAs2 xs ys = 
          [(s, r `composR` r', t') | (s,r,t)<-xs, (s',r',t')<-ys, t==s' ]
keepA as (s,r,t) = 
           and (map (\x->(composR r x) == r) [r' | (s',r',t')<-as, s==s', t==t'])


instance Composable Scgraph where 

   composG (G f g as cs) (G g' h as' cs') | g == g' = 
                        G f h (composAs as as') (cs ++ cs')
   
   composG (G f g _ _) (G g' h _ _) = 
                              error ("incompatible size-chance graphs:\n " ++
                                      (show f) ++ " -> " ++ (show g) ++
                                      " can't be composed with " ++
                                      (show g') ++ " -> " ++ (show h) ++ "\n")

--   normaliseG (G f g arcs path) = G f g (sort(nub arcs)) path
   normaliseG a = a
   canCompose (G _ g _ _) (G g' _ _ _) = g == g'
   loopingArc (G f g _ _) = f==g

instance Composable NamedScgraph where 

   composG (NG f g as cs) (NG g' h as' cs') | g == g' = 
                        NG f h (composAs as as') (cs ++ cs')
   
   composG (NG f g _ _) (NG g' h _ _) = 
                              error ("incompatible size-chance graphs:\n " ++
                                      (show f) ++ " -> " ++ (show g) ++
                                      " can't be composed with " ++
                                      (show g') ++ " -> " ++ (show h) ++ "\n")

   --normaliseG (NG f g arcs path) = NG f g (sort(nub arcs)) path
   normaliseG a = a
   canCompose (NG _ g _ _) (NG g' _ _ _) = g == g'
   loopingArc (NG f g _ _) = f==g



type Callgraph = [(Int, (Int, Int))] 
type NCallgraph = [(Int, (Asyntax.Funname, Asyntax.Funname))]

csOfGraphs :: [Scgraph] -> Callgraph
csOfGraphs gs = 
    zip [0..] (map (\ (G f g _ _) -> (f,g)) gs)

csOfNGraphs :: [NamedScgraph] -> NCallgraph
csOfNGraphs gs = 
    zip [0..] (map (\ (NG f g _ _) -> (f,g)) gs)


-- If all G=(V,V,E) in S where G^2 = G has an arc (n,>,n), then p terminates.
-- result is True if size-change termination is detected, False otherwise
check_idempot :: [Scgraph] -> Bool
check_idempot gs = let desc_param (n,rel,n') = rel == Gt && n==n'
                       has_desc_param = any desc_param
                   in all has_desc_param 
            [ as | g@(G f f' as _)<-s gs, f==f', g == normaliseG (composG g g) ]

check_n_idempot :: [NamedScgraph] -> Bool
check_n_idempot gs = let desc_param (n,rel,n') = rel == Gt && n==n'
                         has_desc_param = any desc_param
                   in all has_desc_param 
            [ as | g@(NG f f' as _)<-s gs, f==f', g == normaliseG (composG g g) ]

check_idempot_in :: [Scgraph] -> [Int] -> Bool
check_idempot_in gs fa = let desc_param (n,rel,n') = rel == Gt && n==n'
                             has_desc_param = any desc_param
                         in all has_desc_param 
            [ as | g@(G f f' as _)<-s gs, f==f', elem f fa, g == normaliseG (composG g g) ]

check_n_idempot_in :: [NamedScgraph] -> [String] -> Bool
check_n_idempot_in gs fa = let desc_param (n,rel,n') = rel == Gt && n==n'
                               has_desc_param = any desc_param
                         in all has_desc_param 
            [ as | g@(NG f f' as _)<-s gs, f==f', elem f fa, g == normaliseG (composG g g) ]

-- check whether g is a critical call sequence
mp_critical g@(G f1 f2 _ _) = (not(scg_desc_param g)) 
                              && f1==f2 && g == normaliseG (composG g g)

scg_desc_param (G _ _ as _) =  (any desc_param) as
  where desc_param (n,rel,n') = rel == Gt && n==n'
                         

-- finds the compositions (cycles) that may cause non-termination


-----

instance Show Scgraph where { -- we don't show their paths by default
show (G f f' as path) = (show f)++'\n':(show f')++'\n':showArcs as++"\n" }

instance Show NamedScgraph where { -- we don't show their paths by default
show (NG f f' as path) = (show f)++'\n':(show f')++'\n':showArcs as++"\n" }

showPath g@(G _ _ _ path) = show g ++ "*" ++ show path ++ "*"


showGraphs :: [Scgraph] -> String
showGraphs graphs = combine "\n" (map show graphs)

--- Reading in and printing out SCGraphs in standard format


scgraphOfLines :: [String] -> Callid -> Scgraph
scgraphOfLines (source:dest:arcs) cid = G (read source) (read dest) (arcsOfLines arcs) [cid]
scgraphOfLines _ _ = error "Attempted to make a graph out of too few lines"


verifyGraph :: [(Int, Int)] -> Scgraph -> Scgraph
verifyGraph pmap (G f g arcs cs) =
      case (lookup f pmap, lookup g pmap) of
      (Just n, Just m) -> G f g (map (verifyArc (n,m)) arcs) cs
      _ -> error "Invalid graph!"

verifyGraphs :: [String] -> [Scgraph] -> [Scgraph]
verifyGraphs ["4", nsstr, start, paramstr, ngstr] graphs =
      let nstates = read nsstr
          ngraphs = read ngstr
          paramvals = map read (words paramstr)
          parammap = zip ([0..nstates]) paramvals in
          if (length graphs == ngraphs) && (length paramvals == nstates) 
          then map (verifyGraph parammap) graphs
          else error "Invalid graphs!"
verifyGraphs (a:_) _ = error ("Format " ++ a ++ " not supported!")
verifyGraphs [] _ = error "empty file!"
      
--amir's format
graphsOfSet :: [String] -> [NamedScgraph]
graphsOfSet set = 
  let names =  paramsOfSet set
      param_map = convertParam names
      getArcs f g arcStr =
        let arcs = arcsOfSet arcStr in
        map (\ (a, b, c) -> (param_map f a, b, param_map g c)) arcs
      unwrap lines =
        case lines of
          ('{':_):lns -> aux lns 0
          ('(':_):lns -> aux lns 0
      aux lines i = 
        case lines of
          ['}':_] -> []
          [')':_] -> []
          x:xs -> case words x of
                    ('(':f):g:arcs -> (NG f g (getArcs f g arcs) [i]):(aux xs (i+1))
                    _ -> error ("Illegal line: " ++ x)
          [] -> []
  in unwrap set

--acl's format
graphsOfACLSet :: [String] -> [NamedScgraph]
graphsOfACLSet set = 
  let names =  paramsOfACLSet set
      param_map = convertParam names
      graphs = mapMaybe breakACLGraph set
      cleanArcs f g arcs =
        map (\ (a, b, c) -> (param_map f a, b, param_map g c)) arcs
      aux graphs i = 
        case graphs of
          (f,g,arcs):xs -> (NG f g (cleanArcs f g arcs) [i]):(aux xs (i+1))
          [] -> []
  in aux graphs 0


-- We return the maximum size of any graph, the starting function, and a map
--from names to the number of parameters they might hold.
scgraphsOfLines :: [String] -> Scgraphs 
scgraphsOfLines sts = 
  let (tag:rest) = (nocomments sts)
      getgraphs [] (acc, _) = reverse acc
      getgraphs ("-":strs) (acc, cid) = getgraphs strs (acc, cid)
      getgraphs strs (acc, cid) = 
              let (next, rest) = break (== "-") strs 
                  newgraph = scgraphOfLines next cid
              in getgraphs (rest) ((newgraph:acc), cid+1)
  in case tag of
      "4" -> 
         let (nsstr:start:paramstr:ngstr:graphs) = rest
             paramvals = map read (words paramstr)
             parammap = zip ([0..(read nsstr)]) paramvals 
         in Vec (getgraphs graphs ([], 0), parammap)
      "5" ->
         let (nsstr:start:paramstr:ngstr:fastr:graphs) = rest
             paramvals = map read (words paramstr)
             parammap = zip ([0..(read nsstr)]) paramvals 
             fa = map read (words fastr)
         in Limited (getgraphs graphs ([], 0), parammap, fa)
      ('(':_) ->
         let 
             paramMap = paramsOfACLSet sts
             maxP = maximum (map (\ (_, prms) -> length prms) paramMap)
             sizeOfF = map (\ (f, x) -> (f, length x)) paramMap
         in NamedVec (graphsOfACLSet sts, sizeOfF)
      ('{':_) ->
         let 
             paramMap = paramsOfSet sts
             maxP = maximum (map (\ (_, prms) -> length prms) paramMap)
             sizeOfF = map (\ (f, x) -> (f, length x)) paramMap
         in NamedVec (graphsOfSet sts, sizeOfF)
      _ ->error "Invalid header"

stringOfScgraphs :: Bool -> Scgraphs -> String
stringOfScgraphs False (Limited (gs, fParams, fa))  =
        let numgraphs = length gs
            parvector = map (\ (_, params) -> params) fParams
            numstates = length fParams 
            parvecstr = combine " " (map show parvector)
            fastr = combine " " (map show fa) 
            header = "#Size Change Graph Problem\n5\n#Number of states\n" ++ (show numstates) ++ 
                     "\n#No start state\n-1\n#State to node count vector\n" ++ parvecstr ++ 
                     "\n\n#Number of graphs\n" ++ (show numgraphs) ++
                     "\n\n#Accepting states\n" ++ fastr ++ "\n\n"
            body = showGraphs gs
            in (header ++ body)

stringOfScgraphs False (Vec (gs, fParams)) =
        let numgraphs = length gs
            parvector = map (\ (_, params) -> params) fParams
            numstates = length fParams 
            parvecstr = combine " " (map show parvector)
            header = "#Size Change Graph Problem\n4\n#Number of states\n" ++ (show numstates) ++ 
                     "\n#No start state\n-1\n#State to node count vector\n" ++ parvecstr ++ 
                     "\n\n#Number of graphs\n" ++ (show numgraphs) ++ "\n\n"
            body = showGraphs gs
            in (header ++ body)

stringOfScgraphs False (NamedLimited a)  = 
     stringOfScgraphs False (Limited (removeNamesAll a))

stringOfScgraphs False (NamedVec a) = 
     stringOfScgraphs False (Vec (removeNamesWithMap a))

stringOfScgraphs False (Named (gs, ((Prog fds), cs))) =
        let gsl = length gs
            fnames = map (\ (FD x _ _) -> x) fds
            parvector = map (\ (FD _ params _) -> length params) fds 
            numgraphs = if gsl == length cs then gsl else error "Improper number of graphs!"
            numstates = length fds 
            parvecstr = combine " " (map show parvector)
            header = "#Size Change Graph Problem\n4\n#Number of states\n" ++ (show numstates) ++ 
                     "\n#No start state\n-1\n#State to node count vector\n" ++ parvecstr ++ 
                     "\n\n#Number of graphs\n" ++ (show numgraphs) ++ "\n\n"
            ftostate name = (rank fnames name) 
            convertGraph (NG f g as cs) = G (ftostate f) (ftostate g) as cs
            body = showGraphs (map convertGraph gs)
            in (header ++ body)

stringOfScgraphs True (Limited (gs, _, fa)) = 
  let fastr = combine " " (map show fa) in
  combine "\n" (["{"]++(map setStringOfGraph gs)++["}","{",fastr,"}"])

stringOfScgraphs True (Vec (gs, _)) = combine "\n" (["{"]++(map setStringOfGraph gs)++["}"])

stringOfScgraphs True (NamedLimited (gs, _, fa)) = 
  let fastr = combine " " fa in
  combine "\n" (["{"]++(map setStringOfNGraph gs)++["}","{",fastr,"}"])

stringOfScgraphs True (NamedVec (gs, _)) = combine "\n" (["{"]++(map setStringOfNGraph gs)++["}"])

stringOfScgraphs True (Named (gs, _)) = combine "\n" (["{"]++(map setStringOfNGraph gs)++["}"])

setStringOfNGraph (NG f g as _) = "(" ++ f ++ " " ++ g ++ " {" ++ setStringOfArcs as ++ "})"
setStringOfGraph (G f g as _) = "(" ++ (show f) ++ " " ++ (show g) ++ " {" ++ setStringOfArcs as ++ "})"

namesInGraphs :: [NamedScgraph] -> [String]
namesInGraphs a = nub (namesInGraphsAux a)

namesInGraphsAux :: [NamedScgraph] -> [String]
namesInGraphsAux ((NG f g _ _):b) = f:g:(namesInGraphs b)
namesInGraphsAux [] = []

removeNames :: Scgraphs -> Scgraphs
removeNames a = case a of
                 Named (ngs, pgs) -> Vec (removeNamesWithMap (ngs, vecOfNScgraphs ngs)) 
                 NamedLimited g -> Limited (removeNamesAll g) 
                 NamedVec g -> Vec (removeNamesWithMap g)
                 _ -> a

removeName :: [String] -> NamedScgraph -> Scgraph
removeName nm (NG f g as n) = G (rank nm f) (rank nm g) as n

removeNamesOfGraphs :: [NamedScgraph] -> [Scgraph]
removeNamesOfGraphs graphs = let names= namesInGraphs graphs in
                      map (removeName names) graphs

removeNamesInMap :: [String] -> [(String, Int)] -> [(Int, Int)]
removeNamesInMap nm pmap = map (\(a,b) -> (rank nm a, b)) pmap

removeNamesWithMap :: ([NamedScgraph],[(String,Int)]) -> ([Scgraph], [(Int,Int)])
removeNamesWithMap (graphs,pmap) = 
          let names = namesInGraphs graphs in
            ((map (removeName names) graphs), removeNamesInMap names pmap)

removeNamesAll :: ([NamedScgraph],[(String,Int)], [String]) -> ([Scgraph], [(Int,Int)], [Int])
removeNamesAll (graphs,pmap, fa) = 
          let names = namesInGraphs graphs in
            ((map (removeName names) graphs), removeNamesInMap names pmap, map (rank names) fa)


--smartCombine :: forall a b . (Ord a, Ord b) => [(a,b)] -> [(a,b)]
smartCombine assoc =
      let sassoc = sort assoc
          aux name elems lst =
              case lst of
              [] -> [(name, sort (nub elems))]
              (a,b):y -> if a == name 
                         then aux name (b++elems) y
                         else (name, sort (nub elems)):(aux a b y)
          in case sassoc of
              [] -> []
              (a,b):y -> aux a b y
          
zipCombine (a:names) assoc@((n,e):rest) = if a == n
                                        then (a,e):(zipCombine names rest)
                                        else (a,[]):(zipCombine names assoc)
zipCombine (a:names) [] = (a,[]):(map (\ name -> (name, [])) names)

zipCombine [] [] = []

zipCombine [] (a:x) = error ("zipCombine failure."++(show a)++(show x))
          

vecOfScgraphs graphs =
     let funcs = sort (nub (concatMap (\(G f g _ _)  -> [f,g]) graphs))
         get_parms (G f g as _) = [(f,nub (map (\ (a,_,_) -> a) as)),(g, nub (map (\ (_,_,b) -> b) as))]
         all_parms = concatMap get_parms graphs
         paramsByFunc = smartCombine all_parms
         funcs_parms = zipCombine funcs paramsByFunc
         vec = map (\ (a,b) -> (a, length b)) funcs_parms
         in vec
         
vecOfNScgraphs graphs =
     let funcs = sort (nub (concatMap (\(NG f g _ _)  -> [f,g]) graphs))
         get_parms (NG f g as _) = [(f,nub (map (\ (a,_,_) -> a) as)),(g, nub (map (\ (_,_,b) -> b) as))]
         all_parms = concatMap get_parms graphs
         paramsByFunc = smartCombine all_parms
         funcs_parms = zipCombine funcs paramsByFunc
         vec = map (\ (a,b) -> (a, length b)) funcs_parms
         in vec

revdetGraph (G _ _ arcs _) = revdetArcs arcs
revdetNGraph (NG _ _ arcs _) = revdetArcs arcs
