module Main where

--Changed by Seth Fogarty: updated to hierarchical name for posix..
--import System.Posix
import System
import List
import Data.List.Split

--import Asyntax
import Debug.Trace
import Parser
import Scgraph
--import Callgraph import Termcheck
--import Annotation
import Unique
import Scc
import CallgraphExtract
--import SizeRelation
--import Sizeanalysis
import Dependency
import Automata
import SVWGraphs
import GroundedGraphs
import SCRed
import MCGraphs 
import Redox
import Pathologies
import Buechi
import Graphs
import System.Console.GetOpt


--data OptLevel = NoOpt | OptOne | OptTwo | OptThree deriving (Show, Eq)
data Input = Unknown |MCGraphs | ScGraphs | SVWGraphs | Automata deriving (Show, Eq)
--data Format = Set | Goal | Raskin | Deian deriving (Eq, Show)
data Options = Options 
    {input :: Input,
     output,
     pathol :: Int,
     format :: Format,
     csv,
     help,
     orig,
     annotate,
     printSet,
     keepStartSets,
     grounded,
     acl :: Bool,
     simpAut :: OptLevel,
     impMC :: OptLevel,
     file :: String,
     asAutomata :: Bool } deriving (Show, Eq)

defaultOptions = Options 
    {input         = Unknown,
     output        = 0,
     pathol        = 0,
     format        = Raskin,
     csv           = False,
     help          = False,
     orig          = False,
     annotate      = False,
     printSet      = False,
     keepStartSets = False,
     grounded      = False,
     acl           = False,
     simpAut       = NoOpt,
     impMC         = NoOpt,
     file          = "",
     asAutomata    = False } 

main = do
    arguments  <- getArgs
    opts       <- processArguments arguments
    --res        <- getSysVar ClockTick 
    --time       <- return $! (res, (timing opts))
    endStr     <- return $! if csv opts then ", " else "\n"
    --timeStart  <- getProcessTimes
    if pathol opts > 0
      then generatePathological opts 
      else if help opts 
        then putStr (usageInfo header options)
        else (case  output opts of
               0 -> solveProblem 
               1 -> printGraphs 
               2 -> printAutomata
               4 -> assertRevDet
               5 -> solveNFA) opts endStr
      

--do

--  (pgm, cs)  <- return $! (annotateCallsites (parse pgmtext))
--  cg         <- return $! (extractCallgraph pgm)
--  revcomps   <- return $! (reverse (sccLSTP (mainFunction pgm) cg))
--  dep        <- return $! (matdep pgm)
--  gam        <-           (extractGammas pgm cg dep revcomps)
--  gs         <- return $! (computeSCG pgm dep gam)
--  return $! (Named (gs, (pgm, cs)))


simplify level problem  =
    case level of
    NoOpt -> problem
    OptOne -> simplifyD problem
    OptTwo -> simplifyU problem
    OptThree -> simplifyUD problem

 
acl_output_split x = case x of 
  '(':xs -> breakByParens xs

removeDOS str = filter (\ x -> not (x=='\r')) str
lines_split text opts = map removeDOS (if acl opts then acl_output_split text else lines text)

solveProblem opts endStr = do
     text       <- readFile (file opts)
     case input opts of
             MCGraphs -> let problem = ((mcGraphsOfLines (lines_split text opts)), impMC opts) in
                              if asAutomata opts then  finish (toAutomata problem) else finish (problem)
             ScGraphs ->  let problem = (scgraphsOfLines (lines_split text opts)) in
                              if asAutomata opts then  finish (toAutomata problem) else finish (problem)
             SVWGraphs ->  let problem = (svwGraphsOfLines (lines_split text opts)) in
                              if asAutomata opts then  finish (toAutomata problem) else finish (problem)
             Automata -> 
                       let problem = ensureContainment (simplify (simpAut opts) (automataOfString text))
                       in if grounded opts 
                          then finish (toScGraphs problem) 
                          else finish problem
     where                 
        finish problem  = do
          ns         <- return $! (solve problem)
          printSolution ns endStr 

--printAutomata x y | trace (show (x)) False = undefined
printAutomata opts endStr= do
      text        <- (readFile (file opts))
      case input opts of
             ScGraphs ->  finish (scgraphsOfLines (lines_split text opts)) 
             SVWGraphs ->  finish (svwGraphsOfLines (lines_split text opts)) 
             Automata -> finish (automataOfString text) 
             MCGraphs -> finish ((mcGraphsOfLines (lines_split text opts)), impMC opts)
      where
        finish problem  = do
          red            <- return $! (simplify (simpAut opts) 
              (if orig opts 
              then Left (reduce problem True) 
              else toAutomata problem))
          --timeRed        <- getProcessTimes 
          rankstr        <- return $! if (annotate opts) then ("#Rank (optional)\n" ++ (show (maxRO red)) ++ "\n") else ""
          --timeEnd        <- getProcessTimes
          --timeString     <- return $! timingString time timeStart timeRed timeEnd
          putStr ((stringOfAutomata (format opts) red)++rankstr)

printGraphs opts endStr= do
      text        <- (readFile (file opts))
      case input opts of
             ScGraphs ->  switch (scgraphsOfLines (lines_split text opts))
             SVWGraphs -> switch (svwGraphsOfLines (lines_split text opts))
             Automata ->  switch (ensureContainment (simplify (simpAut opts) (automataOfString text)))
             MCGraphs -> switch ((mcGraphsOfLines (lines_split text opts)), impMC opts)
      where
        switch problem = 
             let graphs = toScGraphs problem in
             case (grounded opts, keepStartSets opts) of
                  (_,True) -> error "Not yet implemented" --finish (toScGraphs problem, printSet opts)
                  (True,_) -> finish (graphs, printSet opts) 
                  (False,_) -> finish (toVecGraphs graphs, printSet opts) 
        finish problem  = do
          --res        <- getSysVar ClockTick 
          --time       <- return $! (res, (timing opts))
          --timeEnd        <- getProcessTimes 
          --timeString     <- return $! timingString time timeStart timeEnd timeEnd
          putStr ((toString problem))

assertRevDet opts endStr= do
      text        <- (readFile (file opts))
      case input opts of
             ScGraphs ->  switch (scgraphsOfLines (lines_split text opts))
             SVWGraphs -> switch (svwGraphsOfLines (lines_split text opts))
             Automata ->  switch (ensureContainment (simplify (simpAut opts) (automataOfString text)))
             MCGraphs -> switch ((mcGraphsOfLines (lines_split text opts)), impMC opts)
      where
        switch problem =  
             let graphs = toScGraphs problem in
             do putStr ((if revdetGraphs graphs then "Reverse Determinsitic" else "Not Reverse Deterministic")++endStr)

solveNFA opts endStr= do
      text        <- (readFile (file opts))
      case input opts of
             Automata -> 
                case (simplify (simpAut opts) (automataOfString text)) of
                    Left _ -> error "--nfa switch only valid for single input automaton.\n"
                    Right aut -> switch aut
             _ -> error "--nfa switch only valid for automaton.\n"
      where
        switch aut =  
             do putStr ((if checkFWuniversality aut then "Universal" else "Not Universal") ++endStr)
     
generatePathological opts = do
                 n              <- return $! read (file opts) --cheap hack
                 (pgm, graphs)  <- return $! case pathol opts of
                                       1 -> bitstring n
                                       2 -> tightbitstring n
                                       3 -> loeding n
                                       4 -> compactloeding n
                 putStr (stringOfScgraphs (printSet opts) (Named (graphs, pgm)))

printSolution ::  Bool -> [Char] -> IO ()
printSolution  ns endStr =  
       (putStr ((if ns then "Contained" else "Not Contained")++endStr))

--timingString (res, False) timeStart timeSct timeEnd =  ""
--timingString (res, True) timeStart timeSct timeEnd =  
--                 "\n#Generation/Reduction time: " ++ (timeString timeStart timeSct res)  ++ 
--                 "\n#Analysis  time: " ++ (timeString timeSct timeEnd res)
--
--timeString start end res = 
--  let
--    dt = 0 -- round (toRational ((elapsedTime end) - (elapsedTime start))) 
--    tint = show (div dt res)
--    tfrac = show (mod dt res)
--    n = (length (show (res-1))) - (length tfrac)
--    leadspc = concat (replicate n "0")
--  in tint++"."++leadspc++tfrac++"s"

header = "Size Change Analyzer and Ramsey-Based Automata Containment Checker 1.2:\n\n"++
          "Usage: sctp [options..] file"


options :: [OptDescr (Options -> Options)]
options =
    [ Option ['i'] ["input"]
        (ReqArg (\ d opts -> 
                    case d of
                        "g" -> opts {input=ScGraphs}
                        "a" -> opts {input=Automata}
                        "mc" -> opts {input=MCGraphs}
                        "svw" -> opts {input=SVWGraphs}
                        "acl" -> opts {input=ScGraphs, acl=True}
                        _ -> error ("Unknown input format.\n"++usageInfo header options)

        ) "s")
        ("Input format:\n"++
        "\tg\tSize-change graphs (guessed)\n"++
        "\tacl\tSize-change graphs in ACL2S output format (guessed)\n"++
        "\ta\tAutomata containment (guessed)\n"++
        "\tmc\tMonotonicity constraints (guessed)\n"++
        "\t\t(projected to size-change graphs)\n"++
        "\tsvw\tSVW Graphs")
    , Option ['o']     ["output"]
        (ReqArg (\ d opts -> 
                    case d of
                        "g" -> opts {output=1}
                        "gsn" -> opts {output=1, printSet=True, format=Set}
                        "a" -> opts {output=2}
                        "ad" -> opts {output=2, format=Deian}
                        "asn" -> opts {output=2, format=Set, printSet=True}
                        "agff" -> opts {output=2, format=Goal}
                        "gg" -> opts {output=1, grounded=True}
                        "svw" -> opts {output=1, grounded=True, keepStartSets=True}
                        "smv" -> opts {output=2, format=(SMV AG)}
                        "ismv" -> opts {output=2, format=(SMV Invar)}
                        "fsmv" ->  opts {output=2, format=(SMV Invarspec)}
                        _ -> error ("Unknown output format.\n"++usageInfo header options)

        ) "s")
        ("Instead of solving, output file in format:\n"++
        "\tg\tSize-change graphs\n"++
        "\tgsn\tSize-change graphs in set notation (SCT/SCP format)\n"++
        "\ta\tAutomata (rank format)\n"++
        "\tad\tAutomata (adjacency matrix instead of list)\n"++
        "\tasn\tAutomata in set notation (ramsey format)\n"++
        "\tagff\tAutomata in GOAL's gff format\n"++
        "\tgg\tGrounded graphs for automata containment (accepting subset)\n"++
        "\tsmv\tA nuSMV file for NFA universality.\n"++
        "\tfsmv\tA nuSMV file for on-the-fly NFA universality.\n"++
        "\tismv\tA nuSMV file for NFA universality, using INVAR instead of [INVAR]SPEC.\n")
        --"\tsvw\tGrounded graphs (initial/accepting subsets)")
    , Option [] ["revdet"] 
        (NoArg (\ opts -> opts { output = 4}))
        "Instead of solving, test size-change graphs for reverse determinism."
    , Option [] ["nfa"]
        (NoArg (\ opts -> opts { output=5 }))
        "Check finite word universality (subset construction), for one automaton only."
    , Option ['l'] ["LJB"]
        (NoArg (\ opts -> opts { orig = True }))
        "When reducing SCT to automata, use LJB's original reduction"
    , Option ['r'] ["reduce"]
        (NoArg (\ opts -> opts { asAutomata = True }))
        "When solving SCT problems, reduce to automata first"
    , Option ['g'] ["grounded"]
        (NoArg (\ opts -> opts { grounded = True }))
        "When solving automata containment, used grounded graphs"
    , Option ['c'] ["csv"]
        (NoArg (\ opts -> opts { csv= True }))
        "When solving, print a comma instead of a newline"
    , Option ['m'] ["maxrank"]
        (NoArg (\ opts -> opts { annotate= True }))
        "When printing automata (in rank format), annotate with bound on rank"
    , Option ['h','?'] ["help"]
        (NoArg (\ opts -> opts { help= True }))
        "Print this message"
    , Option ['S']     ["simplify"]
        (ReqArg (\ d opts -> case d of
                    "0" -> opts {simpAut=NoOpt }
                    "1" -> opts {simpAut=OptOne }
                    "2" -> opts {simpAut=OptTwo }
                    "3" -> opts {simpAut=OptThree }
                    _ -> error ("Invalid simplification level.\n"++usageInfo header options)) "n")
          ("Prune automata of: 1 - unreachable states; 2 - dead states; 3 - both")
    , Option ['I']     ["addimps"]
        (ReqArg (\ d opts -> case d of
                    "0" -> opts {impMC=NoOpt }
                    "1" -> opts {impMC=OptOne }
                    "2" -> opts {impMC=OptTwo }
                    "3" -> opts {impMC=OptThree }
                    _ -> error ("Invalid implication.\n"++usageInfo header options)) "n")
          ("Add implied arcs to monotonicity constraints before projecting:\n"++
          "\t\t1 - Equality implications; 2 - Transitive implications")
    , Option [] ["pb"]
        (ReqArg (\ n opts -> opts { pathol = 1, file = n }) "n")
        "Produce bitstring-based pathological graphs of size n"
    , Option [] ["pbt"]
        (ReqArg (\ n opts -> opts { pathol = 2, file = n }) "n")
        "Produce tight bitstring-based pathological graphs of size n"
    , Option [] ["pl"]
        (ReqArg (\ n opts -> opts { pathol = 3, file = n }) "n")
        "Produce graphs based on Loeding's pathological automata of size n"
    , Option [] ["plc"]
        (ReqArg (\ n opts -> opts { pathol = 4, file= n }) "n")
        "Produce graphs based on compact version of Loeding's automata (n)"
    ]          

--processArguments = cleanOptions . (foldr processArgument defaultOptions)
processArguments args = 
    let (o,f,errs) = getOpt Permute options args
        opts = foldl (flip id) defaultOptions o 
        opts' = case (f,errs) of 
            ([fname],[]) -> opts {file = fname}
            ([],[]) -> if (pathol opts > 0)  then opts else opts {help=True}
            (_,errs) -> (error (concat errs ++ usageInfo header options))
    in cleanOptions opts'

cleanOptions opts = 
     if ((file opts) == "") || (help opts) || (pathol opts > 0)
     then return $! opts
     else do
            filetext      <- readFile (file opts)
            firstline <- return $! head (nocomments (lines filetext))
            if (input opts == Unknown)
             then case firstline of
                    '{':_ -> return $! opts{input=ScGraphs} 
                    '(':'(':_ -> return $! opts{input=ScGraphs, acl=True} 
                    --at the moment we don't properly detect annotated set graphs
                    "6" -> return $! opts{input=SVWGraphs}
                    "5" -> return $! opts{input=ScGraphs}
                    "4" -> return $! opts{input=ScGraphs}
                    "3" -> return $! opts{input=Automata}
                    "2" -> return $! opts{input=Automata}
                    _ ->   return $! if isSuffixOf ".mc" (file opts) 
                                      then opts{input=MCGraphs}
                                      else opts{input=ScGraphs}
             else return $! opts
