{--
  Set implementation
  ------------------
  Requires an ordering on the elements

  Simon Thompson, 1999
  Haskell: The Craft of Functional Programming - 2nd ed.
 
  Except: 
    Added:   show 
    Removed: showSet
    NOTE:    diff not implemented (in the book)

module Set
( Set,
  empty,             -- Set a
  sing,              -- a -> Set a
  memSet,            -- Ord a=> Set a -> a -> Bool
  union,inter,diff,  -- Ord a => Set a -> Set a -> Set a
  eqSet,             -- Eq a => Set a -> Set a -> Bool
  subSet,            -- Ord a => Set a -> Set a -> Bool
  makeSet,           -- Ord a => [a] -> Set a
  mapSet,            -- Ord b => (a -> b) -> Set a -> Set b
  filterSet,         -- (a->Bool) -> Set a -> Set a
  foldSet,           -- (a -> a -> a) -> a -> Set a -> a
  card               -- Set a -> Int
  selectSet          -- Set a -> (a, Set a)
  zipSet             -- Set a -> [b] -> Set (a, b)
)
--}
module Set where

import List hiding ( union )

instance Eq a => Eq (Set a) where
  (==) = eqSet
instance Ord a => Ord (Set a) where
  (<=) = leqSet

newtype Set a = SetI [a]

empty :: Set a
empty = SetI []

sing :: a-> Set a
sing x = SetI [x]

memSet :: Ord a => Set a -> a -> Bool
memSet (SetI []) y = False
memSet (SetI (x:xs)) y
  | x<y       = memSet (SetI xs) y
  | x==y      = True
  | otherwise = False

union :: Ord a => Set a -> Set a -> Set a
union (SetI xs) (SetI ys) = SetI (uni xs ys)

uni :: Ord a => [a] -> [a] -> [a]
uni [] ys = ys
uni xs [] = xs
uni (x:xs) (y:ys)
  | x<y       = x : uni xs (y:ys)
  | x==y      = x : uni xs ys
  | otherwise = y : uni (x:xs) ys

inter :: Ord a => Set a -> Set a -> Set a
inter (SetI xs) (SetI ys) = SetI (int xs ys)

int :: Ord a => [a] -> [a] -> [a]
int [] ys = []
int xs [] = []
int (x:xs) (y:ys)
  | x<y       = int xs (y:ys)
  | x==y      = x : int xs ys
  | otherwise = int (x:xs) ys

subSet :: Ord a => Set a -> Set a -> Bool
subSet (SetI xs) (SetI ys) = subS xs ys

subS :: Ord a => [a] -> [a] -> Bool
subS [] ys = True
subS xs [] = False
subS (x:xs) (y:ys)
  | x<y  = False
  | x==y = subS xs ys
  | x>y  = subS (x:xs) ys

diff :: Ord a => Set a -> Set a -> Set a
diff (SetI xs) (SetI ys) = SetI (dif xs ys)
dif [] ys = []
dif xs [] = xs
dif xx@(x:xs) yy@(y:ys)
  | x<y  = x : (dif xs yy)
  | x==y = dif xs ys
  | x>y  = dif xx ys

eqSet :: Eq a => Set a -> Set a -> Bool
eqSet (SetI xs) (SetI ys) = (xs == ys)

leqSet :: Ord a => Set a -> Set a -> Bool
leqSet (SetI xs) (SetI ys) = (xs <= ys)

makeSet :: Ord a => [a] -> Set a
makeSet = SetI . remDups . sort

remDups []    = []
remDups [x]   = [x]
remDups (x:y:xs)
  | x<y       = x : remDups (y:xs)
  | otherwise = remDups (y:xs)

mapSet :: Ord b => (a -> b) -> Set a -> Set b
mapSet f (SetI xs) = makeSet (map f xs)

filterSet :: (a -> Bool) -> Set a -> Set a
filterSet p (SetI xs) = SetI (filter p xs)

foldSet :: (a -> a -> a) -> a -> Set a -> a
foldSet f x (SetI xs) = (foldr f x xs)

card :: Set a -> Int
card (SetI xs) = length xs

selectSet :: Set a -> (a, Set a)
selectSet (SetI (x:xs)) = (x, SetI xs)

zipSet (SetI a) b = SetI (zip a b)

instance Show a => Show (Set a) where
  show = (\(SetI xs)->"{" ++
           case xs of
             []        -> ""
             otherwise -> (\(x:xs)-> 
                            foldl (\x->(\y->x ++ ", " ++ show y)) (show x) xs) 
                          xs
           ++ "}")
