#
# review-sanity-check.pl
# Dan Wallach <dwallach@cs.rice.edu>
#
# This reads in everything and prints a summary of what it's found.
# If there are bugs in the input (maybe a referee left the rankings
# as "1-7" rather than putting a concrete number there), you'll
# see the program gripe about bogus input.  Investigate these.
# Your goal is to make sure the input phase runs cleanly.  After
# this, the program prints output in three phases.  First, you
# get, for each paper, who has submitted reviews (along with their
# self-rating), who you're still waiting for, and who unexpectedly
# submitted a review for something.  Look carefully to see if you
# have multiple spellings of the same reviewer's name (which would
# tend to show up by having an entry for the same reviewer both
# in the submitted and incomplete sections).  The next two sections
# summarize, for each reviewer, how many reviews they've written
# and what reviews they have remaining.
#
# TODO: add some checks to catch papers with no reviews of high confidence
#

use usenixDB;

usenixDB::readReviews;

$maxReviews = 2;   # only list paper summaries with this many of fewer reviews
$n=0;

# TODO: add logic to look for papers with no reviews of confidence 2 or higher

foreach $id (@usenixDB::ids) {
    @submitted = sort keys %{$usenixDB::reviewDB{$id}{"Submitted"}};
    @unexpected = sort keys %{$usenixDB::reviewDB{$id}{"Unexpected"}};
    @incomplete = sort keys %{$usenixDB::reviewDB{$id}{"Incomplete"}};
    
    $printMe = @submitted <= $maxReviews;
    
    printf  "$id\n" if $printMe;
    printf  "  Submitted: %s\n", join(",", map {$_."(".$usenixDB::reviewDB{$id}{"Self"}{$_}.")";} @submitted) if @submitted && $printMe;
    printf  " Unexpected: %s\n", join(",", @unexpected) if @unexpected && $printMe;
    printf  " Incomplete: %s\n", join(",", @incomplete) if @incomplete && $printMe;
    
    # more sanity checking -- catches misspellings of reviewer names
    foreach my $r (@{$usenixDB::idToReviewers{$id}}) {
	printf "     : Reviewer %s should be assigned paper %s but isn't!\n", $r, $id
	    unless $usenixDB::reviewDB{$id}{"Assigned"}{$r};
    }
    
    foreach $r (@incomplete) {
	$inc{$r}{$id} = 1;
    }
    
    $n++ if $printMe;
}

printf "Papers listed: $n\n";

printf "\nIncomplete review summaries\n";
printf "---------------------------\n";

foreach $r (sort keys %inc) {
    printf "$r: %s\n", join(" ", sort {$a <=> $b} map {s/^L//; $_;} keys %{$inc{$r}});
}

printf "\nReferee review summaries\n";
printf "------------------------\n";

foreach $r (sort keys %usenixDB::numReviewsComplete) {
    printf "%8s: %3d / %3d\n", $r, $usenixDB::numReviewsComplete{$r}, $usenixDB::numReviews{$r};
}
