#
# Copyright (c) 2011, EPFL (Ecole Politechnique Federale de Lausanne)
# All rights reserved.
#
# Created by Marco Canini, Daniele Venzano, Dejan Kostic, Jennifer Rexford
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   -  Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#   -  Redistributions in binary form must reproduce the above copyright notice,
#      this list of conditions and the following disclaimer in the documentation
#      and/or other materials provided with the distribution.
#   -  Neither the names of the contributors, nor their associated universities or
#      organizations may be used to endorse or promote products derived from this
#      software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
# EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
# SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
# PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#

import logging
from ConfigParser import SafeConfigParser
from StringIO import StringIO

default_config = StringIO("""
[model]
name = LoadBalancerModel
app_dir = loadbalancer
app_descr = none
cutoff = -1
faults = 0
flow_entry_expiration = false

[nice_model]
max_pkts = 2
max_burst = 1

[pyswitch_model]
pkts = 2
sequential = False
move_host = False

[eate_model]
connections = 1
app_version = eateb

[loadbalancer_model]
connections = 2
use_fixed_version = false

[twohosts_model]
pkts = 2

[strategy]
dpor = False
name = CompleteCoverage

[randomwalk]
seed = -1

[runtime]
log_level = info
replay_debug = true
test = false
progress = false
graph = none
""")

class NiceConfig:
	def __init__(self, fname):
		self.fname = fname
		parser = SafeConfigParser()
		parser.readfp(default_config)
		parser.readfp(open(fname))
		self.values = {}

		for s in parser.sections():
			for o in parser.options(s):
				self.values[s+"."+o] = parser.get(s, o)

		self.convertTypes()

	def convertTypes(self):
		for k in self.values:
			if k == "runtime.log_level":
				self.values[k] = self.getLogLevel(k)
			elif self.tryToInt(k):
				continue
			elif self.tryToBool(k):
				continue
			elif self.tryToNone(k):
				continue

	def tryToInt(self, k):
		try:
			self.values[k] = int(self.values[k])
		except (ValueError, TypeError):
			return False
		else:
			return True

	def tryToBool(self, k):
		if self.values[k] == "true" or self.values[k] == "True":
			self.values[k] = True
		elif self.values[k] == "false" or self.values[k] == "False":
			self.values[k] = False
		else:
			return False
		return True

	def tryToNone(self, k):
		if self.values[k] == "none" or self.values[k] == "None":
			self.values[k] = None
		else:
			return False
		return True

	def get(self, key):
		return self.values[key]

	def set(self, key, value):
		self.values[key] = value

	def getLogLevel(self, k):
		name = self.get(k)
		if name in [logging.DEBUG, logging.INFO, logging.WARNING, logging.ERROR, logging.CRITICAL]:
			return name
		elif name == "debug":
			return logging.DEBUG
		elif name == "info":
			return logging.INFO
		elif name == "warning":
			return logging.WARNING
		elif name == "error":
			return logging.ERROR
		elif name == "critical":
			return logging.CRITICAL
		else:
			return None

