package edu.rice.cs.mint.runtime;


import java.net.URL;
import java.net.MalformedURLException;
import java.io.File;
import java.io.InputStream;
import java.io.IOException;
import java.io.ByteArrayOutputStream;
import java.util.Enumeration;
import java.util.List;
import java.util.LinkedList;
import java.util.HashMap;
import java.util.NoSuchElementException;
import java.lang.reflect.*;

public class HashMapClassLoader extends ClassLoader {
    /** Return the singleton instance of the HashMapClassLoader. Create it if necessary.
      * @return HashMapClassLoader singleton instance */
    public static synchronized HashMapClassLoader instance() {
        return instance("java.class.path");
    }
    
    /** Return the singleton instance of the HashMapClassLoader. Create it if necessary.
      * Set the name of the Java property containing the class path.
      * @return HashMapClassLoader singleton instance */
    public static synchronized HashMapClassLoader instance(String classPathPropertyName) {
        if (_instance==null) _instance = new HashMapClassLoader(classPathPropertyName);
        return _instance;
    }    
    
    /** Return the singleton instance of the HashMapClassLoader and add the classes in the
      * class table to it. Create the HashMapClassLoader if necessary.
      * @param classTable classes to add to the HashMapClassLoader
      * @return HashMapClassLoader singleton instance */
    public static synchronized HashMapClassLoader instance(HashMap<String,byte[]> classTable) {
        HashMapClassLoader hmcl = instance();
        hmcl.add(classTable);
        return hmcl;
    }
    
    /** Private singleton instance. */
    private static volatile HashMapClassLoader _instance = null;

    /** Private singleton constructor. */
    private HashMapClassLoader(String classPathPropertyName) {
        // FIXME: which parent class loader to use?
        // http://www.onjava.com/pub/a/onjava/2005/01/26/classloading.html
        // says we should use HashMapClassLoader.class.getClassLoader(), but
        // that seems to be null; using thread context class loader instead
        // which seems to be the right thing to do according to
        // http://www.javaworld.com/javaworld/javaqa/2003-06/01-qa-0606-load.html
        super(Thread.currentThread().getContextClassLoader());
        _classTable = new HashMap<String,byte[]>();
        // we still need the class path, though, because we pass it to the compiler as a string
        _classPathPropertyName = classPathPropertyName;
        updateClassPathProperty();
    }
    
    protected HashMap<String, byte[]> _classTable;

    public void add(HashMap<String,byte[]> moreClasses) {
        _classTable.putAll(moreClasses);
    }
    
    public byte[] getClassBytes(String name) {
        byte[] classBytes = _classTable.get(name);
        if (classBytes==null) {
            URL resource = findResource(name.replace('.', '/') + ".class");
            if (resource == null) { return null; }
            try {
                classBytes = toByteArray(resource.openStream());
            }
            catch(java.io.IOException ioe) { return null; }
        }
        return classBytes;
    }

    public Class<?> loadClass(String name, boolean resolve) throws ClassNotFoundException {
        return super.loadClass(name, resolve);
    }

    public Class<?> findClass(String name) throws ClassNotFoundException {
        byte[] classBytes = _classTable.get(name);
        if (classBytes==null) {
            URL resource = findResource(name.replace('.', '/') + ".class");
            if (resource == null) { throw new ClassNotFoundException(name); }
            try {
                classBytes = toByteArray(resource.openStream());
                Class<?> c = defineClass(name, classBytes, 0, classBytes.length);
                return c;
            }
            catch (IOException e) { throw new ClassNotFoundException(name); }
        }
        Class<?> c = defineClass(name, classBytes, 0, classBytes.length);
        return c;
    }
    
    private byte[] toByteArray(InputStream in) throws IOException {
        byte[] buffer = new byte[1024];
        int charsRead = in.read(buffer);
        if (charsRead == -1) { return new byte[0]; }
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        try {
            int totalCharsRead = 0;
            do {
                totalCharsRead += charsRead;
                if (totalCharsRead < 0) { totalCharsRead = Integer.MAX_VALUE; }
                out.write(buffer, 0, charsRead);
                charsRead = in.read(buffer);
            } while (charsRead != -1);
            return out.toByteArray();
        }
        finally {
            out.close();
        }
    }
    
    /** Name of the property with the class path, or null. */
    private final String _classPathPropertyName;

    /** Value of the class path property. */
    private String _classPathPropertyValue = null;
    
    /** Return the value of the class path property. */
    public String getClassPathPropertyValue() { return _classPathPropertyValue; }
    
    private void updateClassPathProperty() {
        if (_classPathPropertyName==null) { return; }
        _classPathPropertyValue = System.getProperty(_classPathPropertyName);
        if (_classPathPropertyValue==null) { return; }
    }
}
