/* an implementation of Code<T> that uses strings for java code */

package edu.rice.cs.mint.runtime;

import edu.rice.cs.mint.runtime.mspTree.*;

import java.util.HashMap;
import java.util.ArrayList;
import java.lang.reflect.*;

import edu.rice.cs.mint.comp.com.sun.tools.javac.util.*;


public class MSPTreeCode<T> implements Code<T> {
    /* context for compiling MSPTreeCode objects */
    /* FIXME: do we want a static context? */
    /* public static Context context = new Context (); */

    private MSPTree mspTree;
    private String retTypeName;
    public boolean isStat;

    private static final boolean _mintverbose = (System.getenv("mintverbose")!=null);
    protected static int gensym_counter = 1;

    protected static synchronized String uniqueCodeName () {
        return "$$Code" + (gensym_counter++) + "$$";
    }
    
    private static int var_gensym_counter = 1;
    
    public static String varGenSym(String prefix) {
        return prefix+"$$$"+(var_gensym_counter++);
    }

    public MSPTreeCode(MSPTree mspTree, String retTypeName, boolean isStat) {
        this.mspTree = mspTree;
        this.retTypeName = retTypeName;
        this.isStat = isStat;
    }

    public static <U> MSPTree escape(Code<U> c) {
        try {
            return ((MSPTreeCode<U>)c).mspTree;
        } catch (ClassCastException e) {
            System.err.println ("Internal error in MSPTreeCode: argument in escape is of type " + c.getClass().toString());
            e.printStackTrace (System.err);
            System.exit (1);
            return null; // never reached
        }
    }
    
    public static class ClassInfo {
        public String text;
        public ArrayList<Pair<Object,String>> csp_value_type_table;
        public Object[] csp_table;
        public String[] csp_type_table;
        public HashMap<String,byte[]> bytes_hash;
    }

    ClassInfo generateClass(String class_name) {
        return generateClass(class_name, initContext());
    }
    
    ClassInfo generateClass(String class_name, Context context) {
        // FIXME: not traverse whole array here instead of code
        Pair<String,ArrayList<Pair<Object,String>>> code_csp_pair = mspTree.generateCodeCspPair();
        String meth_code = code_csp_pair.fst;
        
        // separate the CSP values and types
        ClassInfo result = new ClassInfo();
        result.csp_value_type_table = code_csp_pair.snd;
        result.csp_table = new Object[result.csp_value_type_table.size()];
        result.csp_type_table = new String[result.csp_value_type_table.size()];
        for(int i=0; i<result.csp_value_type_table.size(); ++i) {
            result.csp_table[i] = result.csp_value_type_table.get(i).fst;
            result.csp_type_table[i] = result.csp_value_type_table.get(i).snd;
        }

        /* bundle code in a compilation unit */
        StringBuilder full_code = new StringBuilder();
        
        // class definition
        full_code.append("public class ");
        full_code.append(class_name);
        full_code.append(" implements edu.rice.cs.mint.runtime.Code<");
        String actualRetTypeName = retTypeName;
        full_code.append(retTypeName);
        full_code.append("> { \n");
        
        // csp_value_??? fields
        // full_code.append("private final Object [] $csp_table;\n");
        for(int i=0; i<result.csp_type_table.length; ++i) {
            full_code.append("private final ").append(result.csp_type_table[i]).append(" ").append("$csp_value_");
            full_code.append(i).append(";\n");
        }

        // constructor
        full_code.append("public " + class_name);
        full_code.append("(Object[] csp_table) {\n");
        
        // initialize fields
        for(int i=0; i<result.csp_type_table.length; ++i) {
            full_code.append("this.$csp_value_").append(i).append(" = ((");
            full_code.append(result.csp_type_table[i]).append(")").append("csp_table[");
            full_code.append(i).append("]);\n");
        }
        // full_code.append("this.$csp_table = csp_table;\n");
        full_code.append("}\n");

        // run method
        full_code.append("public " + actualRetTypeName + " run() {\n");
        if (isStat) {
            full_code.append("if (true) {\n"); // gets around the "return null unreachable" situation if meth_code throws
            full_code.append(meth_code);
            full_code.append("\n}\nreturn null;\n");
        }
        else {
            full_code.append("return\n");
            full_code.append(meth_code);
            full_code.append(";\n");
        }
        full_code.append("}\n");
        
        // end of class definition
        full_code.append("}");
        
        result.text = full_code.toString();

        /* generate a compiler for the code */
        StringCodeCompiler comp =
            (StringCodeCompiler) StringCodeCompiler.instance(context);
        
        Options options = Options.instance(context);
        options.put("-classpath", HashMapClassLoader.instance().getClassPathPropertyValue());

        /* compile any and all necessary classes */
        result.bytes_hash = comp.compileStringToBytes(class_name, full_code.toString());

        return result;
    }
    
    @SuppressWarnings("unchecked")
    public T run(Context context) {
        /* get a name for our new class */
        /* FIXME: should this be fully qualified? */
        String class_name = uniqueCodeName();

        ClassInfo ci = generateClass(class_name);
        String full_code = ci.text;

        if (_mintverbose) System.out.println (full_code.toString());

        /* generate a compiler for the code */
        StringCodeCompiler comp =
            (StringCodeCompiler) StringCodeCompiler.instance(context);
        
        Options options = Options.instance(context);
        options.put("-classpath", HashMapClassLoader.instance().getClassPathPropertyValue());

        /* compile any and all necessary classes */
        HashMap<String,byte[]> bytes_hash = ci.bytes_hash;
        
//        for(Map.Entry<String,byte[]> e: bytes_hash.entrySet()) {
//            System.out.println(e.getKey()+" - "+e.getValue().length);
//        }

        /* load the resulting class */
//        HashMapClassLoader loader = new HashMapClassLoader (bytes_hash);
        try {
            Code<T> code = (Code<T>)loadClassAndCreateInstance(class_name, bytes_hash, ci.csp_table);
            return code.run();
        }
        catch(InternalError ie) {
            ie.printStackTrace(System.err);
            System.exit(1);
        }
        return null; // unreachable
    }
    
    static Code<?> loadClassAndCreateInstance(String class_name, HashMap<String,byte[]> bytes_hash, Object[] csp_table) {
        /* load the resulting class */
//        HashMapClassLoader loader = new HashMapClassLoader (bytes_hash);
        HashMapClassLoader loader = HashMapClassLoader.instance(bytes_hash);
        try {
            Class<?> cls = loader.loadClass(class_name);

            /* create an instance, invoke run, and return the result */
            Constructor<?> ctor = cls.getConstructor(Object[].class);
            return (Code<?>)ctor.newInstance(new Object [] {csp_table});
        } catch (ClassNotFoundException e) {
            throw new InternalError("Internal error in MSPTreeCode: class " + class_name + " not found", e);
        } catch (InstantiationException e) {
            throw new InternalError("Internal error in MSPTreeCode: class " + class_name + " could not be instantiated", e);
        } catch (IllegalAccessException e) {
            throw new InternalError("Internal error in MSPTreeCode: illegal access for constructor in class " + class_name, e);
        } catch (NoSuchMethodException e) {
            throw new InternalError("Internal error in MSPTreeCode: constructor for " + class_name + " does not exist", e);
        } catch (InvocationTargetException e) {
            throw new InternalError("Internal error in MSPTreeCode: unexpected exception in constructor for class " + class_name, e);
        }
    }

    /** The context for the entire execution of the program.
      * We just have one long compile session, not many short ones. */
    protected static Context context = null;
    
    /** Get the context, if necessary initialize it the first time.
      * This is synchronized to prevent data races. */
    protected static synchronized Context initContext() {
        if (context==null) {
            context = new Context();
            StringCodeCompiler.preRegister(context);
        }
        return context;
    }
    
    public T run() {
        return run(initContext());
    }
    
    public String toString() {
        // FIXME: not traverse whole array here instead of code
        Pair<String,ArrayList<Pair<Object,String>>> code_csp_pair = mspTree.generateCodeCspPair ();
        return "<| "+code_csp_pair.fst+" |>";
    }    
}
