package edu.rice.cs.mint.comp;

import java.io.*;
import java.util.*;

import edu.rice.cs.mint.comp.com.sun.tools.javac.util.*;
import edu.rice.cs.mint.comp.com.sun.tools.javac.util.List;

import edu.rice.cs.mint.comp.com.sun.tools.javac.code.*;
import edu.rice.cs.mint.comp.com.sun.tools.javac.code.Symbol.*;
import edu.rice.cs.mint.comp.com.sun.tools.javac.tree.*;
import edu.rice.cs.mint.comp.com.sun.tools.javac.tree.JCTree.*;
import edu.rice.cs.mint.comp.com.sun.tools.javac.comp.*;
import edu.rice.cs.mint.comp.com.sun.tools.javac.util.JCDiagnostic.DiagnosticPosition;

/** Prints out a tree as an indented Java source program. */
public class MSPTreeCodeSerializer extends Pretty {

    private Symtab syms;
    private TreeMaker make;
    private Names names;
    private Log log;
    private List<VarSymbol> classVarsInScope;
    private final Resolve rs;
    private DiagnosticPosition make_pos;

    /** Environment for symbol lookup, set by translateTopLevelClass.
     */
    Env<AttrContext> attrEnv;
    
    public MSPTreeCodeSerializer(Symtab syms, TreeMaker make, Names names, Log log, Resolve rs,
                                 DiagnosticPosition make_pos, Env<AttrContext> attrEnv,
                                 List<VarSymbol> classVarsInScope) {
        super(null,false); // this makes out in Pretty null, but we override the methods that would use out
        this.syms = syms;
        this.make = make;
        this.names = names;
        this.log = log;
        this.rs = rs;
        this.make_pos = make_pos;
        this.attrEnv = attrEnv;
        
        this.classVarsInScope = classVarsInScope;
    }

    /** The output stream on which trees are printed.
     */
    MSPTreeCodeWriter msp_out;
    
    public java.util.List<Elem> serializeExpr(JCExpression expr) {
        msp_out = new MSPTreeCodeWriter();
        try { printExpr(expr); }
        catch(IOException ioe) { throw new UncheckedIOException(ioe); }
        return msp_out.getElems();
    }
    public java.util.List<Elem> serializeStats(List<JCStatement> stats) {
        msp_out = new MSPTreeCodeWriter();
        try { printStats(stats); }
        catch(IOException ioe) { throw new UncheckedIOException(ioe); }
        return msp_out.getElems();
    }

    /** Align code to be indented to left margin.
     */
    protected void align() throws IOException {
        for (int i = 0; i < lmargin; i++) msp_out.write(" ");
    }

    /** Enter a new precedence level. Emit a `(' if new precedence level
     *  is less than precedence level so far.
     *  @param contextPrec    The precedence level in force so far.
     *  @param ownPrec        The new precedence level.
     */
    protected void open(int contextPrec, int ownPrec) throws IOException {
        if (ownPrec < contextPrec) msp_out.write("(");
    }

    /** Leave precedence level. Emit a `(' if inner precedence level
     *  is less than precedence level we revert to.
     *  @param contextPrec    The precedence level we revert to.
     *  @param ownPrec        The inner precedence level.
     */
    protected void close(int contextPrec, int ownPrec) throws IOException {
        if (ownPrec < contextPrec) msp_out.write(")");
    }

    /** Print string, replacing all non-ascii character with unicode escapes.
     */
    public void print(Object s) throws IOException {
        msp_out.write(Convert.escapeUnicode(s.toString()));
    }

    /** Print new line.
     */
    public void println() throws IOException {
        msp_out.write(lineSep);
    }

    public void visitBracketExpr(JCBracketExpr tree) {
        // FIXME: recursively translate bracket expression
        throw new AssertionError("Unexpected bracket in MSPTreeCodeSerializer");
    }

    public void visitBracketStat(JCBracketStat tree) {
        // FIXME: recursively translate bracket statement
        throw new AssertionError("Unexpected bracket in MSPTreeCodeSerializer");
    }
    
    /* special methods for "printing" escapes */
    public void visitEscapeExpr(JCEscapeExpr tree) {
        msp_out.writeEscape (tree.body);
    }

    public void visitEscapeStat(JCEscapeStat tree) {
        msp_out.writeEscapeStat (tree.body);
    }

    @Override
    public void visitIdent(JCIdent tree) {
        // System.out.println("visitIdent: "+tree.name+" -- "+" TYPEVAR? "+(tree.sym.type.tag==TypeTags.TYPEVAR));
        if (tree.cspCount>0) {
            // FIXME: handle stage > 1
            // FIXME: what if CSP and gensym?
//            if (tree.genVarSym!=null) {
//                msp_out.writeCspSym(tree.genVarSym, tree.type);
//            }
//            else {
//                msp_out.writeCsp(tree);
//            }
            msp_out.writeCsp(tree);
        }
        else if (tree.genVarSym!=null) {
            msp_out.writeStringExpr(makeIdent(tree.genVarSym, syms.stringType));
        }
        else {
            if (tree.sym.type.tag==TypeTags.TYPEVAR) {
                // this is a type variable
                
                // look at classVarsInScope to find a VarSymbol that we can use for this type variable
                // and call Class.getName on it
                // System.out.println("TypeVar "+tree.sym);
                VarSymbol found = null;
                for(VarSymbol vs: classVarsInScope) {
                    // System.out.println("    in scope: "+vs);
                    // System.out.println("    comparing tree.sym "+tree.sym+" and vs.type.getTypeArguments().head.tsym "+
                    //                    vs.type.getTypeArguments().head.tsym+" : "+
                    //                    (tree.sym==vs.type.getTypeArguments().head.tsym));
                    if (tree.sym==vs.type.getTypeArguments().head.tsym) {
                        found = vs;
                        // System.out.println("======> FOUND "+vs);
                        break;
                    }
                }
                if (found==null) {
                    // System.out.println("==> ERROR: no Class<"+tree.sym+"> found in scope");
                    log.error(tree.pos(), "type.variable.without.class.variable.in.scope.used.in.brackets",
                              tree.sym.toString());
                }
                else {       
                    // use myClassOfT.getName().replace('$','.') to get the name of the type
                    JCMethodInvocation getNameCall = makeCall(makeIdent(found, found.type),
                                                              names.fromString("getName"),
                                                              List.<JCExpression>nil());
                    JCMethodInvocation replaceCall = makeCall(getNameCall,
                                                              names.fromString("replace"),
                                                              List.<JCExpression>of(makeLit(syms.charType, (int)'$'),
                                                                                    makeLit(syms.charType, (int)'.')));
                    // System.out.println("    using: "+replaceCall);
                    msp_out.writeStringExpr(replaceCall);
                }
            }
            else {
                // not a type variable
                super.visitIdent(tree);
            }
        }
    }
     
    // prints the name of a variable declaration; can be overridden
    public void printVarDefName(JCVariableDecl tree) throws IOException {
        if (tree.genVarSym!=null) {
            msp_out.writeStringExpr(makeIdent(tree.genVarSym, syms.stringType));
        }
        else {
            super.printVarDefName(tree);
        }
    }
    
    static class MSPTreeCodeWriter extends Writer {        
        ArrayList<Elem> elems = new ArrayList<Elem>();
        
        StringBuilder currentString = new StringBuilder();
        
        public void write(char[] cbuf, int off, int len)  {
            currentString.append(cbuf, off, len);
        }
        
        public void writeEscape(JCExpression tree) {
            flush();
            elems.add(new EscapeElem(tree));
        }
        
        public void writeEscapeStat(JCExpression tree) {
            flush();
            elems.add(new EscapeStatElem(tree));
        }

        public void writeCsp(JCIdent ident) {
            flush();
            elems.add(new CspElem(ident));
        }
        
        public void writeStringExpr(JCExpression expr) {
            flush();
            elems.add(new StringExprElem(expr));
        }
        
//        public void writeCspSym(VarSymbol sym, Type t) {
//            flush();
//            elems.add(new CspSymElem(sym, t));
//        }
        
        public void flush() {
            if (currentString.length()>0) {
                String current = currentString.toString();
                currentString.setLength(0);
                elems.add(new StringElem(current));
            }
        }
        
        public void close() {
            flush();
        }
        
        public java.util.List<Elem> getElems() {
            flush();
            return new ArrayList<Elem>(elems);
        }
    }
    
    public static abstract class Elem { }
    
    public static class StringElem extends Elem {
        public String contents;
        public StringElem(String contents) {
            this.contents = contents;
        }
    }
    
    // escape expression
    public static class EscapeElem extends Elem {
        public JCExpression contents;
        public EscapeElem(JCExpression contents) {
            this.contents = contents;
        }
    }

    // escape statement
    public static class EscapeStatElem extends Elem {
        public JCExpression contents;
        public EscapeStatElem(JCExpression contents) {
            this.contents = contents;
        }
    }
    
    public static class CspElem extends Elem {
        public JCIdent contents;
        public CspElem(JCIdent contents) {
            this.contents = contents;
        }
    }
    
    public static class StringExprElem extends Elem {
        public JCExpression contents;
        public StringExprElem(JCExpression contents) {
            this.contents = contents;
        }
    }
    
//    public static class CspSymElem extends Elem {
//        public VarSymbol contents;
//        public Type type;
//        public CspSymElem(VarSymbol contents, Type type) {
//            this.contents = contents;
//            this.type = type;
//        }
//    }
    
    /** Create an attributed tree of the form left.name(). */
    private JCMethodInvocation makeCall(JCExpression left, Name name, List<JCExpression> args) {
        assert left.type != null;
        Symbol funcsym = lookupMethod(make_pos, name, left.type,
                                      TreeInfo.types(args));
        return make.App(make.Select(left, funcsym), args);
    }
    
    /** Look up a method in a given scope.
      */
    private MethodSymbol lookupMethod(DiagnosticPosition pos, Name name, Type qual, List<Type> args) {
        return rs.resolveInternalMethod(pos, attrEnv, qual, name, args, null);
    }
    
    /** Make an attributed tree representing a literal. This will be an
      *  Ident node in the case of boolean literals, a Literal node in all
      *  other cases.
      *  @param type       The literal's type.
      *  @param value      The literal's value.
      */
    JCExpression makeLit(Type type, Object value) {
        return make.Literal(type.tag, value).setType(type.constType(value));
    }
    
    /** Make an attributed tree representing an identifier. */
    JCIdent makeIdent(Symbol sym, Type t) {
        JCIdent ident = make.Ident(sym.name);
        ident.setType(t);
        ident.sym = sym;
        return ident;
    }    
}
