/*
 * Copyright 2005-2006 Sun Microsystems, Inc.  All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa Clara,
 * CA 95054 USA or visit www.sun.com if you need additional information or
 * have any questions.
 */

package edu.rice.cs.mint.comp.javax.lang.model.util;

import java.lang.Iterable;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.EnumSet;
import java.util.ArrayList;
import java.util.LinkedHashSet;


/**
 * Filters for selecting just the elements of interest from a
 * collection of elements.  The returned sets and lists are new
 * collections and do use the argument as a backing store.  The
 * methods in this class do not make any attempts to guard against
 * concurrent modifications of the arguments.  The returned sets and
 * lists are mutable but unsafe for concurrent access.  A returned set
 * has the same iteration order as the argument set to a method.
 *
 * <p>If iterables and sets containing {@code null} are passed as
 * arguments to methods in this class, a {@code NullPointerException}
 * will be thrown.
 *
 * <p>Note that a <i>static import</i> statement can make the text of
 * calls to the methods in this class more concise; for example:
 *
 * <blockquote><pre>
 *     import static edu.rice.cs.mint.comp.javax.lang.model.util.ElementFilter.*;
 *     ...
 *         {@code List<VariableElement>} fs = fieldsIn(someClass.getEnclosedElements());
 * </pre></blockquote>
 *
 * @author Joseph D. Darcy
 * @author Scott Seligman
 * @author Peter von der Ah&eacute;
 * @author Martin Buchholz
 * @since 1.6
 */
public class ElementFilter {
    private ElementFilter() {} // Do not instantiate.

    private static Set<edu.rice.cs.mint.comp.javax.lang.model.element.ElementKind> CONSTRUCTOR_KIND =
        Collections.unmodifiableSet(EnumSet.of(edu.rice.cs.mint.comp.javax.lang.model.element.ElementKind.CONSTRUCTOR));

    private static Set<edu.rice.cs.mint.comp.javax.lang.model.element.ElementKind> FIELD_KINDS =
        Collections.unmodifiableSet(EnumSet.of(edu.rice.cs.mint.comp.javax.lang.model.element.ElementKind.FIELD,
                                               edu.rice.cs.mint.comp.javax.lang.model.element.ElementKind.ENUM_CONSTANT));
    private static Set<edu.rice.cs.mint.comp.javax.lang.model.element.ElementKind> METHOD_KIND =
        Collections.unmodifiableSet(EnumSet.of(edu.rice.cs.mint.comp.javax.lang.model.element.ElementKind.METHOD));

    private static Set<edu.rice.cs.mint.comp.javax.lang.model.element.ElementKind> PACKAGE_KIND =
        Collections.unmodifiableSet(EnumSet.of(edu.rice.cs.mint.comp.javax.lang.model.element.ElementKind.PACKAGE));

    private static Set<edu.rice.cs.mint.comp.javax.lang.model.element.ElementKind> TYPE_KINDS =
        Collections.unmodifiableSet(EnumSet.of(edu.rice.cs.mint.comp.javax.lang.model.element.ElementKind.CLASS,
                                               edu.rice.cs.mint.comp.javax.lang.model.element.ElementKind.ENUM,
                                               edu.rice.cs.mint.comp.javax.lang.model.element.ElementKind.INTERFACE,
                                               edu.rice.cs.mint.comp.javax.lang.model.element.ElementKind.ANNOTATION_TYPE));
    /**
     * Returns a list of fields in {@code elements}.
     * @return a list of fields in {@code elements}
     * @param elements the elements to filter
     */
    public static List<edu.rice.cs.mint.comp.javax.lang.model.element.VariableElement>
            fieldsIn(Iterable<? extends edu.rice.cs.mint.comp.javax.lang.model.element.Element> elements) {
        return listFilter(elements, FIELD_KINDS, edu.rice.cs.mint.comp.javax.lang.model.element.VariableElement.class);
    }

    /**
     * Returns a set of fields in {@code elements}.
     * @return a set of fields in {@code elements}
     * @param elements the elements to filter
     */
    public static Set<edu.rice.cs.mint.comp.javax.lang.model.element.VariableElement>
            fieldsIn(Set<? extends edu.rice.cs.mint.comp.javax.lang.model.element.Element> elements) {
        return setFilter(elements, FIELD_KINDS, edu.rice.cs.mint.comp.javax.lang.model.element.VariableElement.class);
    }

    /**
     * Returns a list of constructors in {@code elements}.
     * @return a list of constructors in {@code elements}
     * @param elements the elements to filter
     */
    public static List<edu.rice.cs.mint.comp.javax.lang.model.element.ExecutableElement>
            constructorsIn(Iterable<? extends edu.rice.cs.mint.comp.javax.lang.model.element.Element> elements) {
        return listFilter(elements, CONSTRUCTOR_KIND, edu.rice.cs.mint.comp.javax.lang.model.element.ExecutableElement.class);
    }

    /**
     * Returns a set of constructors in {@code elements}.
     * @return a set of constructors in {@code elements}
     * @param elements the elements to filter
     */
    public static Set<edu.rice.cs.mint.comp.javax.lang.model.element.ExecutableElement>
            constructorsIn(Set<? extends edu.rice.cs.mint.comp.javax.lang.model.element.Element> elements) {
        return setFilter(elements, CONSTRUCTOR_KIND, edu.rice.cs.mint.comp.javax.lang.model.element.ExecutableElement.class);
    }

    /**
     * Returns a list of methods in {@code elements}.
     * @return a list of methods in {@code elements}
     * @param elements the elements to filter
     */
    public static List<edu.rice.cs.mint.comp.javax.lang.model.element.ExecutableElement>
            methodsIn(Iterable<? extends edu.rice.cs.mint.comp.javax.lang.model.element.Element> elements) {
        return listFilter(elements, METHOD_KIND, edu.rice.cs.mint.comp.javax.lang.model.element.ExecutableElement.class);
    }

    /**
     * Returns a set of methods in {@code elements}.
     * @return a set of methods in {@code elements}
     * @param elements the elements to filter
     */
    public static Set<edu.rice.cs.mint.comp.javax.lang.model.element.ExecutableElement>
            methodsIn(Set<? extends edu.rice.cs.mint.comp.javax.lang.model.element.Element> elements) {
        return setFilter(elements, METHOD_KIND, edu.rice.cs.mint.comp.javax.lang.model.element.ExecutableElement.class);
    }

    /**
     * Returns a list of types in {@code elements}.
     * @return a list of types in {@code elements}
     * @param elements the elements to filter
     */
    public static List<edu.rice.cs.mint.comp.javax.lang.model.element.TypeElement>
            typesIn(Iterable<? extends edu.rice.cs.mint.comp.javax.lang.model.element.Element> elements) {
        return listFilter(elements, TYPE_KINDS, edu.rice.cs.mint.comp.javax.lang.model.element.TypeElement.class);
    }

    /**
     * Returns a set of types in {@code elements}.
     * @return a set of types in {@code elements}
     * @param elements the elements to filter
     */
    public static Set<edu.rice.cs.mint.comp.javax.lang.model.element.TypeElement>
            typesIn(Set<? extends edu.rice.cs.mint.comp.javax.lang.model.element.Element> elements) {
        return setFilter(elements, TYPE_KINDS, edu.rice.cs.mint.comp.javax.lang.model.element.TypeElement.class);
    }

    /**
     * Returns a list of packages in {@code elements}.
     * @return a list of packages in {@code elements}
     * @param elements the elements to filter
     */
    public static List<edu.rice.cs.mint.comp.javax.lang.model.element.PackageElement>
            packagesIn(Iterable<? extends edu.rice.cs.mint.comp.javax.lang.model.element.Element> elements) {
        return listFilter(elements, PACKAGE_KIND, edu.rice.cs.mint.comp.javax.lang.model.element.PackageElement.class);
    }

    /**
     * Returns a set of packages in {@code elements}.
     * @return a set of packages in {@code elements}
     * @param elements the elements to filter
     */
    public static Set<edu.rice.cs.mint.comp.javax.lang.model.element.PackageElement>
            packagesIn(Set<? extends edu.rice.cs.mint.comp.javax.lang.model.element.Element> elements) {
        return setFilter(elements, PACKAGE_KIND, edu.rice.cs.mint.comp.javax.lang.model.element.PackageElement.class);
    }

    // Assumes targetKinds and E are sensible.
    private static <E extends edu.rice.cs.mint.comp.javax.lang.model.element.Element> List<E> listFilter(Iterable<? extends edu.rice.cs.mint.comp.javax.lang.model.element.Element> elements,
                                                          Set<edu.rice.cs.mint.comp.javax.lang.model.element.ElementKind> targetKinds,
                                                          Class<E> clazz) {
        List<E> list = new ArrayList<E>();
        for (edu.rice.cs.mint.comp.javax.lang.model.element.Element e : elements) {
            if (targetKinds.contains(e.getKind()))
                list.add(clazz.cast(e));
        }
        return list;
    }

    // Assumes targetKinds and E are sensible.
    private static <E extends edu.rice.cs.mint.comp.javax.lang.model.element.Element> Set<E> setFilter(Set<? extends edu.rice.cs.mint.comp.javax.lang.model.element.Element> elements,
                                                        Set<edu.rice.cs.mint.comp.javax.lang.model.element.ElementKind> targetKinds,
                                                        Class<E> clazz) {
        // Return set preserving iteration order of input set.
        Set<E> set = new LinkedHashSet<E>();
        for (edu.rice.cs.mint.comp.javax.lang.model.element.Element e : elements) {
            if (targetKinds.contains(e.getKind()))
                set.add(clazz.cast(e));
        }
        return set;
    }
}
