/*
 * Copyright 1999-2008 Sun Microsystems, Inc.  All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa Clara,
 * CA 95054 USA or visit www.sun.com if you need additional information or
 * have any questions.
 */

package edu.rice.cs.mint.comp.com.sun.tools.javac.comp;

import edu.rice.cs.mint.comp.com.sun.tools.javac.tree.*;
import java.util.Iterator;
import java.util.NoSuchElementException;
import edu.rice.cs.mint.comp.com.sun.tools.javac.util.List;
import static edu.rice.cs.mint.comp.com.sun.tools.javac.tree.JCTree.*;

/* for SEPARABLE */
import static edu.rice.cs.mint.comp.com.sun.tools.javac.code.Flags.*;

/** A class for environments, instances of which are passed as
 *  arguments to tree visitors.  Environments refer to important ancestors
 *  of the subtree that's currently visited, such as the enclosing method,
 *  the enclosing class, or the enclosing toplevel node. They also contain
 *  a generic component, represented as a type parameter, to carry further
 *  information specific to individual passes.
 *
 *  <p><b>This is NOT part of any API supported by Sun Microsystems.  If
 *  you write code that depends on this, you do so at your own risk.
 *  This code and its internal interfaces are subject to change or
 *  deletion without notice.</b>
 */
public class Env<A> implements Iterable<Env<A>> {

    /** The next enclosing environment.
     */
    public Env<A> next;

    /** The environment enclosing the current class.
     */
    public Env<A> outer;

    /** The tree with which this environment is associated.
     */
    public JCTree tree;

    /** The enclosing toplevel tree.
     */
    public JCTree.JCCompilationUnit toplevel;

    /** The next enclosing class definition.
     */
    public JCTree.JCClassDecl enclClass;

    /** The next enclosing method definition.
     */
    public JCTree.JCMethodDecl enclMethod;

    // emw4: staging additions

    /** A list of all enclosing brackets, innermost to outermost
     */
    public List<JCExpression> enclBrackets;

    /** The escape under which we are operating, or null if there was no escape
      * or if we entered a bracket inside an escape. */
    public JCTree escape = null;
    public int escapeCount = 0;
    
    /** A generic field for further information.
     */
    public A info;

    /** Is this an environment for evaluating a base clause?
     */
    public boolean baseClause = false;

    /** Create an outermost environment for a given (toplevel)tree,
     *  with a given info field.
     */
    public Env(JCTree tree, A info) {
        this.next = null;
        this.outer = null;
        this.tree = tree;
        this.toplevel = null;
        this.enclClass = null;
        this.enclMethod = null;
        this.enclBrackets = List.nil ();
        this.escape = null;
        this.escapeCount = 0;
        this.info = info;
    }

    /** Duplicate this environment, updating with given tree and info,
     *  and copying all other fields.
     */
    public Env<A> dup(JCTree tree, A info) {
        return dupto(new Env<A>(tree, info));
    }

    /** Duplicate this environment into a given Environment,
     *  using its tree and info, and copying all other fields.
     */
    public Env<A> dupto(Env<A> that) {
        that.next = this;
        that.outer = this.outer;
        that.toplevel = this.toplevel;
        that.enclClass = this.enclClass;
        that.enclMethod = this.enclMethod;
        that.enclBrackets = this.enclBrackets;
        that.escape = this.escape; 
        that.escapeCount = this.escapeCount;
        return that;
    }

    /** Duplicate this environment, updating with given tree,
     *  and copying all other fields.
     */
    public Env<A> dup(JCTree tree) {
        return dup(tree, this.info);
    }

    // emw4: staging additions

    /** Duplicate this environment, updating with given tree and info
     *  and adding a new bracket to enclBrackets, copying all other fields.
     */
    public Env<A> dupBracket(JCTree tree, A info, JCExpression bracket) {
        Env <A> env = dup(tree, info);
        env.enclBrackets = this.enclBrackets.prepend (bracket);
        env.escape = null;
        return env;
    }

    /** Duplicate this environment, updating with given tree
     *  and adding a new bracket to enclBrackets, copying all other fields.
     */
    public Env<A> dupBracket(JCTree tree, JCExpression bracket) {
        return dupBracket (tree, this.info, bracket);
    }

    /** Duplicate this environment, updating with given tree and info
     *  and removing the most recent bracket from enclBrackets, copying all other fields.
     */
    public Env<A> dupEscape(JCTree tree, A info) {
        Env <A> env = dup(tree, info);
        assert (! enclBrackets.isEmpty ());
        env.enclBrackets = this.enclBrackets.tail;
        env.escape = tree;
        env.escapeCount++;
        return env;
    }
    
    /** Duplicate this environment, updating with given tree
     *  and removing the most recent bracket from enclBrackets,
     *  copying all other fields.
     */
    public Env<A> dupEscape(JCTree tree) {
        return dupEscape(tree, info);
    }
    
    public int getLevel() {
        return enclBrackets.size();
    }

    private static final boolean _mintstrict = (System.getenv("mintstrict")!=null);
    
    /* FIXME: this should really find the most recent enclosing tree that is
    ** - a method, in which case return true iff SEPARABLE is set
    ** - an escape, in which case always return true
    ** - a bracket, in which case return the safe field of that bracket
    */
    public boolean isSeparable () {
        boolean currentMethodSeparable =
            (enclMethod != null && (enclMethod.sym.flags() & SEPARABLE)!=0);
        return (currentMethodSeparable || escape!=null);
//        return ((currentMethodSeparable && getLevel() == 0) || escape!=null);
    }

    /** Return closest enclosing environment which points to a tree with given tag.
     */
    public Env<A> enclosing(int tag) {
        Env<A> env1 = this;
        while (env1 != null && env1.tree.getTag() != tag) env1 = env1.next;
        return env1;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("Env[");
        String[] lines;
        sb.append(info);
        sb.append("\n  escape=");
        sb.append(escape);
        sb.append("\n  escapeCount=");
        sb.append(escapeCount);
        sb.append("\n  brackets=");
        for(JCExpression b: enclBrackets) {
            lines = b.toString().split("\n");
            for(String l: lines) {
                sb.append("\n    ");
                sb.append(l);
            }
        }
        lines = tree.toString().split("\n");
        for(String l: lines) {
            sb.append("\n    ");
            sb.append(l);
        }
        if (outer!=null) {
            sb.append("\n  outer=");
            lines = outer.toString().split("\n");
            for(String l: lines) {
                sb.append("\n    ");
                sb.append(l);
            }
        }
        sb.append("]");
        return sb.toString();
    }

    public Iterator<Env<A>> iterator() {
        return new Iterator<Env<A>>() {
            Env<A> next = Env.this;
            public boolean hasNext() {
                return next.outer != null;
            }
            public Env<A> next() {
                if (hasNext()) {
                    Env<A> current = next;
                    next = current.outer;
                    return current;
                }
                throw new NoSuchElementException();

            }
            public void remove() {
                throw new UnsupportedOperationException();
            }
        };
    }
}
