package edu.rice.cs.mint.runtime;

import java.util.*;
import java.io.*;
import java.util.jar.*;

public class MintSerializer {
    /** Options that can be set. */
    public enum Options { MAIN_METHOD, VERBOSE };
    
    /** The stream used for verbose output. */
    public static PrintStream out = System.out;

    /** Save the piece of code to the file under the given class name.
      * @param c piece of code
      * @param className class name to use
      * @param f file
      * @param mainMethod true to generate a class ('className$Main') with a main method */
    public static <X> void save(Code<X> c, String className, File f, Options... o) throws IOException {
        FileOutputStream fos = new FileOutputStream(f);
        try {
            save(c, className, fos, o);
        }
        finally {
            fos.close();
        }
    }
    
    /** Save the piece of code to a file with the given file name under the given class name.
      * @param c piece of code
      * @param className class name to use
      * @param os output stream */
    public static <X> void save(Code<X> c, String className, String fileName) throws IOException {
        save(c, className, fileName);
    }
    
    /** Save the piece of code to a file with the given file name under the given class name.
      * @param c piece of code
      * @param className class name to use
      * @param os output stream
      * @param mainMethod true to generate a class ('className$Main') with a main method */
    public static <X> void save(Code<X> c, String className, String fileName, Options... o) throws IOException {
        save(c, className, new File(fileName), o);
    }

    /** Save the piece of code to the output stream under the given class name.
      * @param c piece of code
      * @param className class name to use
      * @param os output stream */
    public static <X> void save(Code<X> c, String className, OutputStream os) throws IOException {
        save(c, className, os);
    }

    /** Save the piece of code to the output stream under the given class name.
      * @param c piece of code
      * @param className class name to use
      * @param os output stream
      * @param mainMethod true to generate a class ('className$Main') with a main method */
    public static <X> void save(Code<X> c, String className, OutputStream os, Options... o) throws IOException {
        // System.out.println("======= Saving "+className+" to stream =======");
        Set<Options> options = (o.length>0)?EnumSet.copyOf(java.util.Arrays.asList(o)):EnumSet.noneOf(Options.class);
        
        // Generating a string version of a manifest
        StringBuilder sb = new StringBuilder();
        sb.append("Manifest-Version: 1.0\n");
        sb.append(MANIFEST_MAIN_CLASS_KEY+": ").append(className).append("\n");
        if (options.contains(Options.MAIN_METHOD)) {
            sb.append("Main-Class: "+className+"$Main\n");
        }
    
        // string is converted to an input stream
        InputStream mfis = new ByteArrayInputStream(sb.toString().getBytes("UTF-8"));
    
        // Generating the manifest for the input stream
        Manifest mf = new Manifest(mfis);
        
        JarOutputStream jos = new JarOutputStream(os, mf);
        try {
            MSPTreeCode.ClassInfo ci;
            if (!(c instanceof MSPTreeCode)) {
                ci = getFromCompiledCode(c, className);
            }
            else {
                if (HashMapClassLoader.instance().getClassBytes(className)!=null) {
                    throw new CouldNotSaveCodeException("A class with name "+className+" already exists");
                }
                try {
                    if (HashMapClassLoader.instance().loadClass(className)!=null) {
                        throw new CouldNotSaveCodeException("A class with name "+className+" already exists");
                    }
                }
                catch(ClassNotFoundException cnfe) { /* expected */ }
                
                MSPTreeCode<X> code = (MSPTreeCode<X>)c;
                ci = code.generateClass(className);
                
                if (options.contains(Options.VERBOSE)) {
                    out.println("======= Saving "+className+" =======");
                    out.println(ci.text);
                    out.println("======^ Saving "+className+" ^======");
                }
            }
            if (ci.bytes_hash.size()==0) throw new CouldNotSaveCodeException("Class could not be compiled");
            
            for(Map.Entry<String,byte[]> e: ci.bytes_hash.entrySet()) {
                // System.out.println(e.getKey()+" --> "+e.getValue().length+" bytes");
                byte[] bytes = e.getValue();
                
                JarEntry je = new JarEntry(e.getKey()+".class");
                jos.putNextEntry(je);
                
                jos.write(bytes, 0, bytes.length);
            }
            
            JarEntry je = new JarEntry(className+".csp");
            jos.putNextEntry(je);
            
            ObjectOutputStream oos = new ObjectOutputStream(jos);
            oos.writeInt(ci.csp_table.length);
            
            for(int i=0; i<ci.csp_table.length; ++i) {
                if (!(ci.csp_table[i] instanceof Serializable)) {
                    throw new CouldNotSaveCodeException("CSP table entry "+i+" is not serializable: "+ci.csp_table[i]+
                                                        " (instance of "+ci.csp_table[i].getClass().getName()+")");
                }
                Serializable s = (Serializable)ci.csp_table[i];
                try {
                    oos.writeObject(s);
                }
                catch(InvalidClassException ice) {
                    throw new CouldNotSaveCodeException("CSP table entry "+i+" is not serializable: "+ci.csp_table[i]+
                                                        " (instance of "+ci.csp_table[i].getClass().getName()+")", ice);
                }
                catch(NotSerializableException nse) {
                    throw new CouldNotSaveCodeException("CSP table entry "+i+" is not serializable: "+ci.csp_table[i]+
                                                        " (instance of "+ci.csp_table[i].getClass().getName()+")", nse);
                }
            }
        
            oos.flush();
            
            if (options.contains(Options.MAIN_METHOD)) {
                addClassWithMainMethod(className, jos);
            }
            
            jos.flush();
        }
        finally {
            jos.close();
            os.flush();
        }
        
        // System.out.println("======^ Saving "+className+" to stream ^======");
    }
    
    /** Compile a class with a main method and add it to the jar. */
    protected static void addClassWithMainMethod(String className, JarOutputStream jos) throws java.io.IOException {
        String thisClassName = className+"$Main";
        StringBuilder fullCode = new StringBuilder();
        fullCode.append("import java.io.*;\n"+
                        "public class "+thisClassName+" {\n"+
                        "  public static void main(String[] args) throws Exception {\n"+
                        "    InputStream is = "+thisClassName+".class.getResourceAsStream(\""+className+".csp\");\n"+
                        "    ObjectInputStream ois = new ObjectInputStream(is);\n"+
                        "    int csp_length = ois.readInt();\n"+
                        "    Object[] csp_table = new Object[csp_length];\n"+
                        "    for(int i=0; i<csp_length; ++i) {\n"+
                        "      csp_table[i] = ois.readObject();\n"+
                        "    }\n"+
                        "    System.out.println(new "+className+"(csp_table).run());\n"+
                        "  }\n"+
                        "}");
        
        /* generate a compiler for the code */
        StringCodeCompiler comp = (StringCodeCompiler)StringCodeCompiler.instance(MSPTreeCode.context);
        
        edu.rice.cs.mint.comp.com.sun.tools.javac.util.Options options =
            edu.rice.cs.mint.comp.com.sun.tools.javac.util.Options.instance(MSPTreeCode.context);
        options.put("-classpath", HashMapClassLoader.instance().getClassPathPropertyValue());
        
        /* compile any and all necessary classes */
        HashMap<String,byte[]> bytes_hash = comp.compileStringToBytes(className+"$Main", fullCode.toString());
        for(Map.Entry<String,byte[]> e: bytes_hash.entrySet()) {
            // System.out.println(e.getKey()+" --> "+e.getValue().length+" bytes");
            byte[] bytes = e.getValue();
            
            JarEntry je = new JarEntry(e.getKey()+".class");
            jos.putNextEntry(je);
            
            jos.write(bytes, 0, bytes.length);
        }
    }
    
    /** Only fills out MSPTreeCode.ClassInfo.bytes_hash and MSPTreeCode.ClassInfo.csp_table. */
    protected static <X> MSPTreeCode.ClassInfo getFromCompiledCode(Code<X> c, String className) throws IOException {
        // if this was already compiled, i.e. it is not an MSPTreeCode, then it should be in the HashMapClassLoader
        MSPTreeCode.ClassInfo ci = new MSPTreeCode.ClassInfo();
        byte[] classBytes = HashMapClassLoader.instance().getClassBytes(c.getClass().getName());
        
        if (classBytes==null) {
            throw new CouldNotSaveCodeException("Code was already compiled, but class could not be found: "+
                                                c.getClass().getName());
        }
        
        if (!c.getClass().getName().equals(className)) {
            throw new CouldNotSaveCodeException("Code was already compiled with name "+c.getClass().getName()+
                                                ", cannot save it with different name "+className);
        }
        
        ci.bytes_hash = new HashMap<String,byte[]>();
        ci.bytes_hash.put(c.getClass().getName(), classBytes);
        ci.csp_table = new Object[0]; // TODO
        return ci;
    }

    public static final String MANIFEST_MAIN_CLASS_KEY = "edu-rice-cs-mint-runtime-MintSerializer-main-class";
    
    /** Load a piece of code from the specified file.
      * @param f file
      * @return piece of code */
    protected static <X> Code<X> loadCode(File f) throws IOException {
        FileInputStream fis = new FileInputStream(f);
        try {
            return loadCode(fis);
        }
        finally {
            fis.close();
        }
    }
    
    /** Load a piece of code from a file with the the specified name.
      * @param fileName name of the file
      * @return piece of code */
    protected static <X> Code<X> loadCode(String fileName) throws IOException {
        return loadCode(new File(fileName));
    }
    
    @SuppressWarnings("unchecked")
    /** Load a piece of code from the input stream.
      * @param is input stream
      * @return piece of code */
    protected static <X> Code<X> loadCode(InputStream is) throws IOException {
        // System.out.println("======= Loading from stream =======");        
        final int BUF_SIZE = 1024;
        final byte[] buf = new byte[BUF_SIZE];
        BufferedInputStream bis = new BufferedInputStream(is, BUF_SIZE);
        
        JarInputStream jis = new JarInputStream(bis);
        Manifest mf = jis.getManifest();
        
        String mainClassName = null;
        Object[] csp_table = null;
        
        Attributes attrs = mf.getMainAttributes();
        for (Iterator it = attrs.keySet().iterator(); it.hasNext();) {
            Attributes.Name attrName = (Attributes.Name)it.next();
            String attrValue = attrs.getValue(attrName);
            if (attrName.toString().equals(MANIFEST_MAIN_CLASS_KEY)) {
                mainClassName = attrValue;
            }
        }
        
        if (mainClassName==null) {
            throw new CouldNotLoadCodeException("Main class name not set in manifest file");
        }
        
        JarEntry je;
        HashMap<String,byte[]> bytes_hash = new HashMap<String,byte[]>();
        
        while((je=jis.getNextJarEntry()) != null) {
            String fileName = je.getName();
            // System.out.println("File "+fileName);
            if (fileName.equals(mainClassName+".csp")) {
                ObjectInputStream ois = new ObjectInputStream(jis);
                int csp_length = ois.readInt();
                // System.out.println("csp_length = "+csp_length);                
                csp_table = new Object[csp_length];
                
                for(int i=0; i<csp_length; ++i) {
                    try {
                        csp_table[i] = ois.readObject();
                    }
                    catch(ClassNotFoundException cnfe) {
                        throw new CouldNotLoadCodeException("CSP table entry "+i+" could not be deserialized", cnfe);
                    }
                }
            }
            else if (fileName.endsWith(".class")) {
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                int red = 0;
                while((red=jis.read(buf, 0, BUF_SIZE)) >= 0) {
                    // System.out.println("read="+red);
                    baos.write(buf, 0, red);
                }
                byte[] classBytes = baos.toByteArray();
                // System.out.println("Loaded "+classBytes.length+" bytes");        
                bytes_hash.put(fileName.substring(0, fileName.length() - ".class".length()), classBytes);
            }
        }

        if (csp_table==null) {
            throw new CouldNotLoadCodeException("CSP table not found, file "+mainClassName+".csp missing");
        }

        try {
            Code<X> code = (Code<X>)MSPTreeCode.loadClassAndCreateInstance(mainClassName, bytes_hash, csp_table);
             // System.out.println("Loaded "+code.getClass());
             // System.out.println(code);
             // System.out.println("======^ Loading from stream ^======");
            return code;
        }
        catch(InternalError ie) {
            throw new CouldNotLoadCodeException(ie.getMessage(), ie.getCause());
        }
    }

    @SuppressWarnings("unchecked")
    /** Load a value from the input stream by loading the code and running it.
      * @param is input stream
      * @return value */
    public static <X> X load(InputStream is) throws IOException {
        return ((Code<X>)loadCode(is)).run();
    }

    @SuppressWarnings("unchecked")
    /** Load a value from the specified file by loading the code and running it.
      * @param f file
      * @return value */
    public static <X> X load(File f) throws IOException {
        return ((Code<X>)loadCode(f)).run();
    }
    
    @SuppressWarnings("unchecked")
    /** Load a value from a file with the specified name by loading the code and running it.
      * @param fileName name of the file
      * @return value */
    public static <X> X load(String fileName) throws IOException {
        return ((Code<X>)loadCode(fileName)).run();
    }
}
