package brs.visitor;

import brs.*;
import fp.*;

/**
 * Traverse a binary tree in order:
 * For an empty tree:
 * do the appropriate processing.
 * For a non-empty tree:
 * Traverse the left subtree in order;
 * Process the root;
 * Traverse the right subtree in order;
 * 
 * Uses two lambdas as variants.
 * Let fRight, fLeft be ILambda and b be some input object.
 * empty case: 
 *   InOrder2(empty, fRight, fLeft, b) = b;
 * non-empty case: 
 *   InOder(tree, fRight, fLeft, b) = 
 *     fRight(fLeft(InOrder2(tree.left, fRight, fLeft, b), tree)), 
 *            InOrder2(tree.right, fRight, fLeft, b));
 * @author DXN
 * @author SBW
 * @author Mathias Ricken - Copyright 2008 - All rights reserved.
 * @since 09/22/2004
 */
public class InOrder2<T,P> implements IVisitor<T,P,P> {
    // an abstract function with domain (range of InOrder2, BiTree):
    private ILambda2<P,P,BiTree<T>> _fLeft;
    
    // an abstract function with domain (range of _fLeft, range of InOrder2):
    private ILambda2<P,P,P> _fRight;
    
    
    public InOrder2(ILambda2<P,P,P> fRight, ILambda2<P,P,BiTree<T>> fLeft) {
        _fRight = fRight;
        _fLeft = fLeft;
    }
    
    
    public P emptyCase(BiTree<T> host, P... b) {
        return b[0]; 
    }
    
    public P nonEmptyCase(BiTree<T> host, P... b) {
        return _fRight.apply(_fLeft.apply(host.getLeftSubTree().execute(this, b), host), 
                             host.getRightSubTree().execute(this, b));
    }
    
    
    public static void main(String[] nu) {
        final ILambda2<Integer,Integer,Integer> add =
            new ILambda2<Integer,Integer,Integer>() {
            public Integer apply(Integer arg1, Integer arg2) {
                return arg1+arg2;
            }
        };

        ILambda2<Integer,Integer,BiTree<Integer>> add2 =
            new ILambda2<Integer,Integer,BiTree<Integer>>() {
            public Integer apply(Integer arg1, BiTree<Integer> arg2) {
                return add.apply(arg1,arg2.getRootDat());
            }
        };
        
        // Add the numbers in the tree in in-order fashion:
        InOrder2<Integer,Integer> inOrderAdd = new InOrder2<Integer,Integer>(add, add2);
        
        
        final ILambda2<String,String,String> concat = new ILambda2<String,String,String>() {
            public String apply(String arg1, String arg2) {
                if ("" != arg1.toString()) {
                    if ("" != arg2.toString()) {
                        return arg1.toString() + " " + arg2.toString();
                    }
                    else {
                        return arg1.toString();
                    }
                }
                else {
                    return arg2.toString();
                }
            }
        };
        ILambda2<String,String,BiTree<Integer>> concat2 = new ILambda2<String,String,BiTree<Integer>>() {
            public String apply(String arg1, BiTree<Integer> arg2) {
                return concat.apply(arg1, arg2.getRootDat().toString());
            }
        };
        // Concatenate the String representation of the elements in the tree 
        // in in-order fashion:
        InOrder2<Integer,String> inOrderConcat = new InOrder2<Integer,String>(concat, concat2);
        
        
        BiTree<Integer> bt = new BiTree<Integer>();
        System.out.println(bt + "\nAdd \n" + bt.execute(inOrderAdd, 0));
        System.out.println("In order concat \n" + bt.execute(inOrderConcat, "")); 
        
        
        bt.insertRoot(5);
        System.out.println(bt + "\nAdd \n" + bt.execute(inOrderAdd, 0));
        System.out.println("In order concat \n" + bt.execute(inOrderConcat, "")); 
        
        
        bt.getLeftSubTree().insertRoot(-2);
        System.out.println(bt + "\nAdd \n" + bt.execute(inOrderAdd, 0));
        System.out.println("In order concat \n" + bt.execute(inOrderConcat, ""));   
        
        
        bt.getRightSubTree().insertRoot(10);
        System.out.println(bt + "\nAdd \n" + bt.execute(inOrderAdd, 0));
        System.out.println("In order concat \n" + bt.execute(inOrderConcat, "")); 
        
        
        bt.getRightSubTree().getLeftSubTree().insertRoot(-9);
        System.out.println(bt + "\nAdd \n" + bt.execute(inOrderAdd, 0));
        System.out.println("In order concat \n" + bt.execute(inOrderConcat, ""));                
    }
}