package lrs;

/**
 * Represents the abstract list state. Has a concrete toString () method that
 * uses anynomous inner classes to compute the String representation of the
 * LRStruct owner.
 * @author Dung X. Nguyen and Stephen Wong  Copyright 2005 - All rights reserved.
 * @author Mathias Ricken - Copyright 2008 - All rights reserved.
 * @since 8/25/05
 * @stereotype abstract state
 */
abstract class ANode<T> {
  /**
   * Uses anonymous visitor class to compute a String representation.
   */
  private final IAlgo<T,String,Void> ToStringAlgo = new IAlgo<T,String,Void>() {
    private final IAlgo<T,String,Void> Helper = new IAlgo<T,String,Void>() {
      public String emptyCase(LRStruct<T> h, Void... i) {
        return ")";
      }
      
      public String nonEmptyCase(LRStruct<T> h, Void... i) {
        return " " + h.getFirst() + h.getRest().execute (this);
      }
    };
    
    public String emptyCase(LRStruct<T> host, Void... inp) {
      return "()";
    }

    public String nonEmptyCase(LRStruct<T> host, Void... inp) {
      return "(" + host.getFirst() + host.getRest().execute(Helper);
    }
  };
  
  String toString(LRStruct<T> owner) {
    return (String)owner.execute (ToStringAlgo);
  }
  
  /**
   * Returns the tail LRStruct of the referencing LRStruct.
   * @param owner the LRStruct referencing this ANode.
   * @return the tail LRStruct of owner.
   * @throw java.util.NoSuchElementException if empty.
   */
  abstract LRStruct<T> getRest(LRStruct<T> owner);
  
  /**
   * Returns the first data object of the referencing LRStruct.
   * @param owner the LRStruct referencing this ANode.
   * @return the tail LRStruct of owner.
   * @throw java.util.NoSuchElementException if empty.
   */
  abstract T getFirst(LRStruct<T> owner);
  
  /**
   * Sets a new tail for the referencing LRStruct.
   * @param tail the new tail for the owner LRStruct.
   * @param owner the LRS referencing this ANode.
   * @throw java.util.NoSuchElementException if empty.
   * @return <code>LRStruct</code> owner
   */
  abstract LRStruct<T> setRest(LRStruct<T> tail, LRStruct<T> owner);
  
  /**
   * Sets a new first data object for the referencing LRStruct.
   * @param first the new data object for this ANode.
   * @param owner the LRS referencing this ANode.
   * @throw java.util.NoSuchElementException if empty.
   * @return <code>LRStruct</code> owner
   */
  abstract LRStruct<T> setFirst(T dat, LRStruct<T> owner);
  
  /**
   * Inserts a data object at the front of the LRStruct owner.
   * @param dat the object to be inserted at the front.
   * @param owner the LRS referencing this ANode.
   * @return <code>LRStruct</code> owner
   */
  abstract LRStruct<T> insertFront(T dat, LRStruct<T> owner);
  
  /**
   * Removes and returns the first data object for the referencing LRStruct.
   * @param owner the LRS referencing this ANode.
   * @return the front data of the LRStruct owner.
   */
  abstract T removeFront(LRStruct<T> owner);
  
  /**
   * Executes a visitor algorithm and returns the output.
   * @param owner the LRStruct referencing this ANode.
   * @param algo the visitor algorithm to be executed.
   * @param inp the inputs needed by the algorithm.
   */
  abstract <R,P> R execute(LRStruct<T> owner, IAlgo<T, R, P> algo, P ... inp);
}

