package rac;

import listFW.*;
import listFW.factory.*;
import lrs.*;

/**
 * Implements a factory for restricted access containers.  These
 * restricted access containers are implemented using an LRStruct to
 * hold the data objects.
 * @author Mathias Ricken - Copyright 2008 - All rights reserved.
 */
public abstract class ALRSRACFactory<T> implements IRACFactory<T> {
  /**
   * Implements a general-purpose restricted access container using
   * an LRStruct.  How? 
   *
   * The next item to remove is always at the front of the list of
   * contained objects.  This is invariant!
   *
   * Insertion is, however, delegated to a strategy routine; and
   * this strategy is provided to the container.  This strategy
   * varies to implement the desired kind of container, e.g., queue
   * vs. stack.
   *
   * This inner class is protected so that classes derived from its
   * factory can reuse it to create other kinds of restricted access
   * container. 
   */
  protected class LRSRAContainer<T> implements IRAContainer<T> {
    private IAlgo<T,LRStruct<T>,T> _insertStrategy;
    private LRStruct<T> _lrs;
    
    public LRSRAContainer(IAlgo<T,LRStruct<T>,T> strategy) {
      _insertStrategy = strategy;
      _lrs = new LRStruct<T>();
    }
    
    /**
     * Empty the container.
     */
    public void clear() {
      _lrs = new LRStruct<T>();
    }
        
    /**
     * Return TRUE if the container is full; otherwise, return
     * FALSE. 
     *
     * This implementation can hold an arbitrary number of
     * objects.  Thus, always return false.
     */
    public boolean isFull() {
      return false;
    }
    
    /**
     * Return an immutable list of all elements in the container.
     */
    public IList<T> elements(final IListFactory<T> fact) {
      return _lrs.execute(new IAlgo<T,IList<T>,Void>() {               
        public IList<T> emptyCase(LRStruct<T> host, Void... input) {
          return fact.makeEmptyList();
        }
        
        public IList<T> nonEmptyCase(LRStruct<T> host, Void... input) {
          return fact.makeNEList(host.getFirst(),
                                 host.getRest().execute(this, input));
        }
      });
    }
    
    /**
     * Remove the next item from the container and return it.
     */
    public T get() {
      return _lrs.removeFront();
    }
    
    /**
     * Add an item to the container.
     */
    @SuppressWarnings("unchecked")
    public void put(T input) {
      _lrs.execute(_insertStrategy, input);
    }
    
    /**
     * Return the next element in this IRAContainer withour removing it.
     * @throw an Exception if this IRAContainer is empty.
     */
    public T peek() {
      return _lrs.getFirst();
    }
    
    /**
     * Extensibility hook to accept a visitor algorithm.
     * @param v  The visitor to execute
     * @param inp An arbitrary input parameter for the visitor
     * @return The return value of the calculation performed by the visitor.
     */
    public <R,P> R execute(final IRACVisitor<T,R,P> v, P... inp) { 
      return _lrs.execute(new IAlgo<T,R,P>() {
        public R emptyCase(LRStruct<T> host, P... i) {
          return v.emptyCase(LRSRAContainer.this, i);
        }
        public R nonEmptyCase(LRStruct<T> host, P... i) {
          return v.nonEmptyCase(LRSRAContainer.this, i);          
        }
      }, inp);
    }
    
  }    
}

