package poly.op;

import poly.*;

/**
 * Performs long division where the host is the divisor and the input the
 * dividend.
 * @author DXN
 * @since Copyright 2002 by DXN - All rights reserved.
 */
public class LongDiv implements IPolyOp {

    private IPolyFact _fact;
    private IPolyOp _mult;
    private IPolyOp _add;

    public LongDiv(IPolyFact f) {
        _fact = f;
        _mult = new Mul(f);
        _add = new Add(f);
    }

    /**
     * The quotient is the dividend divided by the divisor's coefficient.
     * The remainder is the zero polynomial.
     * @exception IllegalArgumentException if divisor is the zero polynomial.
     */
    public Object constCase(IConstPoly divisor, Object... dividend) {
        final double coef = divisor.getLeadCoef();
        if (coef < 0 || 0 < coef)  { // avoid using == for inexact arithmetics.
            return new QRPair((IPoly)((IPoly)dividend[0]).execute(new IPolyOp() {
                /**
                 * Divides the host pol by the divisor's coefficient.
                 */
                public Object constCase(IConstPoly pol, Object... nu) {
                    return _fact.makeConstPoly(pol.getLeadCoef() / coef);
                }

                /**
                 * Divides the host pol by the divisor's coefficient.
                 * Recurs on the rest of pol.
                 */
                public Object nonConstCase(INCPoly pol, Object... inp) {
                    return _fact.makeNCPoly(
                                pol.getLeadCoef() / coef,
                                pol.getOrder(),
                                (IPoly)pol.getLowerPoly().execute(this));
                }
            })            // this is the quotient.
            , _fact.makeConstPoly(0));  // the remainder is the zero polynomial.

        }
        throw new IllegalArgumentException ("Divide by 0!");
    }

    /**
     * Performs different computations for three different cases:
     * dividend's order < divisor's order
     * dividend's order == divisor's order
     * dividend's order > divisor's order
     */
    public Object nonConstCase(INCPoly divisor, Object... dividend) {
        IPoly Dividend = (IPoly)dividend[0];
        int dividendOrder = Dividend.getOrder();
        int divisorOrder = divisor.getOrder();

        // Consider the case the dividend's order < the divisor's order.
        // The quotient = zero polynomial and the remainder = dividend:
        if (dividendOrder < divisorOrder) {
            return new QRPair(_fact.makeConstPoly(0), Dividend);
        }

        // If we get to this point, we know the dividend's order is at least
        // the divisor's order.  Since the divisor is an INCPoly, dividend is
        // also an INCPoly.

        // The quotient's leading coefficient is:
        double quotCoef = Dividend.getLeadCoef() / divisor.getLeadCoef();

        // Consider the case the dividend's order == the divisor's order.
        // The quotient is the constant polynomial whose leading coeffient is
        // quotCoef.  The remainder is the difference between the rest of the
        // dividend and the product of the quotient and the rest of the divisor.
        if (dividendOrder == divisorOrder) {
            IConstPoly q = _fact.makeConstPoly(-quotCoef);
            Object p = divisor.getLowerPoly().execute(_mult, q);
            Object r = ((INCPoly)Dividend).getLowerPoly().execute(_add, p);
            return new QRPair(_fact.makeConstPoly(quotCoef), (IPoly)r);
        }

        // If we get to this point, the quotient must be an INCPoly.  We
        // The quotient's order is:
        int quotOrder = dividendOrder - divisorOrder; // > 0.
        // Consider the monomial whose leading coef is -quotCoef:
        INCPoly mono
            = _fact.makeNCPoly(-quotCoef, quotOrder, _fact.makeConstPoly(0));
        // Multiply the divisor lower poly by mono:
        Object prod = divisor.getLowerPoly().execute(_mult, mono);
        // Add the resulting product to the dividend's lower poly:
        Object diff = ((INCPoly)Dividend).getLowerPoly().execute(_add, prod);
        // Recursively perform long division on diff:
        QRPair qr = (QRPair)divisor.execute(this, diff);
        return
            new QRPair(_fact.makeNCPoly(quotCoef, quotOrder, qr.getQuotient()),
                        qr.getRemainder());
    }
}


