package poly.factory;

import poly.*;
import poly.op.*;

/**
 * Manufacture polynomials using the composite design pattern.
 * @author Dung X. Nguyen
 * @since Copyright 2002 - DXN - All rights reserved
 * @dependency poly.factory.CompPolyFact$NCPoly 
 * @dependency poly.factory.CompPolyFact$ConstPoly instantiates
 */
public class CompPolyFact implements IPolyFact {

    /**
     * Singleton Design Pattern.
     */
    public static final CompPolyFact Singleton = new CompPolyFact();
    private CompPolyFact() {
    }


    private static abstract class APoly implements IPoly {
        private double _coef;

        protected APoly(double coef) {
            _coef = coef;
        }

        public double getLeadCoef() {
            return _coef;
        }

        public String toString() {
            return (String)execute(ToString.Singleton);
        }
    }

    /**
     * Represents constant polynomials.
     */
    class ConstPoly extends CompPolyFact$APoly
		implements IConstPoly  {

        public ConstPoly(double coef) {
            super(coef);
        }

        /**
         * @return 0;
         */
        public int getOrder() {
            return 0;
        }


        /**
         * Calls op.constCase to carry out the task.
         */
        public Object execute(IPolyOp op, Object[] inp) {
            return op.constCase(this, inp);
        }
    }

    /**
    * Represents the non-constant polynomial type.  Holds a postive order,
    * and a polynomial of lower order.
    */
    class NCPoly extends CompPolyFact$APoly
		implements INCPoly {

        /**
         * The order of this NCPoly.
         * Data invariant: _order > 0.
         */
        private int _order;

        /**
         * The lower order polynomial of this NCPoly.
         * Data invariant: _lowPoly != null and _lowPoly.getOrder < _order.
         */
        private IPoly _lowPoly;

        /**
        * Initializes this NCPoly with a given leading coefficient, a given
        * order, and a given lower order polynomial.
        * Checks for legal input parameters.
        * Ignores zero leading coefficient.
        * @param coef != 0.
        * @param order > 0.
        * @param lowPoly != null, the lower ordered polynomial with degree < order.
        * @exception IllegalArgumentException if conditions on coef, order and
        * lowerPoly are violated.
         */
        public NCPoly(double coef, int order, IPoly lowPoly) {
            super(coef);
            if (order <= 0) {
                throw new IllegalArgumentException("order <= 0!");
            }
            if (null == lowPoly) {
                throw new IllegalArgumentException("lowPoly == null!");
            }
            if (order <= lowPoly.getOrder()) {
                throw new IllegalArgumentException("order <= lowPoly's order!");
            }
            if (!(coef < 0 || 0 < coef)) {
                throw new IllegalArgumentException("coefficent == 0!");
            }
            _order = order;
            _lowPoly = lowPoly;
        }

        public int getOrder() {
            return _order;
        }

        public IPoly getLowerPoly() {
            return _lowPoly;
        }

        /**
        * Calls op.nonConstCase (...).
        */
        public Object execute(IPolyOp op, Object[] inp)  {
                return op.nonConstCase (this, inp);
        }
    }


   /**
   * Creates a constant Polynomial.
   */
   public IConstPoly makeConstPoly (double coef) {
      return new ConstPoly(coef);
   }

   /**
   * Creates a non-constant Polynomial with a given leading coefficient, a given
   * order, and a given lower order polynomial.
   * Checks for legal input parameters.
   * Ignores zero leading coefficient.
   * @param coef != 0.
   * @param order > 0.
   * @param lowPoly != null, the lower ordered polynomial with degree < order.
   * @exception IllegalArgumentException if conditions on coef, order and
   * lowerPoly are violated.
   */
   public INCPoly makeNCPoly (double coef, int order, IPoly lowPoly) {
      return new NCPoly(coef, order, lowPoly);
   }
}

